<?php
namespace App\Console\Commands;

use App\Models\Admin;
use Illuminate\Console\Command;
use App\Models\Booking;
use App\Models\BookingItem;
use App\Models\BookingUserDetail;
use App\Services\PaymentLinkService;
use Illuminate\Support\Facades\Mail;
use App\Models\Notification;
use Carbon\Carbon;




class SendExpiredPaymentLinks extends Command
{
    protected $signature = 'payments:resend-expired-links';
    protected $description = 'Resend expired payment links';

    /**
     * Execute the console command.
     */
      public function handle(PaymentLinkService $paymentLinkService)
    {
        $bookings = Booking::where('payment_status', 'unpaid')
            ->where('status', 'pending')
            ->whereNotNull('last_payment_link_sent_at')
            ->where('last_payment_link_sent_at', '<=', now()->subHours(24))
            ->whereHas('bookingItems', function ($query) {
                $query->whereDate('pickup_date', '>', Carbon::today());
            })
            ->get();

        foreach ($bookings as $booking) {
            $paymentLink = $paymentLinkService->regenerate($booking);

            $bookingItems = BookingItem::where('booking_id', $booking->id)->get();
            $userDetails = BookingUserDetail::where('booking_id', $booking->id)->first();
            $admin = Admin::first();

            $vendorId = $bookingItems->pluck('vendor_id')->filter()->first();
            $vendor = $vendorId ? \App\Models\Vendor::find($vendorId) : null;
        
            $pickupInfo = $vendor
            ? [
                'name' => $vendor->name,
                'street' => $vendor->street,
                'zip_code' => $vendor->zip_code,
                'city' => $vendor->city,
                'state' => $vendor->country 
            ]
            : [
                'name' => $admin->name,
                'street' => $admin->address,
                'zip_code' => $admin->zip_code,
                'city' => $admin->city,
                'state' => $admin->state
            ];
             $productNames = $bookingItems
            ->pluck('product_name')
            ->unique()
            ->implode(', ');

            $rentalFrom = $bookingItems->min('pickup_date');
            $rentalTo   = $bookingItems->max('dropoff_date');
        
            $emailData = [
                'booking' => $booking,
                'customer' => [
                    'first_name' => $userDetails->first_name,
                    'last_name' => $userDetails->last_name,
                ],
                'product_name' => $productNames,
                'rental_period' => $rentalFrom . ' to ' . $rentalTo,
                'total_amount' => number_format($booking->total, 2, ',', '.'),
                'payment_link' => $paymentLink,
                'bookingItems' => $bookingItems,
                'userDetails' => $userDetails,
                'pickupInfo' => $pickupInfo,
                'admin' => $admin
            ];
        
            Mail::send('email.admin-booking-payment-link', $emailData, function ($message) use ($userDetails, $booking) {
                $message->to($userDetails->email)
                        ->subject('Zahlungslink für Ihre Buchung bei Dachboxmiete – 24 Stunden gültig');
            });
            
            Mail::send('email.admin-booking-payment-link', $emailData, function ($message) use ($booking) {
                $message->to('info@dachboxit.de')
                        ->subject('Zahlungslink für Ihre Buchung bei Dachboxmiete – 24 Stunden gültig');
            });
            
            if ($vendor) {
                Mail::send('email.admin-booking-payment-link', $emailData, function ($message) use ($vendor, $booking) {
                    $message->to($vendor->email)
                    ->subject('Zahlungslink für Ihre Buchung bei Dachboxmiete – 24 Stunden gültig');
                });

                 Notification::create([
                    'type' => 'payment_link_expired',
                    'status' => 'Payment Link Expired',
                    'is_read' => false,
                    'user_id' => $userDetails->id,
                    'customer_name' => $userDetails->first_name . ' ' . $userDetails->last_name,
                    'customer_email' => $userDetails->email,
                    'customer_phone' => $userDetails->phone,
                    'vendor_id' => $vendorId,

                    'data' => [
                        [
                            'label' => 'booking_id',
                            'value' => $booking->id,
                        ],
                       
                    ],
                ]);
            }else{
                Notification::create([
                    'type' => 'payment_link_expired',
                    'status' => 'Payment Link Expired',
                    'is_read' => false,
                    'user_id' => $userDetails->id,
                    'customer_name' => $userDetails->first_name . ' ' . $userDetails->last_name,
                    'customer_email' => $userDetails->email,
                    'customer_phone' => $userDetails->phone,

                    'data' => [
                        [
                            'label' => 'Booking ID',
                            'value' => $booking->id,
                        ],
                       
                    ],
                ]);
            }

            $this->info("Payment link resent for booking {$booking->id}");
        }
    }
}
