<?php

namespace App\Exports;

use App\Models\Booking;
use Carbon\Carbon;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;

class BookingsExport implements FromCollection, WithHeadings
{
    protected $timeFilter;
    protected $statusFilter;
    protected $vendorFilter;

    public function __construct($timeFilter, $statusFilter = 'all' , $vendorFilter = null)
    {
        $this->timeFilter = $timeFilter;
        $this->statusFilter = $statusFilter;
        $this->vendorFilter = $vendorFilter;
    }

    public function collection()
    {
        $query = Booking::with(['userDetails', 'items']);
        
        // Apply time filter
        switch ($this->timeFilter) {
            case 'day':
                $query->whereDate('created_at', Carbon::today());
                break;
            case 'week':
                $query->whereBetween('created_at', [
                    Carbon::now()->startOfWeek(),
                    Carbon::now()->endOfWeek()
                ]);
                break;
            case 'month':
                $query->whereMonth('created_at', Carbon::now()->month)
                      ->whereYear('created_at', Carbon::now()->year);
                break;
            case 'year':
                $query->whereYear('created_at', Carbon::now()->year);
                break;
        }
        
        // Apply status filter
        if ($this->statusFilter !== 'all') {
            $query->where('status', $this->statusFilter);
        }

        if ($this->vendorFilter != null) {
            $query->whereHas('items', function($q) {
                $q->where('vendor_id', $this->vendorFilter);
            });
        }else{
            $query->whereHas('items', function($q) {
                $q->where('vendor_id', null);
            });
        }

        $flatData = collect();
            foreach ($query->get() as $booking) {
                foreach ($booking->items as $item) {
                    if(!$item->related_item){

                 
                    $flatData->push([
                        'Booking ID' => $booking->id,
                        'Booking Request ID' => $item->booking_request_id,
                        'Invoice #' => $booking->invoice_number ?? 'N/A',
                        'Customer Name' => optional($booking->userDetails)->first_name . ' ' . optional($booking->userDetails)->last_name,
                        'Email' => optional($booking->userDetails)->email,
                        'Phone' => preg_replace('/^(\+\d{0,2})(\d+)/', '$1 $2', optional($booking->userDetails)->phone) ,
                        
                        'Car' => $item->car_name . ' ' . $item->car_year . ' ' . $item->model,
                        'Roof Type' => $item->roof_type,
                        'Trailer Hitch' => $item->trailer_hitch ? 'Yes' : 'No',
                        'Product' => $item->product_name,
                        'Color' => $item->color,
                        'Size/Variation' => $item->size,
                        'Roof Rack Included' => $item->is_carrier_included ? 'Yes' : 'No',
                        'Insured' => $item->is_insured ? 'Yes' : 'No',
                        'Carrier Length' => $item->carrier_length,
                        'Foot SKU' => $item->foot_sku,
                        'Foot Kit SKU' => $item->footkit_sku,
                        'Pickup Date' => optional($item->pickup_date)->format('d.m.Y'),
                        'Pickup Time' => $item->pickup_time,
                        'Dropoff Date' => optional($item->dropoff_date)->format('d.m.Y'),
                        'Dropoff Time' => $item->dropoff_time,
                        'Days' => $item->days,
                        'Rental Price (€)' => number_format($item->calculated_rent, 2, ',', '.'),
                        'Coupon' => $booking->coupon_code ?? 'N/A',
                        'Discount (€)' => number_format($booking->discount, 2, ',', '.'),
                        'Subtotal (€)' => number_format($booking->subtotal, 2, ',', '.'),
                        'Tax (€)' => number_format($booking->tax, 2, ',', '.'),
                        'Total (€)' => number_format($booking->total, 2, ',', '.'),
                        'Payment Method' => $booking->payment_method ?? 'N/A',
                        'Payment Status' => ucfirst($booking->payment_status ?? 'N/A'),
                        'Refund (€)' => number_format($booking->refund_amount, 2, ',', '.'),
                        'Additional Days' => $item->additional_days,
                        'Delay Fees (€)' => number_format($booking->delay_fees, 2, ',', '.'),
                        'Additional Days Rent(€)' => number_format($booking->additional_days_rent, 2, ',', '.'),
                        'New Dropoff Date' => optional($booking->new_dropoff_date ? \Carbon\Carbon::parse($booking->new_dropoff_date) : null)->format('d.m.Y'),
                        'New Dropoff Time' => optional($item->new_dropoff_time  ? \Carbon\Carbon::parse($item->new_dropoff_time) : null)->format('H:i'),
                        'Damaged Product' => $item->damage ? [$item->product_name] : [],
                        'Delay Link Sent At' => $booking->last_delay_payment_link_sent_at 
    ? \Carbon\Carbon::parse($booking->last_delay_payment_link_sent_at)->format('d.m.Y H:i') 
    : null,
                        'Extra Fees Paid' => ($booking->additional_days_rent + $booking->delay_fees) == 0 
                                    ? 'N/A' 
                                    : (isset($booking->delay_fees_paid) 
                                        ? ($booking->delay_fees_paid ? 'Paid' : 'Pending') 
                                        : 'N/A'),
                        'Status' => ucfirst($booking->status),
                        'Cancelled Reason' => $booking->cancellation_reason ?? '',
                        'Created At' => $booking->created_at->format('d.m.Y H:i'),
                    ]);
                }
                }
            }

            return $flatData;
    }
        
        

    public function headings(): array
    {
        return [
            'Booking ID',
            'Booking Request ID',
            'Invoice #',
            'Customer Name',
            'Email',
            'Phone',
            'Car',
            'Roof Type',
            'Trailer Hitch',
            'Product', 
            'Color',
            'Size/Variation',
            'Roof Rack Included',
            'Insured',
            'Carrier Length',
            'Foot SKU',
            'Foot Kit SKU',
            'Pickup Date',
            'Pickup Time',
            'Dropoff Date',
            'Dropoff Time',
            'Days',
            'Rental Price (€)',
            'Coupon',
            'Discount (€)',
            'Subtotal (€)',
            'Tax (€)',
            'Total (€)',
            'Payment Method',
            'Payment Status',
            'Refund (€)',
            'Additional Days',
            'Delay Fees (€)',
            'Additional Days Rent(€)',
            'New Dropoff Date',
            'New DropOff Time',
            'Damaged Product',
            'Delay Link Sent At',
            'Extra Fees Paid',
            'Status',
            'Cancelled Reason',
            'Created At'
        ];
    }
}

