<?php

namespace App\Exports;

use App\Models\Booking;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Illuminate\Support\Carbon;

class DailyTasksExport implements FromCollection, WithHeadings
{
    protected $fromDate;
    protected $toDate;

    public function __construct($fromDate = null, $toDate = null)
    {
        $this->fromDate = $fromDate;
        $this->toDate   = $toDate;
    }

    /**
     * Build the collection of rows for export
     */
    public function collection()
    {
        $fromDate = $this->fromDate;
        $toDate   = $this->toDate;

        $query = Booking::with(['userDetails', 'items' => function($q) {
            $q->whereNull('related_item'); // ✅ Only include items without related_item
        }])
            ->where('status', 'confirmed')
            ->whereNull('vendor_id')
            ->orderBy('created_at');

        // restrict by date range on any of the item dates
        $query->whereHas('items', function($q) use ($fromDate, $toDate) {
            if ($fromDate && $toDate) {
                $q->whereBetween('pickup_date', [$fromDate, $toDate])
                  ->orWhereBetween('new_dropoff_date', [$fromDate, $toDate])
                  ->orWhereBetween('dropoff_date', [$fromDate, $toDate]);
            }
        });

        // Flatten into rows: each booking item may yield multiple rows
        return $query->get()->flatMap(function ($booking) use ($fromDate, $toDate) {
            return $booking->items->flatMap(function ($item) use ($booking, $fromDate, $toDate) {
                $rows = [];
        
                if ($item->pickup_date && $this->dateInRange($item->pickup_date, $fromDate, $toDate)) {
                    $rows[] = $this->mapRow($item, $booking, 'pickup');
                }
        
                if ($item->new_dropoff_date && $this->dateInRange($item->new_dropoff_date, $fromDate, $toDate)) {
                    $rows[] = $this->mapRow($item, $booking, 'new_dropoff');
                } elseif ($item->dropoff_date && $this->dateInRange($item->dropoff_date, $fromDate, $toDate)) {
                    $rows[] = $this->mapRow($item, $booking, 'dropoff');
                }
        
                return $rows;
            });
        });
        
    }

    /**
     * Column headings
     */
    public function headings(): array
    {
        return [
            'Activity',
            'Date',
            'Time',
            'Product',
            'Size',
            'Insurance',
            'Customer',
            'Vehicle',
            'Roof Type',
            'Roof Carrier',
            'Traverse Length',
            'Foot SKU',
            'Footkit SKU',
            'BookingID',
            'Invoice Number'
        ];
    }

    /**
     * Map a booking item row based on activity type
     */
    protected function mapRow($item, $booking, $type)
    {
        $activity = 'Other';
        $date = 'N/A';
        $time = 'N/A';

        if ($type === 'pickup') {
            $activity = 'Pick-Up';
            $date = Carbon::parse($item->pickup_date)->format('d.m.Y');
            $time = $item->pickup_time ? Carbon::parse($item->pickup_time)->format('H:i') : 'N/A';
        }

        if ($type === 'new_dropoff') {
            $activity = 'Drop-Off (Delayed)';
            $date = Carbon::parse($item->new_dropoff_date)->format('d.m.Y');
            $time = $item->new_dropoff_time ? Carbon::parse($item->new_dropoff_time)->format('H:i') : 'N/A';
        }

        if ($type === 'dropoff') {
            $activity = 'Drop-Off';
            $date = Carbon::parse($item->dropoff_date)->format('d.m.Y');
            $time = $item->dropoff_time ? Carbon::parse($item->dropoff_time)->format('H:i') : 'N/A';
        }

        return [
            $activity,
            $date,
            $time,
            $item->product_name ?? 'N/A',
            $item->size ?? 'N/A',
            $item->is_insured ? 'Yes' : 'No',
            $booking->userDetails ? $booking->userDetails->first_name.' '.$booking->userDetails->last_name : 'N/A',
            trim(($item->car_name ?? '') . ' ' . ($item->car_year ?? '') . ' ' . ($item->model ?? '')),
            $item->roof_type ?? 'N/A',
            $item->is_carrier_included ? 'Yes' : 'No',
            $item->carrier_length ?? 'N/A',
            $item->foot_sku ?? 'N/A',
            $item->footkit_sku ?? 'N/A',
            $booking->id,
            $booking->invoice_number ?? 'N/A'
        ];
    }

    protected function dateInRange($date, $fromDate, $toDate)
    {
        $d = \Illuminate\Support\Carbon::parse($date);
        if ($fromDate && $toDate) {
            return $d->betweenIncluded(Carbon::parse($fromDate), Carbon::parse($toDate));
        }
        return true; // if no filter applied
    }
}
