<?php
namespace App\Exports;

use App\Models\Booking;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;

class RevenueReportExport implements FromCollection, WithHeadings, WithMapping
{
    protected $fromDate;
    protected $toDate;

    public function __construct($fromDate = null, $toDate = null)
    {
        $this->fromDate = $fromDate;
        $this->toDate = $toDate;
    }

    public function collection()
    {
       
        return Booking::with(['items', 'userDetails'])
            ->where('status', 'confirmed')
            ->where('vendor_id',null)
            ->when($this->fromDate, fn($q) => $q->whereDate('created_at', '>=', $this->fromDate))
            ->when($this->toDate, fn($q) => $q->whereDate('created_at', '<=', $this->toDate))
            ->get();
    }

    public function headings(): array
    {
        return [
            'Booking ID', 'Created At','Customer Name', 'Address', 'Email', 'Phone Number',
            'Brand', 'Model', 'Year', 'Railing Type', 'Trailer Hitch',
            'Rental Period', 'Pickup Date/Pickup Time', 'Drop-Off Date/DropOff Time', 'Products',
            'Mounting', 'Insurance', 'Subtotal €', 'Discount €', 'Insurance Net Price €',
            'Net Price (Booking) €', 'Tax (19%) €', 'Total Amount €',
            'Refund €', 'Delay Fees €','Additional Rent Fees €','Canceled', 'Damaged', 'Delayed','Additional Days' , 'New DropOff Date','New PickUp Time'
        ];
    }

    public function map($booking): array
    {
        $item = $booking->items->first();
        $user = $booking->userDetails;

        $taxRate = 0.19;
        $insuranceFlatRate = 25;

        $insuranceNet = 0;
        $insuranceTax = 0;


        if ($item->is_insured) {
            $insuranceNet = $insuranceFlatRate / (1 + $taxRate);
            $insuranceTax = $insuranceNet * $taxRate;
        }

        $taxableAmount = $item->is_insured ? $item->calculated_rent - $insuranceFlatRate : $item->calculated_rent;
        $net = $taxableAmount / (1 + $taxRate);
        $tax = $net * $taxRate;

        $subtotal = $net + $tax;
        $discount = $booking->discount;
        $total = $subtotal + $insuranceNet + $insuranceTax - $discount;

        $pickupDateTime = \Carbon\Carbon::parse($item->pickup_date->format('Y-m-d') . ' ' . $item->pickup_time);
        $dropoffDateTime = \Carbon\Carbon::parse($item->dropoff_date->format('Y-m-d') . ' ' . $item->dropoff_time);
        $newDropOffDate = optional($booking->new_dropoff_date ? \Carbon\Carbon::parse($booking->new_dropoff_date) : null)->format('d.m.Y');

        $newPickupTime = optional($item->new_dropoff_time ? \Carbon\Carbon::parse($item->new_dropoff_time) : null)->format('H:i');


        return [
            $booking->id,
            $booking->created_at->format('d.m.Y'),
            $user ? $user->first_name . ' ' . $user->last_name : '',
            $user ? $user->address . ', ' . $user->zip . ', ' . $user->city . ', ' . $user->country : '',
            $user->email ?? '',
            $user->phone ?? '',
            $item->car_name ?? '',
            $item->model ?? '',
            $item->car_year ?? '',
            $item->roof_type ?? '',
            $item->trailer_hitch ? 'Yes' : 'No',
            $item->days . ' days',
            $pickupDateTime->format('d.m.Y H:i'),
            $dropoffDateTime->format('d.m.Y H:i') ,
            $item->product_name ?? '',
            $item->is_carrier_included ? 'Yes' : 'No',
            $item->is_insured ? 'Yes' : 'No',
            number_format($booking->subtotal, 2, ',' , '.'),
            number_format($booking->discount, 2, ',' , '.'),
            number_format($insuranceNet, 2, ',' , '.'),
            number_format($net, 2, ',' , '.'),
            number_format($tax + $insuranceTax, 2, ',' , '.'),
            number_format($total, 2, ',' , '.'),
            number_format($booking->refund_amount, 2, ',' , '.'),
            number_format($booking->delay_fees, 2, ',' , '.'),
            number_format($booking->additional_days_rent, 2, ',' , '.'),
            $booking->cancellation_reason ? 'Yes' : 'No',
            $item->damage ? [$item->product_name] : [],
            $item->delay ? 'Yes' : 'No',
            $item->additional_days,
            $newDropOffDate,
            $newPickupTime

        ];
    }
}
