<?php

namespace App\Exports;

use App\Models\Product;
use App\Models\BookingItem;
use App\Models\Vendor;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;

class SPOProductsExport implements FromCollection, WithHeadings, WithMapping
{
    protected $vendor;
    protected $rentedCounts;

    public function __construct($vendor = null)
    {
        $this->vendor = $vendor;
        $this->rentedCounts = $this->getRentedItemsCount();
    }

    public function collection()
    {
        $query = Product::with([
            'category',
            'variations.options.color',
            'variations.size',
            'variations.variation',
        ]);

        if ($this->vendor) {
            $query = $this->vendor->products()->with([
                'category',
                'variations.options.color',
                'variations.size',
                'variations.variation',
            ]);
        }

        $products = $query->get();

        // inject vendor quantities
        foreach ($products as $product) {
            foreach ($product->variations as $variation) {
                foreach ($variation->options as $option) {
                    $vendorQty = $option->vendorQuantities()
                        ->where('vendor_id', $this->vendor->id ?? null)
                        ->first();
                    $option->quantity = $vendorQty?->quantity ?? 0;
                }
            }
        }

        return $products;
    }

    public function headings(): array
    {
        $baseHeadings = [
            'Product Name',
            'Category',
            'SKU',
            'Size/Variation',
            'Color',
            'Total Quantity',
            'Remaining Quantity',
            'Dimensions (LxWxH)',
            'Weight',
            'Volume',
            'Max Load',
            'Status',
            'Created At',
        ];

        if ($this->vendor) {
            array_unshift($baseHeadings, 'SPO ID');
        } else {
            array_unshift($baseHeadings, 'Product ID');
        }

        return $baseHeadings;
    }

    public function map($product): array
    {
        $rows = [];

        // Simple product
        if ($product->variations->isEmpty()) {
            $rentedCount = $this->rentedCounts['product'][$product->id] ?? 0;
            $remainingQty = (int) ($product->quantity ?? 0);
            $totalQuantity = $remainingQty - (int) $rentedCount;

            $rows[] = [
                $this->vendor ? 'SPO' . $this->vendor->id : $product->id,
                $product->name,
                $product->category->name ?? '-',
                $product->sku ?? '-',
                '-',
                $product->color->name ?? '-',
                $this->formatQuantity($remainingQty),
                $this->formatQuantity($totalQuantity),
                $this->formatDimensions($product->length, $product->width, $product->height),
                $this->formatQuantity($product->weight),
                $this->formatQuantity($product->volume),
                $this->formatQuantity($product->max_load),
                $product->status ?? '-',
                $product->created_at->format('d.m.Y')
            ];
        } else {
            foreach ($product->variations as $variation) {
                // With options
                foreach ($variation->options as $option) {
                    $rentedCount = $this->rentedCounts['option'][$option->id] ?? 0;
                    $remainingQty = (int) ($option->quantity ?? 0);
                    $totalQuantity = $remainingQty - (int) $rentedCount;

                    $rows[] = [
                        $this->vendor ? 'SPO' . $this->vendor->id : $product->id,
                        $product->name,
                        $product->category->name ?? '-',
                        $option->sku ?? '-',
                        $variation->size->name ?? $variation->variation->name ?? '-',
                        $option->color->name ?? '-',
                        $this->formatQuantity($remainingQty),
                        $this->formatQuantity($totalQuantity),
                        $this->formatDimensions($variation->length, $variation->width, $variation->height),
                        $this->formatQuantity($variation->weight),
                        $this->formatQuantity($variation->volume),
                        $this->formatQuantity($variation->max_load),
                        $variation->status ?? '-',
                        $product->created_at->format('d.m.Y')
                    ];
                }

                // Without options
                if ($variation->options->isEmpty()) {
                    $rentedCount = $this->rentedCounts['variation'][$variation->id] ?? 0;
                    $remainingQty = (int) ($variation->quantity ?? 0);
                    $totalQuantity = $remainingQty - (int) $rentedCount;

                    $rows[] = [
                        $this->vendor ? 'SPO' . $this->vendor->id : $product->id,
                        $product->name,
                        $product->category->name ?? '-',
                        $variation->sku ?? '-',
                        $variation->size->name ?? $variation->variation->name ?? '-',
                        $variation->color->name ?? '-',
                        $this->formatQuantity($remainingQty),
                        $this->formatQuantity($totalQuantity),
                        $this->formatDimensions($variation->length, $variation->width, $variation->height),
                        $this->formatQuantity($variation->weight),
                        $this->formatQuantity($variation->volume),
                        $this->formatQuantity($variation->max_load),
                        $variation->status ?? '-',
                        $product->created_at->format('d.m.Y')
                    ];
                }
            }
        }

        return $rows;
    }

    private function getRentedItemsCount()
    {
        $today = now()->format('Y-m-d');

        $query = BookingItem::where('damage', '!=', 1)
            ->where('already_booked', 0)
            ->whereHas('booking', function ($q) {
                $q->where('status', 'confirmed');
            })
            ->where(function ($query) use ($today) {
                $query->where(function ($q) use ($today) {
                    $q->where('pickup_date', '<=', $today)
                        ->where(function ($q2) use ($today) {
                            $q2->where(function ($q3) use ($today) {
                                $q3->whereNotNull('new_dropoff_date')
                                    ->where('new_dropoff_date', '>=', $today);
                            })->orWhere(function ($q4) use ($today) {
                                $q4->whereNull('new_dropoff_date')
                                    ->where('dropoff_date', '>=', $today);
                            });
                        });
                });
            });

        if ($this->vendor) {
            $query->where('vendor_id', $this->vendor->id);
        }

        $rentedItems = $query->get();

        $rentedCounts = [
            'product' => [],
            'variation' => [],
            'option' => []
        ];

        foreach ($rentedItems as $item) {
            if ($item->product_id) {
                $rentedCounts['product'][$item->product_id] = ($rentedCounts['product'][$item->product_id] ?? 0) + 1;
            }

            if ($item->product_id && $item->size) {
                $product = Product::find($item->product_id);
                if ($product && $product->variations) {
                    foreach ($product->variations as $variation) {
                        if (($variation->size && $variation->size->name == $item->size) ||
                            ($variation->variation && $variation->variation->name == $item->size)) {

                            $rentedCounts['variation'][$variation->id] = ($rentedCounts['variation'][$variation->id] ?? 0) + 1;
                            $name = strtolower($item->product_name);

                            if (in_array($name, ['carrier', 'foot', 'footkit', 'traverse', 'fuß', 'fußkit'])) {
                                $item->color = 'Schwarz';
                            }
                            if ($item->color && $variation->options) {
                                foreach ($variation->options as $option) {
                                    if ($option->color && $option->color->name == $item->color) {
                                        $rentedCounts['option'][$option->id] = ($rentedCounts['option'][$option->id] ?? 0) + 1;
                                        break;
                                    }
                                }
                            }
                            break;
                        }
                    }
                }
            }
        }

        return $rentedCounts;
    }

    private function formatDimensions($length, $width, $height)
    {
        $length = $length ?? 0;
        $width = $width ?? 0;
        $height = $height ?? 0;
        
        return "{$length}x{$width}x{$height}";
    }

    private function formatQuantity($value)
    {
        // Convert null to 0 and ensure it's displayed as a string "0"
        return ($value === null || $value === '') ? '0' : (string) $value;
    }
}