<?php

namespace App\Exports;

use App\Models\User;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Illuminate\Http\Request;

class UsersExport implements FromCollection, WithHeadings
{
    protected $request;

    public function __construct(Request $request)
    {
     
        $this->request = $request;
    }

    public function collection()
    {
        $query = User::where('is_guest', '!=', 1)
        ->with('bookings')
        ->select('id', 'first_name', 'last_name', 'email', 'country_code', 'number', 'street', 'city', 'region', 'country','zip_code', 'status', 'created_at','marketing_aggrement');
        
        if ($this->request->has('search')) {
            $searchQuery = $this->request->input('search');
            $query->where(function($q) use ($searchQuery) {
                $q->where('first_name', 'like', "%{$searchQuery}%")
                  ->orWhere('last_name', 'like', "%{$searchQuery}%")
                  ->orWhere('email', 'like', "%{$searchQuery}%")
                  ->orWhere('street', 'like', "%{$searchQuery}%");
            });
        }
    
        if ($this->request->has('time_filter') && $this->request->time_filter != 'year') {
            $timeFilter = $this->request->input('time_filter');
            switch ($timeFilter) {
                case 'week':
                    $query->where('created_at', '>=', now()->subWeek());
                    break;
                case 'month':
                    $query->where('created_at', '>=', now()->subMonth());
                    break;
                case 'day':
                    $query->whereDate('created_at', today());
                    break;
            }
        } else {
            $query->where('created_at', '>=', now()->subYear());
        }
    
        return $query->get()->map(function ($user) {
            $confirmedCompletedCount = $user->bookings->whereIn('status', ['confirmed', 'completed'])->count();
            $cancelledCount = $user->bookings->where('status', 'cancelled')->count();
            $marketing = $user->marketing_aggrement == 1 ? 'Yes' : 'No';
        
            return [
                $user->id,
                $user->first_name,
                $user->last_name,
                $user->email,
                $user->country_code . ' '.$user->number,
                $user->street . ',' . $user->zip_code . ',' . $user->city . ',' . $user->region . ',' . $user->country,
                $confirmedCompletedCount,
                $cancelledCount,
                $marketing,
                $user->status == 1 ? 'Active' : 'Blocked',
                $user->created_at->format('d.m.Y H:i'),

            ];
        });
    }

    public function headings(): array
    {
        return [
            'ID',
            'First Name',
            'Last Name',
            'Email',
            'Phone Number',
            'Address',
            'Confirmed + Completed Bookings',
            'Cancelled Bookings',
            'Marketing Agreement',
            'Status',
            'Created At'
        ];
    }
}