<?php

namespace App\Http\Controllers\Admin\Auth;

use App\Http\Controllers\Controller;
use App\Mail\AdminForgetMail;
use App\Models\Admin;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{

    public function index()
    {
        return view("admin.auth.login");
    }

    public function login(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'email' => 'required|max:50',
            'password' => 'required',
        ]);

        if ($validator->fails()) {

            $errors = $validator->errors()->all();
            foreach ($errors as $error) {
                toastr()->error($error);
            }
            return redirect()->back();
        }
        // Attempt to login
        if (Auth::guard('admin')->attempt(['email' => $request->email, 'password' => $request->password], $request->remember_me)) {
            // Redirect to dashboard
            $admin =  Admin::where('id', Auth::guard('admin')->user()->id)->first();
            if ($admin->status != 1) {
                toastr()->error('Account is Inactive');
                return redirect()->back();
            }
            toastr()->success('Login successfully!');

            return redirect()->route('admin.dashboard');
        }
        toastr()->error('Please enter valid email address or password.');
        return back();
    }

    public function logout(Request $request)
    {
        Auth::guard('admin')->logout();
        toastr()->success('Logout successfully!');
        return redirect()->route('admin.login');
    }

    public function forgotPasswordView()
    {
        return view('admin.auth.forgot-password');
    }

    public function forgotPassword(Request $request)
    {
    

        $request->validate(['email' => 'required|email']);

        $admin = Admin::where('email', $request->email)->first();

        if (!$admin) {
            return back()->with('error','Please enter registered email address.');
        }
        $existingToken = DB::table('password_reset_tokens')
            ->where('email', $admin->email)
            ->first();

        if ($existingToken) {
            $tokenCreatedAt = Carbon::parse($existingToken->created_at);
            $currentTime = Carbon::now();

            if ($tokenCreatedAt->diffInMinutes($currentTime) <= 15) {
                try{

                    Mail::to($admin->email)->send(new AdminForgetMail($existingToken->token, $admin->first_name));
                }catch(Exception $e){
                 
                }
                
                return back()->with(['success' => 'Password reset link sent successfully']);
            } else {

                DB::table('password_reset_tokens')
                    ->where('email', $admin->email)
                    ->update([
                        'token' => Str::random(60),
                        'created_at' => now(),
                    ]);


                return back()->with(['success' => 'Password reset link sent successfully']);
            }
        }

        $token = Str::random(60);

        DB::table('password_reset_tokens')->insert([
            'email' => $admin->email,
            'token' => $token,
            'created_at' => now(),
        ]);

        Mail::to($admin->email)->send(new AdminForgetMail($token, $admin->first_name));

        return back()->with(['success' => 'Password reset link sent successfully']);
    }

    public function showResetForm($token)
    {
        $existingToken = DB::table('password_reset_tokens')
            ->where('token', $token)
            ->first();

        if ($existingToken) {
            $tokenCreatedAt = Carbon::parse($existingToken->created_at);
            $currentTime = Carbon::now();

            if ($tokenCreatedAt->diffInMinutes($currentTime) <= 15) {

                return view('admin.auth.reset-password', ['token' => $token]);
            } else {

                DB::table('password_reset_tokens')
                    ->where('token', $token)
                    ->delete();

                return back()->withErrors(['email' => 'Password reset link has expired. Please request a new one.']);
            }
        }
        return view('admin.auth.reset-password', ['token' => $token]);
    }

    public function reset(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'password' => 'required|min:8',
            'token' => 'required',
        ]);
        if ($validator->fails()) {

            $errors = $validator->errors()->all();
            foreach ($errors as $error) {
                toastr()->error($error);
            }
            return redirect()->back();
        }
        $resetRecord = DB::table('password_reset_tokens')
            ->where('token', $request->token)
            ->first();

        if (!$resetRecord) {
            return back()->withErrors(['email' => 'Invalid Token']);
        }

        $admin = Admin::where('email', $resetRecord->email)->first();

        if (!$admin) {
            return back()->withErrors(['email' => 'Please enter registered email address.']);
        }

        $admin->update(['password' => Hash::make($request->password)]);

        $resetRecord = DB::table('password_reset_tokens')
            ->where('email', $resetRecord->email)
            ->where('token', $request->token)
            ->delete();

        return redirect()->route('admin.login')->with('success','Password Reset Successfully!');
    }
}
