<?php

namespace App\Http\Controllers\Admin\Booking;
use App\Mail\GuestAccountMail;
use App\Http\Controllers\Controller;
use App\Mail\BookingCancellationMail;
use App\Models\Booking;
use App\Models\BookingItem;
use App\Models\BookingUserDetail;
use Illuminate\Support\Facades\Validator;
use App\Models\Category;
use App\Models\Color;
use App\Models\MotherFile;
use App\Models\Product;
use App\Models\ProductPrice;
use App\Models\Size;
use App\Models\User;
use App\Models\UserAddressDetail;
use App\Models\Notification;
use App\Models\Variation;
use App\Models\CartItem;
use App\Models\Voucher;
use App\Models\WorkingHours;
use Carbon\Carbon;
use DB;
use App\Models\Admin;
use Hash;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Str;
use Stripe\Stripe;
use Stripe\Checkout\Session;
use Illuminate\Support\Facades\Mail;
use App\Exports\BookingsExport;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\Holidays;
use App\Models\Vendor;
use App\Models\BookingRequest;


use App\Models\ProductVariationOption;
use App\Models\ProductVariation;
class BookingController extends Controller
{
    public $filePath='admin.booking';

    // public function index()
    // {
    //     $bookings = Booking::with(['userDetails', 'items'])
    //     ->orderBy('created_at', 'desc')
    //     ->get();
    //     return view($this->filePath.'.index',compact('bookings'));
    // }

    public function index(Request $request)
    {
        $query = Booking::with(['userDetails', 'items']);
        
        $timeFilter = $request->input('time_filter', 'year');
        $this->applyTimeFilter($query, $timeFilter);
        
        if ($request->has('search')) {
            $searchQuery = $request->input('search');
            $query->whereHas('userDetails', function($q) use ($searchQuery) {
                $q->where('first_name', 'like', "%{$searchQuery}%")
                  ->orWhere('last_name', 'like', "%{$searchQuery}%")
                  ->orWhere('email', 'like', "%{$searchQuery}%")
                  ->orWhere('number', 'like', "%{$searchQuery}%");
            });
        }

        if ($request->filled('vendor_id')) {
            $query->whereHas('items', function($q) use ($request) {
                $q->where('vendor_id', $request->vendor_id);
            });
        }else{
            $query->whereHas('items', function($q) use ($request) {
                $q->where('vendor_id', null);
            });
        }
        


        if ($request->has('status') && $request->status != 'all') {
            $query->where('status', $request->status);
        }
        
        $bookings = $query->orderBy('created_at', 'desc')->get();


        foreach ($bookings as $booking) {
            $booking->highlight_pickup = false;
        
            if ($booking->items->isNotEmpty()) {
                foreach ($booking->items as $item) {
                    $pickupDate = $item->pickup_date instanceof \Carbon\Carbon
                        ? $item->pickup_date->format('Y-m-d')
                        : $item->pickup_date;
        
                    $pickupTime = $item->pickup_time;
        
                    // Check if any other booking item has a new dropoff on the same date
                    // and time greater than pickup time
                    $conflictingBookingIds = \App\Models\BookingItem::where('booking_id', '!=', $booking->id)
                        ->whereNotNull('new_dropoff_date')
                        ->whereNotNull('new_dropoff_time')
                        ->whereDate('new_dropoff_date', $pickupDate)
                        ->whereTime('new_dropoff_time', '>', $pickupTime)
                        ->whereHas('booking', function ($q) use ($request) {
                            $q->where('status', 'confirmed');

                            // Vendor filter
                            if ($request->filled('vendor_id')) {
                                $q->where('vendor_id', $request->vendor_id);
                            } else {
                                $q->whereNull('vendor_id');
                            }
                        })
                        ->pluck('booking_id')
                        ->unique();
        
                        if ($conflictingBookingIds->isNotEmpty()) {
                            $booking->highlight_pickup = true;
                            $booking->conflicting_booking_ids = $conflictingBookingIds->implode(', ');
                        }
                }
            }
        }
        

        $vendors = Vendor::select('id', 'street', 'zip_code', 'city')->where('status','active')->get();

        // $analytics = $this->calculateBookingAnalytics($timeFilter);
        $analytics = $this->calculateBookingAnalytics($request);
        
        return view($this->filePath.'.index', compact('bookings', 'analytics', 'timeFilter','vendors'));
    }
    public function downloadPDF(Request $request)
    {

        $query = Booking::with(['userDetails', 'items']);
        $timeFilter = $request->input('time_filter', 'year');
        $statusFilter = $request->input('status', 'all');
        $vendorFilter = $request->input('vendor_id', null);

        if ($request->has('time_filter')) {
            $this->applyTimeFilter($query, $request->time_filter);
        }
        if ($request->has('status') && $statusFilter != 'all') {
            $this->applyStatusFilter($query, $request->status);
        }

        if($request->has('vendor_id')){
            $vendors = Vendor::get()->keyBy('id');

            $selectedVendor = $vendorFilter && isset($vendors[$vendorFilter])
                ? $vendors[$vendorFilter]
                : null;
            $query->whereHas('items', function($q) use ($request) {
                $q->where('vendor_id', $request->vendor_id);
            });
        }else{
            $query->whereHas('items', function($q) use ($request) {
                $q->where('vendor_id', null);
            });
        }

        

        $bookings = $query->orderBy('created_at', 'desc')->get();
        $pdf = \PDF::loadView($this->filePath.'.pdf', compact('bookings', 'timeFilter', 'statusFilter','vendorFilter','selectedVendor'))
                ->setPaper('a4', 'landscape');
        
        return $pdf->download('bookings_report.pdf');
    }

    public function downloadCSV(Request $request)
    {
        $timeFilter = $request->input('time_filter', 'year');
        $statusFilter = $request->input('status', 'all');
        $vendorFilter = $request->input('vendor_id', null);
        return \Excel::download(new BookingsExport($timeFilter,$statusFilter,$vendorFilter), 'bookings_report.csv');
    }

    private function applyTimeFilter($query, $timeFilter)
    {
        switch ($timeFilter) {
            case 'day':
                $query->whereDate('created_at', Carbon::today());
                break;
            case 'week':
                $query->whereBetween('created_at', [
                    Carbon::now()->startOfWeek(),
                    Carbon::now()->endOfWeek()
                ]);
                break;
            case 'month':
                $query->whereMonth('created_at', Carbon::now()->month)
                      ->whereYear('created_at', Carbon::now()->year);
                break;
            case 'year':
                $query->whereYear('created_at', Carbon::now()->year);
                break;
        }
    }


    private function applyStatusFilter($query,$statusFilter){
        $query->where('status', $statusFilter);
    }

    // private function calculateBookingAnalytics($timeFilter)
    // {
       
    //     $query = Booking::query()->where('payment_status', 'paid')->where('status','confirmed');
    //     $this->applyTimeFilter($query, $timeFilter);
        
    //     $netRevenue = Booking::query()->whereIn('status', ['confirmed', 'completed'])->where('payment_status', 'paid')->sum('subtotal');
        
    //     $totalBookings = Booking::query()
    //         ->whereIn('status', ['confirmed', 'completed'])
    //         ->count();
        
    //     $cancellations = Booking::query()->where('status', 'cancelled')->count();
        
    //     $totalRevenue = Booking::query()->whereIn('status', ['confirmed', 'completed'])->where('payment_status', 'paid')->sum('total');
        
    //     return [
    //         'net_revenue' => number_format($netRevenue, 2 , ',' , '.'),
    //         'total_bookings' => $totalBookings,
    //         'cancellations' => $cancellations,
    //         'total_revenue' => number_format($totalRevenue, 2 , ',' , '.')
    //     ];
    // }

    private function calculateBookingAnalytics($request)
    {
        $timeFilter = $request->input('time_filter', 'year');
        
        // Base query for confirmed/completed paid bookings
        $confirmedQuery = Booking::query()
            ->whereIn('status', ['confirmed', 'completed','cancelled']);

        if ($request->filled('vendor_id')) {
            $refundQuery = Booking::query()
                ->where('payment_status', 'refund')
                ->where('vendor_id',$request->vendor_id)
                ->whereIn('status', ['confirmed', 'completed', 'cancelled']);
        }else{
            $refundQuery = Booking::query()
            ->where('payment_status', 'refund')
            ->where('vendor_id',null)
            ->whereIn('status', ['confirmed', 'completed', 'cancelled']);
        }
               
        

        $this->applyTimeFilter($confirmedQuery, $timeFilter);
        $this->applyTimeFilter($refundQuery, $timeFilter);
        
        // Apply vendor filter if present

        if ($request->filled('vendor_id')) {
            $confirmedQuery->whereHas('items', function($q) use ($request) {
                $q->where('vendor_id', $request->vendor_id);
            });
        } else {
            $confirmedQuery->whereHas('items', function($q) use ($request) {
                $q->where('vendor_id', null);
            });
        }
        
        // Apply search filter if present
        if ($request->has('search')) {
            $searchQuery = $request->input('search');
            $confirmedQuery->whereHas('userDetails', function($q) use ($searchQuery) {
                $q->where('first_name', 'like', "%{$searchQuery}%")
                ->orWhere('last_name', 'like', "%{$searchQuery}%")
                ->orWhere('email', 'like', "%{$searchQuery}%")
                ->orWhere('number', 'like', "%{$searchQuery}%");
            });
        }
        
        // Query for cancellations
        $cancelledQuery = Booking::query()->where('status', 'cancelled');
        $this->applyTimeFilter($cancelledQuery, $timeFilter);
        
        // Apply vendor filter to cancellations if present
        if ($request->filled('vendor_id')) {
            $cancelledQuery->whereHas('items', function($q) use ($request) {
                $q->where('vendor_id', $request->vendor_id);
            });
        } else {
            $cancelledQuery->whereHas('items', function($q) use ($request) {
                $q->where('vendor_id', null);
            });
        }
        
        // Apply search filter to cancellations if present
        if ($request->has('search')) {
            $searchQuery = $request->input('search');
            $cancelledQuery->whereHas('userDetails', function($q) use ($searchQuery) {
                $q->where('first_name', 'like', "%{$searchQuery}%")
                ->orWhere('last_name', 'like', "%{$searchQuery}%")
                ->orWhere('email', 'like', "%{$searchQuery}%")
                ->orWhere('number', 'like', "%{$searchQuery}%");
            });
        }
        
        // Calculate metrics
        $refundAmount = $refundQuery->sum('refund_amount');
        $netRevenue = $confirmedQuery->sum('subtotal') - $refundAmount;
        $totalBookings = $confirmedQuery->count();
        $cancellations = $cancelledQuery->count();
        $totalRevenue = $confirmedQuery->sum('total')- $refundAmount;
        
        return [
            'net_revenue' => number_format($netRevenue, 2, ',', '.'),
            'total_bookings' => $totalBookings,
            'cancellations' => $cancellations,
            'total_revenue' => number_format($totalRevenue, 2, ',', '.')
        ];
    }


    public function addBooking()
    {
        $manufacturers = MotherFile::distinct()->pluck('car_make');
        $products = Product::with('brand', 'category')->get();
        // dd($products);
        $roofTypes = [
            'Normal Roof',
            'Normal roof without glass',
            'Normal Roof with glass',
            'Fixpoints',
            'Fixpoints without glass',
            'Fixpoints with glass',
            'Integrated Roof Railing',
            'Elevated Roof Rails',
            'T-Profile',
            'Raingutter'
        ];
        $holidays = Holidays::pluck('date')->toArray();

        return view($this->filePath.'.add_booking',compact('manufacturers', 'products', 'roofTypes','holidays'));
    }

    public function bookingDetail(Request $request)
    {
        $booking = Booking::with(['userDetails', 'items'])->where('id', $request->id)
        ->orderBy('created_at', 'desc')
        ->first();
        $vendors = Vendor::select('id', 'street', 'zip_code', 'city')->where('status','active')->get();
        // dd($booking);
        return view($this->filePath.'.booking_detail', compact('booking','vendors'));
    }
    public function editBookingDetail2(Request $request)
    {
        $booking = Booking::with(['userDetails', 'items'])->findOrFail($request->id);

     
        return view($this->filePath.'.edit_booking_detail', compact('booking'));
    }

    public function editBookingDetail(Request $request)
    {
        $booking = Booking::with(['userDetails', 'items.product'])->findOrFail($request->id);

        $warnings = [];
        foreach ($booking->items as $item) {
            $impact = $this->computeImpactOfMarkingDamaged($item);
            if (!empty($impact['affected_bookings'])) {
                $warnings[] = [
                    'booking_item_id' => $item->id,
                    'product_name' => $item->product->name ?? 'Product',
                    'message' => $impact['message'],
                    'affected_bookings' => $impact['affected_bookings'],
                ];
            }
        }

        return view($this->filePath.'.edit_booking_detail', compact('booking', 'warnings'));
    }

    protected function computeImpactOfMarkingDamaged(BookingItem $item)
    {
        $productId = $item->product_id;
        $product = Product::where('id',$productId)->first();
        $colorName = $item->color;
        $colorId= Color::where('name',$colorName)->first()->id;
        $sizeName  = $item->size;
        $vendorId  = $item->vendor_id ?? 'admin';
    
        $from = Carbon::parse($item->pickup_date)->startOfDay();
        $to   = Carbon::parse($item->new_dropoff_date ?? $item->dropoff_date)->startOfDay();


        $productName = strtolower($product->name);
        $productVariations = collect();

        $productNameNormalized = strtolower(trim($productName));

        if ($productNameNormalized === 'carrier' || $productNameNormalized === 'traverse') {
            $carrierLength = $item->carrier_length;
            $sizeValue = intval(preg_replace('/[^0-9]/', '', $carrierLength));
            $item->size = $sizeValue/10;
            $sizeId = Size::whereRaw('LOWER(name) = ?', [strtolower(trim($item->size))])->first()?->id;
             $productVariations = $product->variations()
                ->where('size_id', $sizeId)
                ->with(['options' => function($q) use ($colorId) {
                    if ($colorId) $q->where('color_id', $colorId);
                }])
                ->get();
                
        }elseif (preg_match('/\b(foot kit|footkit)\b/i', $productName) || ($productNameNormalized == 'fußkit')) {
            $footkitSku = $item->footkit_sku;
            $variationId = Variation::where('name', $footkitSku)->first()?->id;
            
            $productVariations = $product->variations()
                ->where('variation_id', $variationId)
                ->with(['options' => function($q) use ($colorId) {
                    if ($colorId) $q->where('color_id', $colorId);
                }])
                ->get();

                
        } elseif (preg_match('/\bfoot\b/i', $productName) || ($productNameNormalized == 'fuß')) {
            $footSku = $item->foot_sku;
            $variationId = Variation::where('name', $footSku)->first()?->id;
            
            $productVariations = $product->variations()
                ->where('variation_id', $variationId)
                ->with(['options' => function($q) use ($colorId) {
                    if ($colorId) $q->where('color_id', $colorId);
                }])
                ->get();

                
        } else {
            $sizeId = Size::whereRaw('LOWER(name) = ?', [strtolower(trim($item->size))])->first()?->id;

            if($sizeId == null){
                $sizeId = Variation::whereRaw('LOWER(name) = ?', [strtolower(trim($item->size))])->first()?->id;
                $productVariations = $product->variations()
                ->where('variation_id', $sizeId)
                ->with(['options' => function($q) use ($colorId) {
                    if ($colorId) $q->where('color_id', $colorId);
                }])
                ->get();
            }else{
                $productVariations = $product->variations()
                    ->where('size_id', $sizeId)
                    ->with(['options' => function($q) use ($colorId) {
                        if ($colorId) $q->where('color_id', $colorId);
                    }])
                    ->get();
            }

           


        }

    
        $totalQty = 0;
        foreach ($productVariations as $variation) {
            foreach ($variation->options as $option) {
                if ($vendorId !== 'admin') {
                    $vendorQty = $option->vendorQuantities()->where('vendor_id', $vendorId)->first();
                    $totalQty += $vendorQty?->quantity ?? 0;
                } else {
                    $totalQty += $option->quantity ?? 0;
                }
            }
        }

      
    
        $query = BookingItem::query()
            ->where('product_id', $productId)
            ->where('color', $colorName)
            ->where('size', $sizeName)
            ->whereHas('booking', fn($q) => $q->where('status','confirmed'))
            ->where('already_booked', '!=', 1)
            ->where('damage', '!=', 1)
            ->where('id', '!=', $item->id);
    
        if ($vendorId !== 'admin') {
            $query->where('vendor_id', $vendorId);
        } else {
            $query->whereNull('vendor_id');
        }
    
        $otherBookings = $query
            ->whereDate('pickup_date', '>=', $from->toDateString())
            ->get();

    
        $maxEndDate = $to->copy();
        foreach ($otherBookings as $b) {
            $bEnd = Carbon::parse($b->new_dropoff_date ?? $b->dropoff_date)->startOfDay();
            if ($bEnd->gt($maxEndDate)) {
                $maxEndDate = $bEnd;
            }
        }
    
        $periodStart = $from->copy()->startOfDay();
        $period = [];
        for ($d = $periodStart; $d->lte($maxEndDate); $d->addDay()) {
            $period[] = $d->format('Y-m-d');
        }
    
        $dailyBookings = [];
        $bookingById = [];
    
        foreach ($otherBookings as $b) {
            $bStart = Carbon::parse($b->pickup_date)->startOfDay();
            $bEnd = Carbon::parse($b->new_dropoff_date ?? $b->dropoff_date)->startOfDay();
            $bQty = intval($b->quantity ?? 1);
            
            $pickupDate = $bStart->format('Y-m-d');
            if (!isset($dailyBookings[$pickupDate])) {
                $dailyBookings[$pickupDate] = ['pickups' => 0, 'dropoffs' => 0];
            }
            $dailyBookings[$pickupDate]['pickups'] += $bQty;
            
            $dropoffDate = $bEnd->copy()->addDay()->format('Y-m-d');
            if (!isset($dailyBookings[$dropoffDate])) {
                $dailyBookings[$dropoffDate] = ['pickups' => 0, 'dropoffs' => 0];
            }
            $dailyBookings[$dropoffDate]['dropoffs'] += $bQty;
            
            $bookingById[$b->id] = $b;
        }
    
        $removedQty = intval($item->quantity ?? 1);
        
        $currentStock = $totalQty - $removedQty; // Permanently remove damaged item
        $shortageDays = [];
        $affectedBookingIds = [];
    
        // Debug the new approach
       
    
        foreach ($period as $date) {
            $stockBefore = $currentStock;
            
            if (isset($dailyBookings[$date])) {
                $currentStock -= $dailyBookings[$date]['pickups'];
                $currentStock += $dailyBookings[$date]['dropoffs'];
            }
            
            $pickupsToday = $dailyBookings[$date]['pickups'] ?? 0;
            
            // Check for shortages
            if ($pickupsToday > 0 && $pickupsToday > $stockBefore) {
                $shortageDays[] = $date;
                
                foreach ($bookingById as $id => $bItem) {
                    $bStart = Carbon::parse($bItem->pickup_date)->startOfDay();
                    $bStartDate = $bStart->format('Y-m-d');
                    
                    if ($date === $bStartDate) {
                        $affectedBookingIds[$id] = $bItem;
                    }
                }
            }
        }
    
        $affectedBookingsArr = [];
        foreach ($affectedBookingIds as $id => $bItem) {
            $affectedBookingsArr[] = [
                'booking_item_id' => $bItem->id,
                'booking_id'      => $bItem->booking_id ?? null,
                'pickup_date'     => $bItem->pickup_date,
                'dropoff_date'    => $bItem->new_dropoff_date ?? $bItem->dropoff_date,
                'quantity'        => $bItem->quantity ?? 1,
                'pickup_time'     => $bItem->pickup_time,
                'dropoff_time' => $bItem->dropoff_time,
            ];
        }
        $msg = '';
        if (!empty($affectedBookingsArr)) {
            $msg = "Marking this booking item (ID {$item->id}) as damaged removes {$removedQty} unit(s) from stock ({$totalQty} → " . ($totalQty - $removedQty) . "), causing shortage on dates: " .
            implode(', ', array_slice(array_map(fn($d) => \Carbon\Carbon::parse($d)->format('d.m.y'), $shortageDays), 0, 10));
                }
    
        return [
            'total_quantity'     => $totalQty,
            'removed_quantity'   => $removedQty,
            'remaining_quantity' => $totalQty - $removedQty,
            'shortage_days'      => $shortageDays,
            'affected_bookings'  => $affectedBookingsArr,
            'message'            => $msg,
        ];
    }


//     public function updateBookingDetail(Request $request, $id)
// {
//     $validated = $request->validate([
//         'title' => 'nullable|string|max:10',
//         'first_name' => 'required|string|max:255',
//         'last_name' => 'required|string|max:255',
//         'address' => 'required|string|max:255',
//         'zip' => 'required|string|max:20',
//         'city' => 'required|string|max:255',
//         'country' => 'required|string|max:255',
//         'email' => 'required|email|max:255',
//         'phone_code' => 'required|string|max:5',
//         'phone' => 'required|string|max:20',
//         'note' => 'nullable|string',
//     ]);

//     $booking = Booking::findOrFail($id);
    
//     // Update or create user details
//     $booking->userDetails()->updateOrCreate(
//         ['booking_id' => $booking->id],
//         [
//             'first_name' => $validated['first_name'],
//             'last_name' => $validated['last_name'],
//             'email' => $validated['email'],
//             'phone' => $validated['phone_code'] . $validated['phone'],
//             'address' => $validated['address'],
//             'zip' => $validated['zip'],
//             'city' => $validated['city'],
//             'region' => $validated['country'],
//         ]
//     );

//     // You could also update other booking-related information here

//     return redirect()->route('admin.booking.index')
//         ->with('success', 'Booking details updated successfully');
// }
public function updateBookingDetail(Request $request, $id)
{

    $validated = $request->validate([
        'title' => 'nullable|string|max:10',
        'first_name' => 'required|string|max:255',
        'last_name' => 'required|string|max:255',
        'address' => 'required|string|max:255',
        'zip' => 'required|string|max:20',
        'city' => 'required|string|max:255',
        'country' => 'required|string|max:255',
        'email' => 'required|email|max:255',
        'phone_code' => 'required|string|max:5',
        'phone' => 'required|string|max:20',
        'notes' => 'nullable|string',
        'items' => 'required|array',
        'items.*.pickup_date' => 'nullable|date',
        'items.*.pickup_time' => 'nullable|string',
        'items.*.dropoff_date' => 'nullable|date|after_or_equal:items.*.pickup_date',
        'items.*.dropoff_time' => 'nullable|string',
        'items.*.change_reason' => 'nullable|string',
    ]);

    DB::beginTransaction();

    try {
        $booking = Booking::findOrFail($id);
        
        // Update or create user details
        $booking->userDetails()->updateOrCreate(
            ['booking_id' => $booking->id],
            [
                'first_name' =>  $validated['first_name'],
                'last_name' => $validated['last_name'],
                'email' => $validated['email'],
                'phone' => $validated['phone_code'] . $validated['phone'],
                'address' => $validated['address'],
                'zip' => $validated['zip'],
                'city' => $validated['city'],
                'country' => $validated['country'],
            ]
        );

        // Update booking items
        foreach ($validated['items'] as $itemId => $itemData) {
            $bookingItem = BookingItem::findOrFail($itemId);
            
            $updateData = [];
            
            // Only update fields that were provided
            if (!empty($itemData['pickup_date'])) {
                $updateData['pickup_date'] = $itemData['pickup_date'];
            }
            if (!empty($itemData['pickup_time'])) {
                $updateData['pickup_time'] = $itemData['pickup_time'];
            }
            if (!empty($itemData['dropoff_date'])) {
                $updateData['dropoff_date'] = $itemData['dropoff_date'];
            }
            if (!empty($itemData['dropoff_time'])) {
                $updateData['dropoff_time'] = $itemData['dropoff_time'];
            }
            if (isset($itemData['change_reason'])) {
                $updateData['change_reason'] = $itemData['change_reason'];
            }
            
            if (!empty($updateData)) {
                $bookingItem->update($updateData);
            }
        }

        // Update booking note
        $booking->update([
            'notes' => $validated['notes'] ?? null
        ]);


        DB::commit();

        return redirect()->route('admin.booking.index')
            ->with('success', 'Booking details updated successfully');

    } catch (\Exception $e) {
        DB::rollBack();
        return back()->withInput()->with('error', 'Failed to update booking: ' . $e->getMessage());
    }
}
    public function searchBookingUser(Request $request){
        
        $validated = $request->validate([
            'search' => 'nullable|string|max:255',
        ]);
        $searchQuery = $request->input('search', '');
            $bookings = Booking::with('userDetails');
            if ($searchQuery) {
                $bookings = $bookings->whereHas('userDetails', function ($query) use ($searchQuery) {
                    $query->where('first_name', 'like', "%{$searchQuery}%")
                          ->orWhere('last_name', 'like', "%{$searchQuery}%")
                          ->orWhere('email', 'like', "%{$searchQuery}%")
                          ->orWhere('address', 'like', "%{$searchQuery}%");
                });
            }
        $bookings = $bookings->orderBy('created_at', 'desc')->get(); 
        $count = $bookings->count();
        return response()->json([
            'html' => view('admin.booking.search_booking_user', compact('bookings', 'count'))->render()
     ]);
    
}
public function updateStatus(Request $request, Booking $booking)
{
    $validStatuses = ['pending', 'cancelled', 'confirmed', 'completed'];
    
    $request->validate([
        'status' => ['required', 'string', Rule::in($validStatuses)]
    ]);
    
    $previousStatus = $booking->status;

    if ($request->status == 'confirmed' && $previousStatus == 'cancelled') {
    
        foreach ($booking->items as $item) {
            $pickupDate = date('Y-m-d', strtotime($item->pickup_date));
            $dropoffDate = date('Y-m-d', strtotime($item->dropoff_date));
            $pickupDateTime = $pickupDate . ' ' . $item->pickup_time;
            $dropoffDateTime = $dropoffDate . ' ' . $item->dropoff_time;
    
            $newDropoffDateTime = null;
            if ($item->new_dropoff_date && $item->new_dropoff_time) {
                $newDropoffDate = date('Y-m-d', strtotime($item->new_dropoff_date));
                $newDropoffDateTime = $newDropoffDate . ' ' . $item->new_dropoff_time;
            }
    

            $timesToCheck = array_filter([$pickupDateTime, $dropoffDateTime, $newDropoffDateTime]);

            $conflict = \App\Models\BookingItem::whereHas('booking', function ($q) {
                $q->where('status', 'confirmed');
                if (!empty($booking->vendor_id)) {
                    $q->where('vendor_id', $booking->vendor_id);
                }
            })

           
            ->where('id', '!=', $item->id)
            ->where(function ($q) use ($timesToCheck) {
                foreach ($timesToCheck as $datetime) {
                    $date = date('Y-m-d', strtotime($datetime));
                    $time = date('H:i:s', strtotime($datetime));

                    $q->orWhere(function ($sub) use ($date, $time) {
                        $sub->where(function ($x) use ($date, $time) {
                            $x->where('pickup_date', $date)->where('pickup_time', $time);
                        })
                        ->orWhere(function ($x) use ($date, $time) {
                            $x->where('dropoff_date', $date)->where('dropoff_time', $time);
                        })
                        ->orWhere(function ($x) use ($date, $time) {
                            $x->where('new_dropoff_date', $date)->where('new_dropoff_time', $time);
                        });
                    });
                }
            })
            ->first();
            if ($conflict) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot confirm this booking — another confirmed booking (ID: ' . $conflict->booking_id . ') already uses one or more of these date/time slots.'
                ], 409);
            }
        }
    }
    





    $booking->update(['status' => $request->status]);

    if($request->status == 'confirmed'){
        $booking->update(['payment_status' => 'paid'] );
    }else if($request->status == 'cancelled'){
        $booking->update(['payment_status'=> 'unpaid'] );
    }else if($request->status == 'completed'){
        $booking->update(['payment_status' => 'paid' ] );
    }else if($request->status == 'pending'){
        $booking->update(['payment_status' => 'reconciliation'] );
    }


    
    // Send cancellation email if status changed to cancelled
    if ($request->status === 'cancelled' && $previousStatus !== 'cancelled') {
        $bookingItems = BookingItem::where('booking_id', $booking->id)->get();
        $userDetails = BookingUserDetail::where('booking_id', $booking->id)->first();

        
        $vendor = null;
        $vendorId = $bookingItems->pluck('vendor_id')->filter()->first();
        if ($vendorId) {
            $vendor = \App\Models\Vendor::find($vendorId);
        }


        
        Mail::to($userDetails->email)->send(new BookingCancellationMail(
            $booking,
            $userDetails,
            $bookingItems,
            $vendor
        ));

        Mail::to('info@dachboxit.de')->send(new BookingCancellationMail(
            $booking,
            $userDetails,
            $bookingItems,
            $vendor
        ));

        if ($vendor) {
            Mail::to($vendor->email)->send(new BookingCancellationMail(
                $booking,
                $userDetails,
                $bookingItems,
                $vendor
            ));
        }
        
        // Optional: Log the cancellation
    }


   

    if ($request->status === 'completed' && $previousStatus !== 'completed') {
        $bookingItems = BookingItem::where('booking_id', $booking->id)->get();
        $userDetails = BookingUserDetail::where('booking_id', $booking->id)->first();
        


        Mail::send('email.booking-completed', [
            'userDetails' => $userDetails,
        ], function ($message) use ($request, $userDetails) {
            $message->to($userDetails->email)
                    ->subject('Thank You for the booking.');
        });

        Mail::send('email.booking-completed', [
            'userDetails' => $userDetails,
        ], function ($message) use ($request, $userDetails) {
            $message->to('info@dachboxit.de')
            ->subject('Thank You for the booking.');
        });
        
        // Optional: Log the cancellation
    }

    
    
    return response()->json([
        'success' => true,
        'message' => 'Booking status updated successfully'
    ]);
}

// In your AdminController or BookingController

public function addBookingNew()
{
    $manufacturers = MotherFile::distinct()->pluck('car_make');
    $products = Product::with('brand', 'category')->get();
    $roofTypes = [
        'Normal Roof',
        'Normal roof without glass',
        'Normal Roof with glass',
        'Fixpoints',
        'Fixpoints without glass',
        'Fixpoints with glass',
        'Integrated Roof Railing',
        'Elevated Roof Rails',
        'T-Profile',
        'Raingutter'
    ];
    return view($this->filePath.'.add_booking', compact('manufacturers', 'products','roofTypes'));
}

public function getProductDetails2($slug)
{
    $categoryId = Category::where('slug',$slug)->pluck('id');
    $product = Product::with([
        'category',
        'brand',
        'variations.options.color',
        'variations',
        'images',
        'models',
        'otherInfo',
    ])->where('category_id', $categoryId)->first();

    $priceRecord = ProductPrice::where('product_name', $product->name)->first();
    $roof_rackPrice = ProductPrice::where('product_name', 'Dachträger ergänzt zu Dachbox/ Dachfahrradträger')->first();

    if ($priceRecord && $priceRecord->price) {
        $prices = $priceRecord->price;
        $product->first_price = $prices[0] ?? null;
        $product->dailyPrices = array_map(function ($price) {
            return floatval(str_replace(['€', ',', ' '], ['.', '.', ''], $price));
        }, $prices);

        $rawRoofRack = ($roof_rackPrice->price);
        $product->roofRackPrices = array_map(function ($price) {
            return floatval(str_replace(['€', ',', ' '], ['.', '.', ''], $price));
        }, $rawRoofRack);
    } else {
        $product->first_price = null;
        $product->dailyPrices = [];
        $product->roofRackPrices = [];
    }

    $category = $product->category;

    $sizeIds = is_array($category->size_ids) ? $category->size_ids : json_decode($category->size_ids, true);
    $variationIds = is_array($category->variation_ids) ? $category->variation_ids : json_decode($category->variation_ids, true);
    $colorIds = is_array($category->color_ids) ? $category->color_ids : json_decode($category->color_ids, true);

    $product->sizes = $sizeIds ? Size::whereIn('id', $sizeIds)->get() : collect();
    $product->variations_product = $variationIds ? Variation::whereIn('id', $variationIds)->get() : collect();
    $product->colors = $colorIds ? Color::whereIn('id', $colorIds)->get() : collect();
    // dd($product);
    return response()->json($product);
}

// public function storeBooking(Request $request)
// {
//     $validated = $request->validate([
//         'first_name' => 'required|string|max:255',
//         'last_name' => 'required|string|max:255',
//         'address' => 'required|string|max:255',
//         'zip_code' => 'required|string|max:20',
//         'city' => 'required|string|max:255',
//         'country' => 'required|string|max:255',
//         'email' => 'required|email|max:255',
//         'phone' => 'required|string|max:20',
//         'product_id' => 'required|exists:products,id',
//         'manufacturer' => 'required|string|max:255',
//         'car_year' => 'required|string|max:4',
//         'model' => 'required|string|max:255',
//         'trailer_hitch' => 'required|string|in:yes,no',
//         'color_id' => 'required|exists:colors,id',
//         'size_id' => 'required',
//         'rental_from' => 'required|date',
//         'rental_to' => 'required|date|after_or_equal:rental_from',
//         'days' => 'required|integer|min:1',
//         'calculated_rent' => 'required|numeric|min:0',
//         'is_carrier_included' => 'boolean',
//         'is_insured' => 'boolean',
//         'discount' => 'numeric|min:0',
//     ]);

//     try {
//         // Create customer record if not exists
//         $user = User::firstOrCreate(
//             ['email' => $request->email],
//             [
//                 'title' => $request->title,
//                 'first_name' => $request->first_name,
//                 'last_name' => $request->last_name,
//                 'address' => $request->address,
//                 'zip_code' => $request->zip_code,
//                 'city' => $request->city,
//                 'region' => $request->country,
//                 'phone' => $request->phone,
//             ]
//         );

//         // Create booking
//         $booking = new Booking();
//         // $booking->customer_id = $customer->id;
//         $booking->product_id = $request->product_id;
//         $booking->product_name = $request->product_name;
//         $booking->manufacturer = $request->manufacturer;
//         $booking->car_year = $request->car_year;
//         $booking->model = $request->model;
//         $booking->trailer_hitch = $request->trailer_hitch;
//         $booking->roof_type = $request->roof_type;
//         $booking->color_id = $request->color_id;
//         $booking->size_id = $request->size_id;
//         $booking->rental_from = $request->rental_from;
//         $booking->rental_to = $request->rental_to;
//         $booking->days = $request->days;
//         $booking->calculated_rent = $request->calculated_rent;
//         $booking->is_carrier_included = $request->is_carrier_included ?? 0;
//         $booking->is_insured = $request->is_insured ?? 0;
//         $booking->discount = $request->discount ?? 0;
//         $booking->status = 'confirmed';
//         $booking->save();
     
//         return response()->json([
//             'success' => true,
//             'message' => 'Booking created successfully',
//             'booking_id' => $booking->id
//         ]);

//     } catch (\Exception $e) {
//         return response()->json([
//             'success' => false,
//             'message' => 'Error creating booking: ' . $e->getMessage()
//         ], 500);
//     }
// }


public function saveBooking(Request $request)
{
    // Validate the request data
    $validated = $request->validate([
        'title' => 'required|string',
        'first_name' => 'required|string',
        'last_name' => 'required|string',
        'address' => 'required|string',
        'zip_code' => 'required|string',
        'city' => 'required|string',
        'country' => 'required|string',
        'email' => 'required|email',
        'phone' => 'required|string',
        
    ]);

    // Calculate pricing
    $taxRate = 0.19;
    $taxableAmount = $request->calculated_rent;
    
    if ($request->is_insured) {
        $taxableAmount -= 25; // Remove insurance from taxable amount
    }
    
    $netPrice = $taxableAmount / (1 + $taxRate);
    $taxAmount = $netPrice * $taxRate;
    $subtotal = $netPrice + $taxAmount;
    $insuranceTotal = $request->is_insured ? 25 : 0;
    $total = ($subtotal - ($request->discount ?? 0)) + $insuranceTotal;
    $randomPassword = Str::random(10);
            
    $user = User::where('email', $request->email)->first();
    if ($user) {
        // If user exists, return a response indicating the email is already in use
        return response()->json([
            'success' => false,
            'message' => 'Email is already exist.'
        ]); // 400 Bad Request
    }


if (!$user) {
    $randomPassword = Str::random(10);

    $user = User::create([
        'first_name' => $request->first_name,
        'last_name' => $request->last_name,
        'email' => $request->email,
        'zip_code' => $request->zip_code,
        'city' => $request->city,
        'region' => $request->country,
        'street' => $request->address,
        'password' => Hash::make($randomPassword),
        'status' => 1,
    ]);

    // Mail::raw("Your account has been created with email {$request->email}. Your password is: {$randomPassword}", function ($message) use ($request) {
    //     $message->to($request->email)
    //             ->subject('Your Customer Account Details');
    // });
    Mail::to($request->email)->send(new GuestAccountMail(
        $request->email,
        $randomPassword
    ));
}
    // Create the booking with pending payment status
    
    $booking = Booking::create([
        'user_id' => $user->id,
        'status' => 'pending',
        'subtotal' => $subtotal,
        'discount' => $request->discount ?? 0,
        'coupon_code' => json_encode([$request->coupon_code]) ?? null,
        'tax' => $taxAmount,
        'total' => $total,
        'payment_method' => 'N/A',
        'payment_status' => 'unpaid',
        'admin_created' => true,
    ]);

    // Save customer details
    BookingUserDetail::create([
        'booking_id' => $booking->id,
        'user_id' => $user->id,
        'title' => $request->title,
        'first_name' => $request->first_name,
        'last_name' => $request->last_name,
        'email' => $request->email,
        'phone' => $request->phone,
        'address' => $request->address,
        'zip' => $request->zip_code,
        'city' => $request->city,
        'country' => $request->country,
    ]);

    // Save booking items
    BookingItem::create([
        'booking_id' => $booking->id,
        'user_id' => null,
        'product_id' => $request->product_id,
        'product_name' => $request->product_name,
        'car_name' => $request->manufacturer,
        'car_year' => $request->car_year,
        'model' => $request->model,
        'trailer_hitch' => $request->trailer_hitch,
        'roof_type' => $request->roof_type,
        'color' => $request->color_name,
        'size' => $request->size_name,
        'is_carrier_included' => $request->is_carrier_included,
        'is_insured' => $request->is_insured,
        'pickup_date' => $request->rental_from,
        'dropoff_date' => $request->rental_to,
        'days' => $request->days,
        'calculated_rent' => $request->calculated_rent,
        'status' => 'pending',
    ]);

    // Initialize Stripe
    Stripe::setApiKey(env('STRIPE_SECRET_KEY'));

    // Create Stripe Checkout Session
    $session = Session::create([
        'payment_method_types' => ['card', 'paypal', 'klarna', 'sepa_debit'],
        'line_items' => [[
            'price_data' => [
                'currency' => 'eur',
                'product_data' => [
                    'name' => $request->product_name,
                ],
                'unit_amount' => (int)($total * 100),
            ],
            'quantity' => 1,
        ]],
        'mode' => 'payment',
        'customer_email' => $request->email,
        'metadata' => [
            'booking_id' => $booking->id,
        ],
        'success_url' => route('checkout.success').'?session_id={CHECKOUT_SESSION_ID}',
        'cancel_url' => route('checkout.cancel').'?session_id={CHECKOUT_SESSION_ID}',
    ]);

    // Update booking with Stripe session ID
    $booking->update([
        'stripe_session_id' => $session->id,
        'last_payment_link_sent_at' => now(),
    ]);

    // Send payment link email
    $paymentLink = $session->url;

    $bookingItems = BookingItem::where('booking_id', $booking->id)->get();
    $userDetails = BookingUserDetail::where('booking_id', $booking->id)->first();
    $admin = Admin::first();

    $vendorId = $bookingItems->pluck('vendor_id')->filter()->first();
    $vendor = $vendorId ? \App\Models\Vendor::find($vendorId) : null;

    $pickupInfo = $vendor
    ? [
        'name' => $vendor->name,
        'street' => $vendor->street,
        'zip_code' => $vendor->zip_code,
        'city' => $vendor->city,
    ]
    : [
        'name' => $admin->name,
        'street' => $admin->address,
        'zip_code' => $admin->zip_code,
        'city' => $admin->city,
    ];

    $emailData = [
        'booking' => $booking,
        'customer' => [
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
        ],
        'product_name' => $request->product_name,
        'rental_period' => $request->rental_from . ' to ' . $request->rental_to,
        'total_amount' => number_format($total, 2, ',', '.'),
        'payment_link' => $paymentLink,
        'bookingItems' => $bookingItems,
        'userDetails' => $userDetails,
        'pickupInfo' => $pickupInfo,
        'admin' => $admin
    ];

    Mail::send('email.admin-booking-payment-link', $emailData, function ($message) use ($request, $booking) {
        $message->to($request->email)
                ->subject('Zahlungslink für Ihre Buchung bei Dachboxmiete – 24 Stunden gültig');
    });
    
    Mail::send('email.admin-booking-payment-link', $emailData, function ($message) use ($booking) {
        $message->to('info@dachboxit.de')
                ->subject('Zahlungslink für Ihre Buchung bei Dachboxmiete – 24 Stunden gültig');
    });
    
    if ($vendor) {
        Mail::send('email.admin-booking-payment-link', $emailData, function ($message) use ($vendor, $booking) {
            $message->to($vendor->email)
            ->subject('Zahlungslink für Ihre Buchung bei Dachboxmiete – 24 Stunden gültig');
        });
    }
    
    // Mail::send('email.admin-booking-payment-link', [
    //     'booking' => $booking,
    //     'customer' => [
    //         'first_name' => $request->first_name,
    //         'last_name' => $request->last_name,
    //     ],
    //     'product_name' => $request->product_name,
    //     'rental_period' => $request->rental_from . ' to ' . $request->rental_to,
    //     'total_amount' => number_format($total, 2, ',', '.'),
    //     'payment_link' => $paymentLink,
    //     'bookingItems' => $bookingItems,
    //     'userDetails' => $userDetails,
    //     'admin' => $admin
    // ], function ($message) use ($request, $booking) {
    //     $message->to($request->email)
    //             ->subject('Zahlungslink für Ihre Buchung bei Dachboxmiete – 24 Stunden gültig');
    // });

    

    // Mail::send('email.admin-booking-payment-link', [
    //     'booking' => $booking,
    //     'customer' => [
    //         'first_name' => $request->first_name,
    //         'last_name' => $request->last_name,
    //     ],
    //     'product_name' => $request->product_name,
    //     'rental_period' => $request->rental_from . ' to ' . $request->rental_to,
    //     'total_amount' => number_format($total, 2, ',', '.'),
    //     'payment_link' => $paymentLink,
    //     'bookingItems' => $bookingItems,
    //     'userDetails' => $userDetails,
    //     'admin' => $admin
    // ], function ($message) use ($request, $booking) {
    //     $message->to('info@dachboxit.de')
    //             ->subject('Zahlungslink für Ihre Buchung bei Dachboxmiete – 24 Stunden gültig');
    // });

    return response()->json([
        'success' => true,
        'message' => 'Booking created successfully. Payment link has been sent to the customer.',
        'booking_id' => $booking->id,
        'payment_link' => $paymentLink 
    ]);
}
public function resendPaymentLink(Request $request, $id)
{
    $booking = Booking::findOrFail($id);
    $user = $booking->userDetails;

    // Initialize Stripe
    Stripe::setApiKey(env('STRIPE_SECRET_KEY'));

    // Create new Stripe Checkout Session
    try {
        $session = Session::create([
            'payment_method_types' => ['card', 'paypal', 'klarna', 'sepa_debit'],
            'line_items' => [[
                'price_data' => [
                    'currency' => 'eur',
                    'product_data' => [
                        'name' => $booking->items->first()->product_name,
                    ],
                    'unit_amount' => (int)($booking->total * 100),
                ],
                'quantity' => 1,
            ]],
            'mode' => 'payment',
            'customer_email' => $user->email,
            'metadata' => [
                'booking_id' => $booking->id,
            ],
            'success_url' => route('checkout.success').'?session_id={CHECKOUT_SESSION_ID}',
            'cancel_url' => route('checkout.cancel').'?session_id={CHECKOUT_SESSION_ID}',
        ]);

        $bookingItems = BookingItem::where('booking_id', $booking->id)->get();
        $userDetails = BookingUserDetail::where('booking_id', $booking->id)->first();
        $admin = Admin::first();
        // Update booking with new session info
        $booking->update([
            'stripe_session_id' => $session->id,
            'last_payment_link_sent_at' => now(),
        ]);
        $paymentLink = $session->url;
        // Send email to customer with the payment link

        $vendorId = $bookingItems->pluck('vendor_id')->filter()->first();
        $vendor = $vendorId ? \App\Models\Vendor::find($vendorId) : null;
    
        $pickupInfo = $vendor
        ? [
            'name' => $vendor->name,
            'street' => $vendor->street,
            'zip_code' => $vendor->zip_code,
            'city' => $vendor->city,
             'state' => $vendor->country 
        ]
        : [
            'name' => $admin->name,
            'street' => $admin->address,
            'zip_code' => $admin->zip_code,
            'city' => $admin->city,
            'state' => $admin->state
        ];
    
        $emailData = [
            'booking' => $booking,
            'customer' => [
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
            ],
            'product_name' => $request->product_name,
            'rental_period' => $request->rental_from . ' to ' . $request->rental_to,
            'total_amount' => number_format($total, 2, ',', '.'),
            'payment_link' => $paymentLink,
            'bookingItems' => $bookingItems,
            'userDetails' => $userDetails,
            'pickupInfo' => $pickupInfo,
            'admin' => $admin
        ];
    
        Mail::send('email.admin-booking-payment-link', $emailData, function ($message) use ($request, $booking) {
            $message->to($request->email)
                    ->subject('Zahlungslink für Ihre Buchung bei Dachboxmiete – 24 Stunden gültig');
        });
        
        Mail::send('email.admin-booking-payment-link', $emailData, function ($message) use ($booking) {
            $message->to('info@dachboxit.de')
                    ->subject('Zahlungslink für Ihre Buchung bei Dachboxmiete – 24 Stunden gültig');
        });
        
        if ($vendor) {
            Mail::send('email.admin-booking-payment-link', $emailData, function ($message) use ($vendor, $booking) {
                $message->to($vendor->email)
                ->subject('Zahlungslink für Ihre Buchung bei Dachboxmiete – 24 Stunden gültig');
            });
        }
        

        // Mail::send('email.admin-booking-payment-link', [
        //     'booking' => $booking,
        //     'customer' => [
        //         'first_name' => $request->first_name,
        //         'last_name' => $request->last_name,
        //     ],
        //     'product_name' => $request->product_name,
        //     'rental_period' => $request->rental_from . ' to ' . $request->rental_to,
        //     'total_amount' => number_format($booking->total, 2),
        //     'payment_link' => $paymentLink,
        //     'bookingItems' => $bookingItems,
        //     'userDetails' => $userDetails,
        //     'admin' => $admin,
        // ], function ($message) use ($request, $booking, $user) {
        //     $message->to($user->email)
        //             ->subject('Payment Link for Your Booking #' . $booking->id);
        // });

        // Mail::send('email.admin-booking-payment-link', [
        //     'booking' => $booking,
        //     'customer' => [
        //         'first_name' => $request->first_name,
        //         'last_name' => $request->last_name,
        //     ],
        //     'product_name' => $request->product_name,
        //     'rental_period' => $request->rental_from . ' to ' . $request->rental_to,
        //     'total_amount' => number_format($booking->total, 2),
        //     'payment_link' => $paymentLink,
        //     'bookingItems' => $bookingItems,
        //     'userDetails' => $userDetails,
        //     'admin' => $admin,
        // ], function ($message) use ($request, $booking, $user) {
        //     $message->to('info@dachboxit.de')
        //             ->subject('Payment Link for Your Booking #' . $booking->id);
        // });

        

        return back()->with('success', 'Payment link has been resent successfully.');

    } catch (\Exception $e) {
        return back()->with('error', 'Failed to resend payment link: ' . $e->getMessage());
    }
}
public function applyCoupon2(Request $request)
{
    $couponCode = strtolower($request->input('coupon_code'));
    $productId = $request->input('product_id');
    $subtotal = $request->input('subtotal');


    $voucher = Voucher::where('code', $couponCode)->first();

    // Basic coupon validation
    if (!$voucher) {
        return response()->json([
            'success' => false,
            'message' => 'Invalid coupon code'
        ]);
    }

    if (!$voucher->is_active) {
        return response()->json([
            'success' => false,
            'message' => 'This coupon is not currently active'
        ]);
    }

    $now = Carbon::now();
    if ($now->lt($voucher->start_date)) {
        return response()->json([
            'success' => false,
            'message' => 'This coupon is not valid yet'
        ]);
    }

    if ($now->gt($voucher->expiration_date)) {
        return response()->json([
            'success' => false,
            'message' => 'This coupon has expired'
        ]);
    }

    // Check if coupon applies to this product
    $applicableProductIds = json_decode($voucher->applied_to, true) ?? [];
    if (!empty($applicableProductIds)) {
        if (!in_array($productId, $applicableProductIds)) {
            return response()->json([
                'success' => false,
                'message' => 'This coupon cannot be applied to the selected product'
            ]);
        }
    }


    $totalDiscountUsed = DB::table('bookings')
    ->whereRaw("JSON_UNQUOTE(JSON_EXTRACT(coupon_code, '$[0]')) = ?", [strtoupper($voucher->code)])
    ->sum('discount');

if ($totalDiscountUsed >= $voucher->max_discount) {
    return response()->json([
        'success' => false,
        'message' => 'This coupon has reached its maximum discount limit'
    ]);
}
    // Check minimum order value
    if ($subtotal < $voucher->min_order_value) {
        return response()->json([
            'success' => false,
            'message' => 'Minimum order value for this coupon is €' . $voucher->min_order_value .
                ' (Your order total: €' . $subtotal . ')'
        ]);
    }

    // Check redemption limit
    // if ($voucher->redemption_limit > 0 && $voucher->times_redeemed >= $voucher->redemption_limit) {
    //     return response()->json([
    //         'success' => false,
    //         'message' => 'This coupon has reached its redemption limit'
    //     ]);
    // }

    // Calculate discount
    if ($voucher->type == 1) {
        // Fixed amount discount
        $discount = $voucher->voucher_value;
    } else {
        // Percentage discount
        $percentageDiscount = $subtotal * ($voucher->voucher_value / 100);
        $discount = $percentageDiscount;
    }

    return response()->json([
        'success' => true,
        'message' => 'Coupon applied successfully!',
        'discount' => $discount,
        'new_total' => $subtotal - $discount
    ]);
}

public function getCouponsForProduct(Request $request)
{
    $productId = $request->input('product_id');
    
    // Get all active coupons that apply to this product or all products
    $coupons = Voucher::where('start_date', '<=', now())
        ->where('expiration_date', '>=', now())
        ->where(function($query) use ($productId) {
            $query->whereNull('applied_to') // Applies to all products
                  ->orWhereJsonContains('applied_to', $productId); // Applies to this specific product
        })
        ->get()
        ->map(function($coupon) {
            return [
                'code' => $coupon->code,
                'type' => $coupon->type,
                'max_discount' => $coupon->max_discount,
                'max_discount_amount' => $coupon->max_discount_amount,
                'min_order_value' => $coupon->min_order_value,
                'description' => $coupon->description
            ];
        });
        // dd($coupons);
    
    return response()->json([
        'success' => true,
        'coupons' => $coupons
    ]);
}

public function getTime(Request $request)
{
    $carbonDate = \Carbon\Carbon::parse($request->date);
    $container = $request->container;
    $date = $carbonDate->toDateString();
    $day = $carbonDate->format('l');

    $slotDuration = 15;
    $slotGap = 15;
    $slots = [];

    $from = '09:00';
    $to = '17:00';


    if ($request->has('vendor_id') && !empty($request->vendor_id)) {
        $vendorId = $request->vendor_id;
        $carbonDay = strtolower($carbonDate->format('l')); // monday, tuesday, ...
        $dayMap = [
            'monday'    => 'mo',
            'tuesday'   => 'di',
            'wednesday' => 'mi',
            'thursday'  => 'do',
            'friday'    => 'fr',
            'saturday'  => 'sa',
            'sunday'    => 'so',
        ];

        $day = $dayMap[$carbonDay] ?? null;
        $workingHour = \App\Models\VendorWorkingHour::where('vendor_id', $vendorId)
        ->where('day', $day)
        ->first();

        if ($workingHour && !$workingHour->is_closed) {
            $from = $workingHour->open_time;
            $to = $workingHour->close_time;
        }
        $from = substr($from, 0, 5); 
        $to   = substr($to, 0, 5);

        $start = \Carbon\Carbon::createFromFormat('H:i', $from);
        $end = \Carbon\Carbon::createFromFormat('H:i', $to);


        $unavailabilities = \App\Models\DateAvailability::where('admin_id', $vendorId)
        ->whereDate('date', $date)
        ->get();

        $unavailableSlots = collect();

        foreach ($unavailabilities as $unavailability) {
            $blockStart = \Carbon\Carbon::parse($unavailability->start_time);
            $blockEnd = \Carbon\Carbon::parse($unavailability->end_time);

            $current = $start->copy();
            while ($current->copy()->addMinutes($slotDuration)->lte($end)) {
                $slotStart = $current->copy();
                $slotEnd = $current->copy()->addMinutes($slotDuration);

                // Check for overlap with each unavailability block
                foreach ($unavailabilities as $unavailability) {
                    $blockStart = \Carbon\Carbon::parse($unavailability->start_time);
                    $blockEnd = \Carbon\Carbon::parse($unavailability->end_time);

                    if ($slotStart->lt($blockEnd) && $slotEnd->gt($blockStart)) {
                        $unavailableSlots->push($slotStart->format('H:i'));
                        break; // no need to check further blocks
                    }
                }

                $current->addMinutes($slotGap);
            }
        }

        $bookedTimes = BookingItem::where('vendor_id', $vendorId)
        ->get()
        ->filter(fn($item) => $item->booking && $item->booking->status === 'confirmed')
        ->flatMap(function ($item) {
            $times = [];
    
            if ($item->pickup_date && $item->pickup_time) {
                $times[] = [
                    'date' => \Carbon\Carbon::parse($item->pickup_date)->toDateString(),
                    'time' => \Carbon\Carbon::parse($item->pickup_time)->format('H:i'),
                ];
            }
    
            if ($item->new_dropoff_date && $item->new_dropoff_time) {
                $times[] = [
                    'date' => \Carbon\Carbon::parse($item->new_dropoff_date)->toDateString(),
                    'time' => \Carbon\Carbon::parse($item->new_dropoff_time)->format('H:i'),
                ];
            }
            elseif ($item->dropoff_date && $item->dropoff_time) {
                $times[] = [
                    'date' => \Carbon\Carbon::parse($item->dropoff_date)->toDateString(),
                    'time' => \Carbon\Carbon::parse($item->dropoff_time)->format('H:i'),
                ];
            }
    
            return $times;
        })
        ->filter(fn($t) => $t['date'] === $date) 
        ->pluck('time')
        ->unique()
        ->values();

        $current = $start->copy();
        while ($current->copy()->addMinutes($slotDuration)->lte($end)) {
            $slotStart = $current->format('H:i');
            $slotEnd = $current->copy()->addMinutes($slotDuration)->format('H:i');
            $slotLabel = "$slotStart - $slotEnd";

            $isUnavailable = $unavailableSlots->contains($slotStart);
            $isBooked = $bookedTimes->contains($slotStart);

            $slots[] = [
                'label' => $slotLabel,
                'disabled' => $isUnavailable || $isBooked,
            ];

            $current->addMinutes($slotGap);
        }




    }else{

   

    $workingHour = WorkingHours::where('day', $day)->first();
    if ($workingHour) {
        $from = $workingHour->open_time;
        $to = $workingHour->close_time;
    }

    $from = \Carbon\Carbon::parse($from)->format('H:i');
    $to = \Carbon\Carbon::parse($to)->format('H:i');

    $start = \Carbon\Carbon::createFromFormat('H:i', $from);
    $end = \Carbon\Carbon::createFromFormat('H:i', $to);

    $unavailabilities = \App\Models\DateAvailability::where('admin_id', 1)
    ->whereDate('date', $date)
    ->get();

    $unavailableSlots = collect();

    foreach ($unavailabilities as $unavailability) {
        $blockStart = \Carbon\Carbon::parse($unavailability->start_time);
        $blockEnd = \Carbon\Carbon::parse($unavailability->end_time);

        $current = $start->copy();
        while ($current->copy()->addMinutes($slotDuration)->lte($end)) {
            $slotStart = $current->copy();
            $slotEnd = $current->copy()->addMinutes($slotDuration);

            foreach ($unavailabilities as $unavailability) {
                $blockStart = \Carbon\Carbon::parse($unavailability->start_time);
                $blockEnd = \Carbon\Carbon::parse($unavailability->end_time);

                if ($slotStart->lt($blockEnd) && $slotEnd->gt($blockStart)) {
                    $unavailableSlots->push($slotStart->format('H:i'));
                    break; 
                }
            }

            $current->addMinutes($slotGap);
        }
    }

    $bookedTimes = BookingItem::whereNull('vendor_id')
    ->get()
    ->filter(fn($item) => $item->booking && $item->booking->status === 'confirmed')
    ->flatMap(function ($item) {
        $times = [];

        // Always consider pickup
        if ($item->pickup_date && $item->pickup_time) {
            $times[] = [
                'date' => \Carbon\Carbon::parse($item->pickup_date)->toDateString(),
                'time' => \Carbon\Carbon::parse($item->pickup_time)->format('H:i'),
            ];
        }

        // If new dropoff exists → consider only that
        if ($item->new_dropoff_date && $item->new_dropoff_time) {
            $times[] = [
                'date' => \Carbon\Carbon::parse($item->new_dropoff_date)->toDateString(),
                'time' => \Carbon\Carbon::parse($item->new_dropoff_time)->format('H:i'),
            ];
        }
        // Else fallback to old dropoff
        elseif ($item->dropoff_date && $item->dropoff_time) {
            $times[] = [
                'date' => \Carbon\Carbon::parse($item->dropoff_date)->toDateString(),
                'time' => \Carbon\Carbon::parse($item->dropoff_time)->format('H:i'),
            ];
        }

        return $times;
    })
    ->filter(fn($t) => $t['date'] === $date) 
    ->pluck('time')
    ->unique()
    ->values();

    $current = $start->copy();
    while ($current->copy()->addMinutes($slotDuration)->lte($end)) {
        $slotStart = $current->format('H:i');
        $slotEnd = $current->copy()->addMinutes($slotDuration)->format('H:i');
        $slotLabel = "$slotStart - $slotEnd";

        $isUnavailable = $unavailableSlots->contains($slotStart);
        $isBooked = $bookedTimes->contains($slotStart);

        $slots[] = [
            'label' => $slotLabel,
            'disabled' => $isUnavailable || $isBooked,
        ];

        $current->addMinutes($slotGap);
    }
}

    return response()->json([
        'from_time' => $from,
        'to_time' => $to,
        'slots' => $slots,
    ]);
}


    function normalizeProductName($name)
    {
        $synonyms = [
            'Roof Bike Rack'     => 'Roof Bike Carrier',
            'Clutch Bike Rack'   => 'Clutch Bike Carrier',
            'Dachfahrradträger' => 'Dachfahrradträger',
            'Kupplungsfahrradträger' => 'Kupplungsfahrradträger'
            // add more pairs if needed
        ];

        // Standardize spacing and casing
        $cleanName = trim(strtolower($name));

        foreach ($synonyms as $a => $b) {
            $a = strtolower($a);
            $b = strtolower($b);
            if ($cleanName === $a || $cleanName === $b) {
                return $a; // or $b, just choose one as canonical
            }
        }

        return $cleanName;
    }

    public function dailyPrice(Request $request) {
        $productName = $request->productName; 
        $normalizedName = $this->normalizeProductName($productName);

        $priceRecord = ProductPrice::get()->first(function ($record) use ($normalizedName) {
            return $this->normalizeProductName($record->product_name) === $normalizedName;
        });
        if ($priceRecord && $priceRecord->price) {
            $prices = $priceRecord->price;

            $rawPrices = ($priceRecord->price);
            $dailyPrices = array_map(function ($price) {
                return floatval(str_replace(['€', ',', ' '], ['.', '.', ''], $price));
            }, $rawPrices);

            

           
        } 

        return $dailyPrices;
    }

    public function returnDetail(Request $request)
{

    $booking = Booking::findOrFail($request->return_booking_id);
if ($request->input('damaged_items') && !empty(array_filter($request->input('damaged_items')))) {
        $selectedIds = $request->damaged_items;
    
        $bookingItems = BookingItem::where('booking_id', $booking->id)->get();
    
        foreach ($bookingItems as $item) {
            $wasDamaged = $item->damage; // old value
            $isDamaged  = in_array($item->id, $selectedIds);
    
            if ($wasDamaged !== $isDamaged) {
                $item->update(['damage' => $isDamaged]);
    
                if ($isDamaged && !$wasDamaged) {
                    $product = Product::find($item->product_id);
                    if (!$product) continue;

                    $name = strtolower($item->product_name);

                    if (in_array($name, ['carrier', 'foot', 'footkit', 'traverse', 'fuß', 'fußkit'])) {
                        $item->color = 'Schwarz';
                    }
    
                    $variationOption = \App\Models\ProductVariationOption::whereHas('variation', function ($q) use ($product, $item) {
                            $q->where('product_id', $product->id)
                              ->where(function ($sub) use ($item) {
                                  $sub->whereHas('size', fn($s) => $s->where('name', $item->size))
                                      ->orWhereHas('variation', fn($v) => $v->where('name', $item->size));
                              });
                        })
                        ->whereHas('color', fn($c) => $c->where('name', $item->color))
                        ->first();
    
                    if ($variationOption) {
                        if (empty($item->vendor_id)) {
                            $variationOption->decrement('quantity', 1);
                        } else {
                            $vendorQty = $variationOption->vendorQuantities()
                                ->where('vendor_id', $item->vendor_id)
                                ->first();
    
                            if ($vendorQty) {
                                $vendorQty->decrement('quantity', 1);
                            } else {
                                $variationOption->vendorQuantities()->create([
                                    'vendor_id' => $item->vendor_id,
                                    'quantity'  => 0,
                                ]);
                            }
                        }
                    }
                }
            }
        }
    
        // Update booking-level flag
        $booking->update(['damage' => true]);
    
    } 
    // else {
    //     // If no damaged items selected, reset all
    //     BookingItem::where('booking_id', $booking->id)->update(['damage' => false]);
    //     $booking->update(['damage' => false]);
    // }

    else {
        // No damaged items selected, reset all & restore if previously damaged
        $bookingItems = BookingItem::where('booking_id', $booking->id)->get();
       
        foreach ($bookingItems as $item) {
            if ($item->damage == 1) {
                // restore +1 if previously damaged
                $product = Product::find($item->product_id);
                if (!$product) continue;
                $name = strtolower($item->product_name);

                            if (in_array($name, ['carrier', 'foot', 'footkit', 'traverse', 'fuß', 'fußkit'])) {
                                $item->color = 'Schwarz';
                            }
    
                $variationOption = \App\Models\ProductVariationOption::whereHas('variation', function ($q) use ($product, $item) {
                        $q->where('product_id', $product->id)
                          ->where(function ($sub) use ($item) {
                              $sub->whereHas('size', fn($s) => $s->where('name', $item->size))
                                  ->orWhereHas('variation', fn($v) => $v->where('name', $item->size));
                          });
                    })
                    ->whereHas('color', fn($c) => $c->where('name', $item->color))
                    ->first();
    
                if ($variationOption) {
                    if (empty($item->vendor_id)) {
                        $variationOption->increment('quantity', 1);
                    } else {
                        $vendorQty = $variationOption->vendorQuantities()
                            ->where('vendor_id', $item->vendor_id)
                            ->first();
                        if ($vendorQty) {
                            $vendorQty->increment('quantity', 1);
                        }
                    }
                }
            }
    
            $item->update(['damage' => false]);
        }
    
        $booking->update(['damage' => false]);
    }

    
        $booking = Booking::findOrFail( $request->return_booking_id);

        foreach ($booking->items as $item) {
            $updateData = [
                'delay' => $request->delay,
                'additional_days' => $request->additional_days,
                'additional_days_rent' => $request->additional_days_rent,
            ];
        
            if ($request->filled('new_dropoff')) {
                $updateData['new_dropoff_date'] = Carbon::createFromFormat('d.m.Y', $request->new_dropoff)->format('Y-m-d');
            }
        
            if ($request->filled('new_dropoff_time')) {
                $updateData['new_dropoff_time'] = Carbon::createFromFormat('H:i', $request->new_dropoff_time)->format('H:i');
            }
        
            $item->update($updateData);
        }

        if($booking->additional_days_rent != $request->additional_days_rent && $booking->delay_fees != $request->delay_fees){

            $reductionSubtotal = $booking->additional_days_rent * 0.81;
            $reductionTax      = ($booking->delay_fees + $booking->additional_days_rent) * 0.19;
            $reductionTotal    = $booking->delay_fees + $booking->additional_days_rent;

            $booking->update([    
                'subtotal' => $booking->subtotal - $reductionSubtotal,
                'tax'      => $booking->tax - $reductionTax,
                'total'    => $booking->total - $reductionTotal,
            ]);


            $booking->update([
                'subtotal' => $booking->subtotal + $request->net_price,
                'tax'      => $booking->tax + $request->tax,
                'total'    => $booking->total + $request->total,
            ]);
        }

        $booking->update(['delay' => $request->delay,
        'delay_fees' => $request->delay_fee , 
        'additional_days' => $request->additional_days , 
        'new_dropoff_date' => $request->new_dropoff 
                                ? Carbon::createFromFormat('d.m.Y', $request->new_dropoff)->format('Y-m-d')
                                : null,


        'damage' => (bool) $request->damaged_items,
        'additional_days_rent' => $request->additional_days_rent,
        ]);


    if($request->total != '0.00' && $request->delay){
        $user = $booking->userDetails;

        Stripe::setApiKey(env('STRIPE_SECRET_KEY'));


        try {
            $session = Session::create([
                'payment_method_types' => [
                    'card',
                    'paypal',
                    
                    'amazon_pay',
                ],
                'line_items' => [[
                    'price_data' => [
                        'currency' => 'eur',
                        'product_data' => [
                            'name' => $booking->items->first()->product_name,
                        ],
                        'unit_amount' => (int)($request->total * 100),
                    ],
                    'quantity' => 1,
                ]],
                'mode' => 'payment',
                'customer_email' => $user->email,
                'metadata' => [
                    'booking_id' => $booking->id,
                ],
                'success_url' => route('delay.success').'?session_id={CHECKOUT_SESSION_ID}',
                'cancel_url' => route('delay.cancel').'?session_id={CHECKOUT_SESSION_ID}',
            ]);

            $bookingItems = BookingItem::where('booking_id', $booking->id)->get();
            $userDetails = BookingUserDetail::where('booking_id', $booking->id)->first();
            $admin = Admin::first();
            $booking->update([
                'delay_session_id' => $session->id,
                'last_delay_payment_link_sent_at' => now(),
            ]);
            $paymentLink = $session->url;

            $vendor_id = $booking->vendor_id;
            if($vendor_id){
                $vendor = Vendor::where('id',$vendor_id)->first();
            } else{
                $vendor = '';
            }
            
             Mail::send('email.delay-payment-link', [
                'booking' => $booking,
                'customer' => [
                    'first_name' => $user->first_name,
                    'last_name' => $user->last_name,
                ],
                'product_name' => $booking->items->first()->product_name,
                'rental_period' => $booking->pickup_date . ' to ' . $booking->new_dropoff_date,
                'total_amount' => number_format($request->total, 2),
                'payment_link' => $paymentLink,
                'bookingItems' => $bookingItems,
                'userDetails' => $userDetails,
                'admin' => $admin,
                'vendor'=>$vendor
            ], function ($message) use ($request, $booking, $user) {
                $message->to($user->email)
                        ->subject('Payment Link for Your Booking #' . $booking->id);
            });
            
            Mail::send('email.delay-payment-link', [
                'booking' => $booking,
                'customer' => [
                    'first_name' => $user->first_name,
                    'last_name' => $user->last_name,
                ],
                'product_name' => $booking->items->first()->product_name,
                'rental_period' => $booking->pickup_date . ' to ' . $booking->new_dropoff_date,
                'total_amount' => number_format($request->total, 2),
                'payment_link' => $paymentLink,
                'bookingItems' => $bookingItems,
                'userDetails' => $userDetails,
                'admin' => $admin,
                'vendor'=>$vendor
            ], function ($message) use ($request, $booking, $user) {
                $message->to('info@dachboxit.de')
                        ->subject('Payment Link for Your Booking #' . $booking->id);
            });

            if($vendor){
                Mail::send('email.delay-payment-link', [
                    'booking' => $booking,
                    'customer' => [
                        'first_name' => $user->first_name,
                        'last_name' => $user->last_name,
                    ],
                    'product_name' => $booking->items->first()->product_name,
                    'rental_period' => $booking->pickup_date . ' to ' . $booking->new_dropoff_date,
                    'total_amount' => number_format($request->total, 2),
                    'payment_link' => $paymentLink,
                    'bookingItems' => $bookingItems,
                    'userDetails' => $userDetails,
                    'admin' => $admin,
                    'vendor'=>$vendor
                ], function ($message) use ($request, $booking, $vendor) {
                    $message->to($vendor->email)
                            ->subject('Payment Link for Your Booking #' . $booking->id);
                });
            }

            return redirect()->route('admin.booking.index')
            ->with('success', 'Payment link has been sent successfully.');

        }catch (\Exception $e) {
            dd($e->getMessage());
            return back()->with('error', 'Failed to resend payment link: ' . $e->getMessage());
        }
    }else if(!$request->delay){
        foreach ($booking->items as $item) {
            $item->update([
                'delay'             => 0,
                'new_dropoff_date'  => null,
                'new_dropoff_time'  => null,
                'additional_days'   => 0,
                'additional_days_rent' => 0,
            ]);
        }
        $reductionSubtotal = $booking->additional_days_rent * 0.81;
        $reductionTax      = ($booking->delay_fees + $booking->additional_days_rent) * 0.19;
        $reductionTotal    = $booking->delay_fees + $booking->additional_days_rent;


        $booking->update([
            'delay'               => 0,
            'delay_fees'          => 0,
            'additional_days'     => 0,
            'new_dropoff_date'    => null,
            'new_dropoff_time'    => null,
            'additional_days_rent'=> 0,

            'subtotal' => $booking->subtotal - $reductionSubtotal,
            'tax'      => $booking->tax - $reductionTax,
            'total'    => $booking->total - $reductionTotal,
        ]);

    }

    return redirect()->route('admin.booking.index')
    ->with('success', 'Booking Updated Successfully');


}


    public function resendPaymentLinkForDelay(Request $request, $id)
    {
        $booking = Booking::findOrFail($id);
        $user = $booking->userDetails;

        // Initialize Stripe
        if($booking->delay || $booking->additional_days_rent){
            Stripe::setApiKey(env('STRIPE_SECRET_KEY'));
            $total= $booking->delay_fees + $booking->additional_days_rent ; 
        
            try {
                $session = Session::create([
                    'payment_method_types' => [
                        'card',
                        'paypal',
                        
                        'amazon_pay',
                    ],
                    'line_items' => [[
                        'price_data' => [
                            'currency' => 'eur',
                            'product_data' => [
                                'name' => $booking->items->first()->product_name,
                            ],
                            'unit_amount' => (int)($total * 100),
                        ],
                        'quantity' => 1,
                    ]],
                    'mode' => 'payment',
                    'customer_email' => $user->email,
                    'metadata' => [
                        'booking_id' => $booking->id,
                    ],
                    'success_url' => route('delay.success').'?session_id={CHECKOUT_SESSION_ID}',
                    'cancel_url' => route('delay.cancel').'?session_id={CHECKOUT_SESSION_ID}',
                ]);
        
                $bookingItems = BookingItem::where('booking_id', $booking->id)->get();
                $userDetails = BookingUserDetail::where('booking_id', $booking->id)->first();
                $admin = Admin::first();
                $booking->update([
                    'delay_session_id' => $session->id,
                    'last_delay_payment_link_sent_at' => now(),
                ]);
                $paymentLink = $session->url;
                Mail::send('email.delay-payment-link', [
                    'booking' => $booking,
                    'customer' => [
                        'first_name' => $user->first_name,
                        'last_name' => $user->last_name,
                    ],
                    'product_name' => $booking->items->first()->product_name,
                    'rental_period' => $booking->pickup_date . ' to ' . $booking->new_dropoff_date,
                    'total_amount' => number_format($total, 2),
                    'payment_link' => $paymentLink,
                    'bookingItems' => $bookingItems,
                    'userDetails' => $userDetails,
                    'admin' => $admin,
                ], function ($message) use ($request, $booking, $user) {
                    $message->to('waza@yopmail.com')
                            ->subject('Payment Link for Your Booking #' . $booking->id);
                });
        
                Mail::send('email.delay-payment-link', [
                    'booking' => $booking,
                    'customer' => [
                        'first_name' => $user->first_name,
                        'last_name' => $user->last_name,
                    ],
                    'product_name' => $booking->items->first()->product_name,
                    'rental_period' => $booking->pickup_date . ' to ' . $booking->new_dropoff_date,
                    'total_amount' => number_format($total, 2),
                    'payment_link' => $paymentLink,
                    'bookingItems' => $bookingItems,
                    'userDetails' => $userDetails,
                    'admin' => $admin,
                ], function ($message) use ($request, $booking, $user) {
                    $message->to('info@dachboxit.de')
                            ->subject('Payment Link for Your Booking #' . $booking->id);
                });
        
                return redirect()->route('admin.booking.index')
                ->with('success', 'Payment link has been resend successfully.');
        
            }catch (\Exception $e) {
                return back()->with('error', 'Failed to resend payment link: ' . $e->getMessage());
            }
            }
        
    }


    public function getQuote()
{
    $bookingRequestIds = BookingRequest::orderBy('created_at', 'asc')
    ->pluck('booking_request_id')
    ->toArray();

   

    $categories = Category::with(['products' => function($query) {
        $query->with(['brand', 'images' => function($imageQuery) {
            $imageQuery->where('type', 'home');
        }]);
    }])->get();

    $manufacturers = MotherFile::distinct()->pluck('car_make');
    
    // Get the first product to pre-load the details
    $firstProduct = Product::with([
        'category',
        'brand',
        'variations.options.color',
        'variations',
        'images',
        'models',
        'otherInfo',
    ])->first();

    if (!$firstProduct) {
        abort(404, 'No products found');
    }

    $normalizedName = $this->normalizeProductName($firstProduct->name);

    $priceRecord = ProductPrice::get()->first(function ($record) use ($normalizedName) {
        return $this->normalizeProductName($record->product_name) === $normalizedName;
    });
    $roof_rackPrice = ProductPrice::where('product_name', 'Dachträger ergänzt zu Dachbox/ Dachfahrradträger')->first();
    
    if ($priceRecord && $priceRecord->price) {
        $prices = $priceRecord->price;
        $firstProduct->first_price = $prices[1] ?? null;

        $rawPrices = ($priceRecord->price);
        $dailyPrices = array_map(function ($price) {
            return floatval(str_replace(['€', ',', ' '], ['.', '.', ''], $price));
        }, $rawPrices);

        $rawRoofRack = ($roof_rackPrice->price);
        $roofRackPrices = array_map(function ($price) {
            return floatval(str_replace(['€', ',', ' '], ['.', '.', ''], $price));
        }, $rawRoofRack);
    } else {
        $firstProduct->first_price = null;
        $dailyPrices = [];
        $roofRackPrices = [];
    }

    $category = $firstProduct->category;

    $roofTypes = [
        'Normal Roof',
        'Normal roof without glass roof',
        'Normal Roof with glass roof',
        'Fixpoints',
        'Fixpoints without glass roof',
        'Fixpoints with glass roof',
        'Integrated Roof Railing',
        'Elevated Roof Rails',
        'T-Profile',
        'Raingutter'
    ];

    $sizeIds = is_array($category->size_ids) ? $category->size_ids : json_decode($category->size_ids, true);
    $variationIds = is_array($category->variation_ids) ? $category->variation_ids : json_decode($category->variation_ids, true);
    $colorIds = is_array($category->color_ids) ? $category->color_ids : json_decode($category->color_ids, true);

    $inStockVariationRecords = \App\Models\ProductVariation::where('product_id', $firstProduct->id)
        ->where('status', 'in_stock')
        ->get();
    $inStockSizeIds = $inStockVariationRecords->pluck('size_id')->filter()->unique()->toArray();
    $inStockVariationIds = $inStockVariationRecords->pluck('variation_id')->filter()->unique()->toArray();
    $sizes = $inStockSizeIds ? Size::whereIn('id', $inStockSizeIds)->get() : collect();
    $variations = $inStockVariationIds ? Variation::whereIn('id', $inStockVariationIds)->get() : collect();



    // $sizes = $sizeIds ? Size::whereIn('id', $sizeIds)->get() : collect();
    // $variations = $variationIds ? Variation::whereIn('id', $variationIds)->get() : collect();
    $colors = $colorIds ? Color::whereIn('id', $colorIds)->get() : collect();
    $holidays = \App\Models\Holidays::pluck('date')->toArray();
    $closedDates = \App\Models\DateAvailability::where('is_closed', true)
        ->pluck('date')
        ->map(fn($date) => $date instanceof \Carbon\Carbon ? $date->format('Y-m-d') : (string) $date)
        ->toArray();
    
    $holidays = array_unique(array_merge($holidays, $closedDates));

    $nonworkingDays = WorkingHours::whereNull('open_time')
        ->orWhereNull('close_time')
        ->pluck('day')
        ->map(fn($day) => strtolower($day)) // make lowercase for matching JS getDay()
        ->toArray();

    $vendors = Vendor::where('status','active')->get(['id', 'street','zip_code','city']);

    $vendors = Vendor::where('status', 'active')
    ->whereHas('products', function($query) use ($firstProduct) {
        $query->where('products.id', $firstProduct->id);
    })
    ->get(['id', 'street','zip_code','city']);
    
    $productType =  $this->determineProductType($firstProduct->name);

    return view($this->filePath . '.quote', compact(
        'categories',
        'manufacturers',
        'firstProduct',
        'priceRecord',
        'colors',
        'sizes',
        'variations',
        'roofTypes',
        'dailyPrices',
        'roofRackPrices',
        'holidays',
        'nonworkingDays',
        'vendors',
        'productType',
        'bookingRequestIds'
    ));
}


public function getBookingRequestData(Request $request)
{
    $bookingRequestData = BookingRequest::where('booking_request_id', $request->bookingID)->first();

    if (!$bookingRequestData) {
        return response()->json(['error' => 'Booking request not found'], 404);
    }

    return response()->json($bookingRequestData);
}

public function getProductDetails(Request $request)
{
    $product = Product::with([
        'category',
        'brand',
        'variations.options.color',
        'variations',
        'images',
        'models',
        'otherInfo',
    ])->find($request->product_id);


    $productType = $this->determineProductType($product->name);



    if (!$product) {
        return response()->json(['error' => 'Product not found'], 404);
    }
    $manufacturers = MotherFile::distinct()->pluck('car_make');

    $normalizedName = $this->normalizeProductName($product->name);

    $priceRecord = ProductPrice::get()->first(function ($record) use ($normalizedName) {
        return $this->normalizeProductName($record->product_name) === $normalizedName;
    });
    $roof_rackPrice = ProductPrice::where('product_name', 'Dachträger ergänzt zu Dachbox/ Dachfahrradträger')->first();
    
    if ($priceRecord && $priceRecord->price) {
        $prices = $priceRecord->price;
        $product->first_price = $prices[1] ?? null;

        $rawPrices = ($priceRecord->price);
        $dailyPrices = array_map(function ($price) {
            return floatval(str_replace(['€', ',', ' '], ['.', '.', ''], $price));
        }, $rawPrices);

        $rawRoofRack = ($roof_rackPrice->price);
        $roofRackPrices = array_map(function ($price) {
            return floatval(str_replace(['€', ',', ' '], ['.', '.', ''], $price));
        }, $rawRoofRack);
    } else {
        $product->first_price = null;
        $dailyPrices = [];
        $roofRackPrices = [];
    }

    $category = $product->category;

    $roofTypes = [
        'Normal Roof',
        'Normal roof without glass roof',
        'Normal Roof with glass roof',
        'Fixpoints',
        'Fixpoints without glass roof',
        'Fixpoints with glass roof',
        'Integrated Roof Railing',
        'Elevated Roof Rails',
        'T-Profile',
        'Raingutter'
    ];

    $sizeIds = is_array($category->size_ids) ? $category->size_ids : json_decode($category->size_ids, true);
    $variationIds = is_array($category->variation_ids) ? $category->variation_ids : json_decode($category->variation_ids, true);
    $colorIds = is_array($category->color_ids) ? $category->color_ids : json_decode($category->color_ids, true);

    $inStockVariationRecords = \App\Models\ProductVariation::where('product_id', $product->id)
        ->where('status', 'in_stock')
        ->get();
    $inStockSizeIds = $inStockVariationRecords->pluck('size_id')->filter()->unique()->toArray();
    $inStockVariationIds = $inStockVariationRecords->pluck('variation_id')->filter()->unique()->toArray();
    $sizes = $inStockSizeIds ? Size::whereIn('id', $inStockSizeIds)->get() : collect();
    $variations = $inStockVariationIds ? Variation::whereIn('id', $inStockVariationIds)->get() : collect();

    // $sizes = $sizeIds ? Size::whereIn('id', $sizeIds)->get() : collect();
    // $variations = $variationIds ? Variation::whereIn('id', $variationIds)->get() : collect();
    $colors = $colorIds ? Color::whereIn('id', $colorIds)->get() : collect();

    $holidays = \App\Models\Holidays::pluck('date')->toArray();
    $closedDates = \App\Models\DateAvailability::where('is_closed', true)
        ->pluck('date')
        ->map(fn($date) => $date instanceof \Carbon\Carbon ? $date->format('Y-m-d') : (string) $date)
        ->toArray();
    
    $holidays = array_unique(array_merge($holidays, $closedDates));
    $nonworkingDays = WorkingHours::whereNull('open_time')
    ->orWhereNull('close_time')
    ->pluck('day')
    ->map(fn($day) => strtolower($day)) // make lowercase for matching JS getDay()
    ->toArray();


    

    // $vendors = Vendor::where('status','active')->get(['id', 'street','zip_code','city']);

    $vendors = Vendor::where('status', 'active')
    ->whereHas('products', function($query) use ($product) {
        $query->where('products.id', $product->id);
    })
    ->get(['id', 'street','zip_code','city']);


    $modalView = match ($productType) {
        'roof_rack' => 'admin.booking.partials.roofRackModal',
        'carrier_foot' => 'admin.booking.partials.carrierModal',
        default => 'admin.booking.partials.modals',
    };
    $modalHtml = view($modalView, [
        'dailyPrices' => $dailyPrices,
    ])->render();

    return response()->json([
        'product' => $product,
        'priceRecord' => $priceRecord,
        'colors' => $colors,
        'sizes' => $sizes,
        'variations' => $variations,
        'dailyPrices' => $dailyPrices,
        'roofRackPrices' => $roofRackPrices,
        'productType' => $productType,
        'nonworkingDays' => $nonworkingDays,
        'modalHtml' => $modalHtml,
        'holidays' => $holidays,
        'view' => view($this->filePath . '.partials.dynamic_product', compact(
            'product',
            'priceRecord',
            'colors',
            'sizes',
            'variations',
            'roofTypes',
            'dailyPrices',
            'roofRackPrices',
            'manufacturers',
            'holidays',
            'nonworkingDays',
            'vendors',
            'productType'
        ))->render()
    ]);
}


private function determineProductType($productName)
{
    if (str_contains($productName, 'Roof Rack')) return 'roof_rack';
    if (str_contains($productName, 'Dachträger')) return 'roof_rack';
    if (in_array($productName, ['Carrier', 'Foot', 'Footkit' , 'Traverse' , 'Fuß', 'Fußkit'])) return 'carrier_foot';
    return 'standard';
}


public function add(Request $request)
{
    $rules = [
        'product_id' => 'required|exists:products,id',
        'product_name' => 'required|string',
        'manufacturer' => 'nullable|string',
        'car_year' => 'nullable|string',
        'model' => 'nullable|string',
        'trailer_hitch' => 'nullable|string',
        'roof_type' => 'nullable|string',
        'rental_from' => 'required|date',
        'rental_to' => 'required|date|after_or_equal:rental_from',
        'days' => 'required|integer|min:1',
        'calculated_rent' => 'required|numeric|min:0',
        'is_carrier_included' => 'boolean',
        'is_insured' => 'boolean',
        'carrier_length' => 'nullable|string',
        'foot_sku' => 'nullable|string',
        'footkit_sku' => 'nullable|string',
        'vendor_id' => 'nullable',
        'bundle_name' => 'nullable|string',
        'booking_item_id' => 'nullable',
        'booking_item_ids' => 'nullable',
        'booking_request_id'=>'nullable',
    ];
    
    $product = Product::find($request->product_id);
    $isRoofRack = $product && (strtolower($product->name) === 'roof rack' || strtolower($product->name) === 'dachträger' );
    if (!$isRoofRack) {
        $rules['color_id'] = 'required|exists:colors,id';
        $rules['color_name'] = 'required|string';
        $rules['size_id'] = 'required';
        $rules['size_name'] = 'required|string';
    } else {
        $rules['color_id'] = 'nullable';
        $rules['color_name'] = 'nullable';
        $rules['size_id'] = 'nullable';
        $rules['size_name'] = 'nullable';
    }
    
    $validated = $request->validate($rules);
   



    if (isset($request->is_guest) && !empty($request->is_guest)) {
        $guestValidator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'surname' => 'required|string|max:255',
            'email' => [
                'required',
                'email',
                'max:255',
                // Rule::unique('users', 'email'),
            ],
        ]);
        if ($guestValidator->fails()) {
            return response()->json(['status' => false, 'message' => $guestValidator->errors()->first()]);
        }

        
        $user = User::where('email', $request->email)->first();
        if (!$user) {
            $user = new User();
            $user->first_name = $request->name;
            $user->last_name = $request->surname;
            $user->email = $request->email;
            $user->number = $request->number;
            $user->is_guest = 1;
            $user->password = Hash::make('12345678');
            $user->street = "";
            $user->zip_code = "";
            $user->city = "";
            $user->region = "";
            $user->save();
        }
        \Illuminate\Support\Facades\Session::put('guest_user', $user);

    }
   
    $user_id = $user->id;

    $bookingItemData = [];
    if (!empty($validated['booking_item_id'])) {
        $bookingItemData[$validated['product_name']] = (int) $validated['booking_item_id'];
    }

    if (!empty($validated['booking_item_ids'])) {
        $extraIds = json_decode($validated['booking_item_ids'], true);
        if (is_array($extraIds)) {
            $bookingItemData = array_merge($bookingItemData, $extraIds);
        }
    }
  
    $cartItemData = [
        'user_id' => $user_id,
        'product_id' => $validated['product_id'],
        'product_name' => $validated['product_name'],
        'car_name' => $validated['manufacturer'],
        'car_year' => $validated['car_year'],
        'model' => $validated['model'],
        'trailer_hitch' => $validated['trailer_hitch'] === 'yes',
        'roof_type' => $validated['roof_type'],
        'is_carrier_included' => $validated['is_carrier_included'] ?? false,
        'is_insured' => $validated['is_insured'] ?? false,
        'pickup_date' => $validated['rental_from'],
        'dropoff_date' => $validated['rental_to'],
        'days' => $validated['days'],
        'footkit_sku' => $validated['footkit_sku'],
        'foot_sku' => $validated['foot_sku'],
        'carrier_length' => $validated['carrier_length'],
        'calculated_rent' => $validated['calculated_rent'],
        'status' => 'pending',
        'bundle_name' => $validated['bundle_name'],
        'vendor_id' => $validated['vendor_id'],
        'booking_item_id' => json_encode($bookingItemData),
        'booking_request_id' => $validated['booking_request_id'] ?? 0,
    ];
    
    if (isset($validated['color_name'])) {
        $cartItemData['color'] = $validated['color_name'];
    }
    if (isset($validated['size_name'])) {
        $cartItemData['size'] = $validated['size_name'];
    }
    
    $cartItem = CartItem::create($cartItemData);

    return response()->json([
        'redirect' => route('admin.cart.confirm', $cartItem->id)
    ]);
}

public function confirm(CartItem $cartItem)
{
    return view('admin.booking.bookingslot', compact('cartItem'));
}

public function confirmSlot(Request $request)
{
    $request->validate([
        'cart_item_id' => 'required|exists:cart_items,id',
        'pickup_time' => ['required', 'string', 'regex:/^\d{2}:\d{2}\s*-\s*\d{2}:\d{2}$/'],
        'dropoff_time' => ['required', 'string', 'regex:/^\d{2}:\d{2}\s*-\s*\d{2}:\d{2}$/'],
        'pickup_date' => ['required'],
        'dropoff_date' => ['required']

    ]);


    [$pickupStart, $pickupEnd] = explode(' - ', $request->pickup_time);
    [$dropoffStart, $dropoffEnd] = explode(' - ', $request->dropoff_time);

    if($request->pickup_date == $request->dropoff_date){
        if ($pickupStart === $dropoffStart) {
            return response()->json(['error' => 'Drop-off time cannot be the same as pick-up time.'], 422);
        }
    }

    $cartItem = CartItem::find($request->cart_item_id);
    $cartItem->pickup_time = $pickupStart;
    $cartItem->dropoff_time = $dropoffStart;
    $cartItem->user_timing = $request->user_timing;
    $cartItem->pickup_end = $pickupEnd;
    $cartItem->dropoff_end = $dropoffEnd;

    $cartItem->save();
    return response()->json([
        'redirect' => route('admin.cart.index')
    ]);
}

public function cartPage()
{
        if (\Illuminate\Support\Facades\Session::has('guest_user') && isset(\Illuminate\Support\Facades\Session::get('guest_user')->id)) {
            $user_id = \Illuminate\Support\Facades\Session::get('guest_user')->id;
        
        try {

            $cartItems = CartItem::where('user_id', $user_id)
            ->where('status', 'pending')
            ->with('product')
            ->get();
        
        $taxRate = 0.19;
        $onePlus = 1 + $taxRate;
        $insuranceFlatRate = 25.00;
        
        // --- totals (gross) ---
        $grossBeforeDiscount = $cartItems->sum('calculated_rent');
        $discount            = (float) $cartItems->sum('discount'); // discount applies to total gross
        $grossTotal          = max($grossBeforeDiscount - $discount, 0.0); // this is your TOTAL
        
        // insurance gross (flat 25 per insured item) – NOT reduced by discount
        $insuranceCount = $cartItems->where('is_insured', true)->count();
        $insuranceGross = $insuranceCount * $insuranceFlatRate;
        
        // --- nets & tax ---
        $netSubtotal         = $grossTotal / $onePlus;              // “Subtotal” (net, tax-free)
        $insuranceNetTotal   = $insuranceGross / $onePlus;          // “Net Price Insurance”
        $rentalNet           = max($netSubtotal - $insuranceNetTotal, 0.0); // “Net Price” (rent only)
        $tax                 = $grossTotal - $netSubtotal;          // total VAT (includes insurance VAT)
        
        // optional: set item-level rent net/tax (discount spread only over the rental part)
        $totalRentalGrossBeforeDiscount = $cartItems->sum(function ($it) use ($insuranceFlatRate) {
            return $it->calculated_rent - ($it->is_insured ? $insuranceFlatRate : 0);
        });
        
        $cartItems->each(function ($item) use ($onePlus, $insuranceFlatRate, $discount, $totalRentalGrossBeforeDiscount) {
            $baseRentalGross = $item->calculated_rent - ($item->is_insured ? $insuranceFlatRate : 0);
            $share = $totalRentalGrossBeforeDiscount > 0 ? ($baseRentalGross / $totalRentalGrossBeforeDiscount) : 0;
            $itemRentalGrossAfterDiscount = max($baseRentalGross - ($discount * $share), 0.0);
        
            $item->net_price  = $itemRentalGrossAfterDiscount / $onePlus;         // rent net (per item)
            $item->tax_amount = $itemRentalGrossAfterDiscount - $item->net_price; // rent VAT (per item)
            // (Insurance per-item net/tax available if you need them:
            // $item->insurance_net = $item->is_insured ? ($insuranceFlatRate / $onePlus) : 0;
            // $item->insurance_tax = $item->is_insured ? ($insuranceFlatRate - $item->insurance_net) : 0;
        });
        
        // keep your address logic
        $adminAddress = 'Esperantostr. 10, 70197 Stuttgart';
        $vendors = Vendor::get()->keyBy('id');
        $cartItems->each(function ($item) use ($vendors, $adminAddress) {
            if ($item->vendor_id && isset($vendors[$item->vendor_id])) {
                $vendor = $vendors[$item->vendor_id];
                $item->vendor_address = trim("{$vendor->street} {$vendor->zip_code} {$vendor->city}");
            } else {
                $item->vendor_address = $adminAddress;
            }
            

        });


      
        
        
        // expose to the view
        return view($this->filePath . '.cartPage', [
            'cartItems'      => $cartItems,
            'subtotal'       => $netSubtotal,        
            'discount'       => $discount,
            'insuranceTotal' => $insuranceNetTotal,  
            'tax'            => $tax,               
            'netPrice'       => $rentalNet,          
            'total'          => $grossTotal,     
            
        ]);
        } catch (\Exception $e) {
            dd($e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong! Please try again later.');
        }
    }
}




public function applyCoupon(Request $request)
{
    $couponCode = strtolower($request->input('coupon_code'));
    $userId = $request->input('user_id');
    $sub_total = $request->input('sub_total');
    $voucher = Voucher::where('code', $couponCode)->first();

    if (!$voucher) {
        return response()->json([
            'success' => false,
            'message' => 'Invalid coupon code'
        ]);
    }
    if (!$voucher->is_active) {
        return response()->json([
            'success' => false,
            'message' => 'This coupon is not currently active'
        ]);
    }
    $now = Carbon::now();
    if ($now->lt($voucher->start_date)) {
        return response()->json([
            'success' => false,
            'message' => 'This coupon is not valid yet'
        ]);
    }

    if ($now->gt($voucher->expiration_date)) {
        return response()->json([
            'success' => false,
            'message' => 'This coupon has expired'
        ]);
    }

    $totalDiscountUsed = DB::table('bookings')
        ->whereRaw("JSON_UNQUOTE(JSON_EXTRACT(coupon_code, '$[0]')) = ?", [strtolower($voucher->code)])
        ->sum('discount');
    if ($totalDiscountUsed >= $voucher->max_discount) {
        $userDetails = User::find($userId);
        if(!$voucher->notification_send){

            Notification::create([
                'type' => 'max_voucher_limit_exceed',
                'status' => 'Voucher Limit Exceed', 
                'user_id' => $userId,
                'customer_name' => $userDetails->first_name . ' ' . $userDetails->last_name,
                'customer_email' => $userDetails->email,
                'customer_phone' => $userDetails->number,
                'data' => [
                    'voucher_code' => $couponCode,
                    'exceeded_at' => now()->toDateTimeString(),
                ],
                'is_read' => false,
            ]);

            $voucher->notification_send = 1;
            $voucher->save();
        }

        return response()->json([
            'success' => false,
            'message' => 'This coupon has reached its maximum discount limit'
        ]);
    }

    if ($userId) {
        $userDiscountUsed = DB::table('bookings')
            ->where('user_id', $userId)
            ->whereRaw("JSON_UNQUOTE(JSON_EXTRACT(coupon_code, '$[0]')) = ?", [strtoupper($voucher->code)])
            ->sum('discount');

        $userRedemptionCount = DB::table('bookings')
            ->where('user_id', $userId)
            ->whereRaw("JSON_UNQUOTE(JSON_EXTRACT(coupon_code, '$[0]')) = ?", [strtoupper($voucher->code)])
            ->count();

        if ($userRedemptionCount >= $voucher->redemption_limit) {
            return response()->json([
                'success' => false,
                'message' => 'You have reached your usage limit for this coupon'
            ]);
        }
    }


    $applicableProductIds = json_decode($voucher->applied_to, true) ?? [];
    $applicableSpos = json_decode($voucher->applied_to_spo, true) ?? [];

    $cartItemsQuery = CartItem::where('user_id', $userId)
        ->where('status', 'pending');

    if (!empty($applicableProductIds)) {
        $cartItemsQuery->whereIn('product_id', $applicableProductIds);
    }
    if (!empty($applicableSpos)) {
        $cartItemsQuery->where(function ($query) use ($applicableSpos) {
            $query->whereIn('vendor_id', $applicableSpos)
                  ->orWhereNull('vendor_id'); // allow admin bookings
        });
    }

    $applicableCartItems = $cartItemsQuery->get();

    if ($applicableCartItems->isEmpty()) {
        return response()->json([
            'success' => false,
            'message' => 'Dieser Gutschein kann auf keine Produkte in Ihrem Warenkorb angewendet werden.'
        ]);
    }

    $alreadyAppliedItems = $applicableCartItems->filter(function ($item) use ($couponCode) {
        if (empty($item->coupon_code)) {
            return false;
        }

        $appliedCoupons = json_decode($item->coupon_code, true) ?? [];
        return in_array($couponCode, $appliedCoupons);
    });

    if ($alreadyAppliedItems->isNotEmpty()) {
        return response()->json([
            'success' => false,
            'message' => 'This coupon is already applied to one or more items in your cart'
        ]);
    }


    $applicableCartTotal = $applicableCartItems->sum('calculated_rent');

    if ($sub_total < $voucher->min_order_value) {
        return response()->json([
            'success' => false,
            'message' => 'Minimum order value for this coupon is €' . $voucher->min_order_value .
                ' (Your order total: €' . $sub_total . ')'
        ]);
    }

    if ($voucher->redemption_limit > 0 && $voucher->times_redeemed >= $voucher->redemption_limit) {
        return response()->json([
            'success' => false,
            'message' => 'This coupon has reached its redemption limit'
        ]);
    }

    $existingCoupons = CartItem::where('user_id', $userId)
        ->where('status', 'pending')
        ->whereNotNull('coupon_code')
        ->distinct('coupon_code')
        ->pluck('coupon_code');

    if ($existingCoupons->isNotEmpty()) {
        if (!$existingCoupons->contains($couponCode)) {
            if (!$voucher->combine_with_other_vouchers) {
                return response()->json([
                    'success' => false,
                    'message' => 'Another coupon is already applied and this coupon cannot be combined'
                ]);
            }

            $existingVouchers = Voucher::whereIn('code', $existingCoupons)->get();
            foreach ($existingVouchers as $existingVoucher) {
                if (!$existingVoucher->combine_with_other_vouchers) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Existing coupon cannot be combined with other coupons'
                    ]);
                }
            }
        }
    }

    $allCartItems = CartItem::where('user_id', $userId)
        ->where('status', 'pending')
        ->get();

    $applicableProductIds = json_decode($voucher->applied_to, true) ?? [];
    $applicableCartItems = $allCartItems->filter(function ($item) use ($applicableProductIds) {
        return in_array($item->product_id, $applicableProductIds);
    });

    $appliesToAllItems = $allCartItems->count() === $applicableCartItems->count();

    if ($appliesToAllItems) {
        $discountBaseAmount = $sub_total;
    } else {
        $discountBaseAmount = $applicableCartItems->sum('calculated_rent');
    }

    $remainingDiscount = $voucher->max_discount - $totalDiscountUsed;
    if ($applicableCartItems->isNotEmpty()) {

        $discountPerItem = [];
        foreach ($applicableCartItems as $item) {
            if ($voucher->type == 1) {
                $potentialDiscount = $voucher->voucher_value;
            } else {
                $potentialDiscount = $discountBaseAmount * ($voucher->voucher_value / 100);
            }
            $totalDiscount = min($potentialDiscount, $remainingDiscount);
            $discountPerItem[$item->id] = $totalDiscount;
        }

    } else {
        if ($voucher->type == 1) {
            $potentialDiscount = $voucher->voucher_value;
        } else {
            $potentialDiscount = $sub_total * ($voucher->voucher_value / 100);
        }

        $discountPerItem = $potentialDiscount;
    }


    $totalDiscount = min($potentialDiscount, $remainingDiscount);


    $this->updateCartWithDiscount($applicableCartItems, $couponCode, $discountPerItem);


    $fullCartTotal = $sub_total;
    $totalDiscount = $applicableCartItems->sum('discount');
    return response()->json([
        'success' => true,
        'message' => 'Coupon applied successfully!',
        'discount' => $totalDiscount,
        'new_total' => $fullCartTotal - $totalDiscount,
        'applicable_products_total' => $applicableCartTotal
    ]);
}



private function updateCartWithDiscount($cartItems, $couponCode, $discountPerItem)
{
    $alreadyApplied = collect($cartItems)->contains(function ($item) use ($couponCode) {
        $appliedCoupons = json_decode($item->coupon_code, true) ?? [];
        return in_array($couponCode, $appliedCoupons);
    });

    foreach ($cartItems as $item) {
        $appliedCoupons = json_decode($item->coupon_code, true) ?? [];
        if (in_array($couponCode, $appliedCoupons) || $alreadyApplied) {
            continue;
        }

        $itemDiscount = is_array($discountPerItem)
            ? ($discountPerItem[$item->id] ?? 0)
            : $discountPerItem;

        $newAppliedCoupons = array_merge($appliedCoupons, [$couponCode]);

        $item->update([
            'coupon_code' => json_encode($newAppliedCoupons),
            'discount' => ($item->discount ?? 0) + $itemDiscount,
            'updated_at' => now()
        ]);

        $alreadyApplied = true;
    }
}

public function removeSingleCoupon(Request $request)
{
    $item = CartItem::find($request->input('item_id'));

    if (!$item) {
        return response()->json(['success' => false, 'message' => 'Item not found']);
    }

    $coupons = json_decode($item->applied_coupons, true) ?? [];
    $remainingCoupons = [];
    $removedDiscount = 0;

    foreach ($coupons as $coupon) {
        if ($coupon['code'] !== $request->input('coupon_code')) {
            $remainingCoupons[] = $coupon;
        } else {
            $removedDiscount = $coupon['discount'];
        }
    }

    $item->update([
        'coupon_code' => !empty($remainingCoupons) ? json_encode($remainingCoupons) : null,
        'discount' => $item->total_discount - $removedDiscount,
    ]);

    return response()->json([
        'success' => true,
        'message' => 'Gutschein entfernt',
    ]);
}

public function checkOutIndex()
{
    try {
       
            $user_id =  \Illuminate\Support\Facades\Session::get('guest_user')->id;
            $is_guest = true;
       


            $cartItems = CartItem::where('user_id', $user_id)
            ->where('status', 'pending')
            ->with('product')
            ->get();
        
        $taxRate = 0.19;
        $onePlus = 1 + $taxRate;
        $insuranceFlatRate = 25.00;
        
        // --- totals (gross) ---
        $grossBeforeDiscount = $cartItems->sum('calculated_rent');
        $discount            = (float) $cartItems->sum('discount'); // discount applies to total gross
        $grossTotal          = max($grossBeforeDiscount - $discount, 0.0); // this is your TOTAL
        
        // insurance gross (flat 25 per insured item) – NOT reduced by discount
        $insuranceCount = $cartItems->where('is_insured', true)->count();
        $insuranceGross = $insuranceCount * $insuranceFlatRate;
        
        // --- nets & tax ---
        $netSubtotal         = $grossTotal / $onePlus;              // “Subtotal” (net, tax-free)
        $insuranceNetTotal   = $insuranceGross / $onePlus;          // “Net Price Insurance”
        $rentalNet           = max($netSubtotal - $insuranceNetTotal, 0.0); // “Net Price” (rent only)
        $tax                 = $grossTotal - $netSubtotal;          // total VAT (includes insurance VAT)
        
        // optional: set item-level rent net/tax (discount spread only over the rental part)
        $totalRentalGrossBeforeDiscount = $cartItems->sum(function ($it) use ($insuranceFlatRate) {
            return $it->calculated_rent - ($it->is_insured ? $insuranceFlatRate : 0);
        });
        
        $cartItems->each(function ($item) use ($onePlus, $insuranceFlatRate, $discount, $totalRentalGrossBeforeDiscount) {
            $baseRentalGross = $item->calculated_rent - ($item->is_insured ? $insuranceFlatRate : 0);
            $share = $totalRentalGrossBeforeDiscount > 0 ? ($baseRentalGross / $totalRentalGrossBeforeDiscount) : 0;
            $itemRentalGrossAfterDiscount = max($baseRentalGross - ($discount * $share), 0.0);
        
            $item->net_price  = $itemRentalGrossAfterDiscount / $onePlus;         // rent net (per item)
            $item->tax_amount = $itemRentalGrossAfterDiscount - $item->net_price; // rent VAT (per item)
            // (Insurance per-item net/tax available if you need them:
            // $item->insurance_net = $item->is_insured ? ($insuranceFlatRate / $onePlus) : 0;
            // $item->insurance_tax = $item->is_insured ? ($insuranceFlatRate - $item->insurance_net) : 0;
        });
        if($is_guest){
            $view = '.guest';
        }

        $billingAddresses = [];
        $defaultAddress = null;
        $formValues = [];


        $bookingRequestData = BookingRequest::where('booking_request_id', $cartItems->first()->booking_request_id)->first();

        $firstName = '';
        $lastName  = '';

        if (!empty($bookingRequestData->customer_name)) {
            $parts = explode(' ', trim($bookingRequestData->customer_name));
            $firstName = implode(' ', array_slice($parts, 0, -1)); // everything except last word
            $lastName  = end($parts); // last word
        }
        if (!empty($bookingRequestData->customer_phone)) {
            // Split by first space
            $parts = preg_split('/\s+/', trim($bookingRequestData->customer_phone), 2);
            $phoneCode = $parts[0] ?? '+49';
            $phoneNumber = $parts[1] ?? '';
        }


        $formValues = [
            'email'        => $bookingRequestData->customer_email ?? '',
            'first_name'   => $firstName  ?? '',
            'last_name'    => $lastName  ?? '',
            'company_name' =>  '',
            'address'      => $bookingRequestData->street ?? '',
            'zip'          => $bookingRequestData->zipcode ?? '',
            'city'         => $bookingRequestData->city ?? '',
            'region'       => $bookingRequestData->state ?? '',
            'phone_code'   => $phoneCode ?? '+49',
            'phone_number' => $phoneNumber ?? '',
            
        ];

        
       
        

        return view($this->filePath . $view, [
            'cartItems' => $cartItems,
            'subtotal' => $netSubtotal,
            'discount' => $discount,
            'insuranceTotal' => $insuranceNetTotal,
            'tax' => $tax,
            'netPrice' => $rentalNet,
            'total' => $grossTotal,
            'billingAddresses' => $billingAddresses,
            'defaultAddress' => $defaultAddress,
      
            'formValues' => $formValues,
        ]);
    } catch (\Exception $e) {
        dd($e->getMessage());
        return redirect()->back()->with('error', 'Something went wrong! Please try again later.');
    }
}
    public function checkout(Request $request)
    {


            $user = \Illuminate\Support\Facades\Session::get('guest_user');
            if ($request->create_customer === 'on') {
                $randomPassword = Str::random(10);
            
                \App\Models\User::where('id', $user->id)->update([
                    'email' => $request->email,
                    'zip_code' => $request->zip,
                    'city' => $request->city,
                    'region' => 'Deutschland',
                    'street' => $request->address,
                    'password' => Hash::make($randomPassword),
                    'status'=> 1,
                    'is_guest' => 0,  
                ]);

                
           
                // Mail::raw("Your account has been created with email {$request->email}. Your password is: {$randomPassword}", function ($message) use ($request) {
                //     $message->to($request->email)
                //             ->subject('Your Customer Account Details');
                // });

                try {
                    Mail::to($request->email)->send(new GuestAccountMail(
                        $request->email,
                        $randomPassword
                    ));
                    
                } catch (\Exception $e) {
    
                    return back()->with('error', 'Email sending failed: ' . $e->getMessage());
                }
            }
        

       
       
        $cartItems = CartItem::where('user_id', $user->id)
            ->where('status', 'pending')
            ->with('product')
            ->get();
    
        if ($cartItems->isEmpty()) {
            return redirect()->back()->with('error', 'Your cart is empty');
        }
    
        $taxRate = 0.19;
        $insuranceFlatRate= 25;
        $insuranceTaxTotal = 0;
        $cartItems->each(function ($item) use ($taxRate,&$insuranceTaxTotal) {
            if ($item->is_insured) {
                $item->taxable_amount = $item->calculated_rent - 25;

                $insuranceNet = 25 / (1 + $taxRate);
                $insuranceTax = $insuranceNet * $taxRate;
                
                $insuranceTaxTotal += $insuranceTax;
            } else {
                $item->taxable_amount = $item->calculated_rent;
            }
            $item->net_price = $item->taxable_amount / (1 + $taxRate);
            $item->tax_amount = $item->net_price * $taxRate;
        });
    
        $netPrice = $cartItems->sum('net_price');
        $tax = $cartItems->sum('tax_amount')  + $insuranceTaxTotal;
        $insuranceTotal = $cartItems->where('is_insured', true)->sum(function($item) {
            return 25;
        });
        $subtotal = $netPrice + $tax - $insuranceTaxTotal ;
        $discount = $cartItems->sum('discount');
        $total = ($subtotal - $discount) + $insuranceTotal;
        $subtotal = ($total+$discount) / (1 + $taxRate)  - $discount;
        $appliedCoupons = $cartItems->flatMap(function ($item) {
            return json_decode($item->coupon_code, true) ?? [];
        })->unique()->filter()->values()->toArray();


        Voucher::whereIn('code', $appliedCoupons)
        ->each(function ($voucher) use ($appliedCoupons) {
            $timesApplied = count(array_filter($appliedCoupons, fn($code) => $code === $voucher->code));
            $voucher->increment('times_redeemed');
        });

        $vendorIds = $cartItems->pluck('vendor_id')->unique();
        $vendorId = $vendorIds->first();


        $booking = Booking::create([
            'user_id' => $user->id,
            'status' => 'pending',
            'subtotal' => $subtotal,
            'discount' => $discount,
            'coupon_code' => !empty($appliedCoupons) ? json_encode($appliedCoupons) : null,
            'tax' => $tax,
            'total' => $total,
            'payment_method' => 'N/A',
            'vendor_id' => $vendorId,
            'payment_status' => 'unpaid',
        ]);
    
        BookingUserDetail::create([
            'booking_id' => $booking->id,
            'user_id' => $user->id,
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'email' => $request->email,
            'company_name' => $request->company_name,
            'phone' => $request->phone_code . $request->phone_number,
            'address' => $request->address,
            'zip' => $request->zip,
            'city' => $request->city,
            'country' => 'Deutschland',
        ]);

       


    
        foreach ($cartItems as $item) {
            $bookingItemIds = json_decode($item->booking_item_id, true) ?? [];
            $mainBookingItem  =  BookingItem::create([
                'booking_id' => $booking->id,
                'user_id' => $user->id,
                'product_id' => $item->product_id,
                'product_name' => $item->product_name,
                'car_name' => $item->car_name,
                'car_year' => $item->car_year,
                'model' => $item->model,
                'trailer_hitch' => $item->trailer_hitch,
                'roof_type' => $item->roof_type,
                'color' => $item->color,
                'size' => $item->size,
                'is_carrier_included' => $item->is_carrier_included,
                'is_insured' => $item->is_insured,
                'pickup_date' => $item->pickup_date,
                'dropoff_date' => $item->dropoff_date,
                'pickup_time' => $item->pickup_time,
                'dropoff_time' => $item->dropoff_time,
                'days' => $item->days,
                'footkit_sku' => $item->footkit_sku,
                'foot_sku' => $item->foot_sku,
                'carrier_length' => $item->carrier_length,
                'calculated_rent' => $item->calculated_rent,
                'status' => 'pending',
                'user_timing' => $item->user_timing,
                'vendor_id' => $item->vendor_id,
                'booking_item_id' => $bookingItemIds[$item->product_name] ?? null, 
                'booking_request_id' => $item->booking_request_id ?? 0,
                'pickup_end' => $item->pickup_end,
                'dropoff_end' => $item->dropoff_end
            ]);

            if($item->is_carrier_included || $item->product_name == 'Roof Rack' || $item->product_name == 'Dachträger'){
                if($item->carrier_length){
                    BookingItem::create([
                        'booking_id' => $booking->id,
                        'user_id' => $user->id,
                        'product_id' => Product::where('name', 'Traverse')->first()->id,
                        'product_name' => 'Traverse',
                        'car_name' => $item->car_name,
                        'car_year' => $item->car_year,
                        'model' => $item->model,
                        'trailer_hitch' => $item->trailer_hitch,
                        'roof_type' => $item->roof_type,
                       'color' => 'Schwarz',
                        'size' => intval(preg_replace('/[^0-9]/', '', $item->carrier_length)) / 10,
                        'is_carrier_included' => 0,
                        'is_insured' => $item->is_insured,
                        'pickup_date' => $item->pickup_date,
                        'dropoff_date' => $item->dropoff_date,
                        'pickup_time' => $item->pickup_time,
                        'dropoff_time' => $item->dropoff_time,
                        'days' => $item->days,
                        'footkit_sku' => $item->footkit_sku,
                        'foot_sku' => $item->foot_sku,
                        'carrier_length' => $item->carrier_length,
                        'calculated_rent' => 0,
                        'status' => 'pending',
                        'user_timing' => $item->user_timing,
                        'vendor_id' => $item->vendor_id,
                        'booking_item_id' => $bookingItemIds['carrier'] ?? $bookingItemIds['Carrier'] ?? null,
                        'related_item' =>  $mainBookingItem->id,
                        'is_roof_rack_product' => $item->product_name == 'Dachträger' ? 1 : 0,
                        'booking_request_id' => $item->booking_request_id ?? 0,
                        'pickup_end' => $item->pickup_end,
                        'dropoff_end' => $item->dropoff_end

                    ]);  
                }

                if($item->foot_sku){
                    BookingItem::create([
                        'booking_id' => $booking->id,
                        'user_id' => $user->id,
                        'product_id' => Product::where('name', 'Fuß')->first()->id,
                        'product_name' => 'Fuß',
                        'car_name' => $item->car_name,
                        'car_year' => $item->car_year,
                        'model' => $item->model,
                        'trailer_hitch' => $item->trailer_hitch,
                        'roof_type' => $item->roof_type,
                        'color' => 'Schwarz',
                        'size' => $item->foot_sku,
                        'is_carrier_included' => 0,
                        'is_insured' => $item->is_insured,
                        'pickup_date' => $item->pickup_date,
                        'dropoff_date' => $item->dropoff_date,
                        'pickup_time' => $item->pickup_time,
                        'dropoff_time' => $item->dropoff_time,
                        'days' => $item->days,
                        'footkit_sku' => $item->footkit_sku,
                        'foot_sku' => $item->foot_sku,
                        'carrier_length' => $item->carrier_length,
                        'calculated_rent' => 0,
                        'status' => 'pending',
                        'user_timing' => $item->user_timing,
                        'vendor_id' => $item->vendor_id,
                        'booking_item_id' => $bookingItemIds['fuß'] ?? null,
                        'related_item' =>   $mainBookingItem->id,
                        'is_roof_rack_product' => $item->product_name == 'Dachträger' ? 1 : 0,
                        'booking_request_id' => $item->booking_request_id ?? 0,
                        'pickup_end' => $item->pickup_end,
                        'dropoff_end' => $item->dropoff_end
                    ]);  
                }
                if($item->footkit_sku){
                    BookingItem::create([
                        'booking_id' => $booking->id,
                        'user_id' => $user->id,
                        'product_id' => Product::where('name', 'Fußkit')->first()->id,
                        'product_name' => 'Footkit',
                        'car_name' => $item->car_name,
                        'car_year' => $item->car_year,
                        'model' => $item->model,
                        'trailer_hitch' => $item->trailer_hitch,
                        'roof_type' => $item->roof_type,
                       'color' => 'Schwarz',
                        'size' => $item->footkit_sku,
                        'is_carrier_included' => 0,
                        'is_insured' => $item->is_insured,
                        'pickup_date' => $item->pickup_date,
                        'dropoff_date' => $item->dropoff_date,
                        'pickup_time' => $item->pickup_time,
                        'dropoff_time' => $item->dropoff_time,
                        'days' => $item->days,
                        'footkit_sku' => $item->footkit_sku,
                        'foot_sku' => $item->foot_sku,
                        'carrier_length' => $item->carrier_length,
                        'calculated_rent' => 0,
                        'status' => 'pending',
                        'user_timing' => $item->user_timing,
                        'vendor_id' => $item->vendor_id,
                        'booking_item_id' => $bookingItemIds['fußkit'] ?? null, 
                        'related_item' =>   $mainBookingItem->id,
                        'is_roof_rack_product' => $item->product_name == 'Dachträger' ? 1 : 0,
                        'booking_request_id' => $item->booking_request_id ?? 0,
                        'pickup_end' => $item->pickup_end,
                        'dropoff_end' => $item->dropoff_end
                    ]);  
                }
                
            }


          
        }


    
        Stripe::setApiKey(env('STRIPE_SECRET_KEY'));
    
        $lineItems = [];
        foreach ($cartItems as $item) {
            $lineItems[] = [
                'price_data' => [
                    'currency' => 'eur',
                    'product_data' => [
                        'name' => $item->product_name,
                    ],
                    'unit_amount' => (int)(($item->calculated_rent - $item->discount??0) * 100), 
                ],
                'quantity' => 1,
            ];
        }
    
       
    
      
        $stripeCustomer = \Stripe\Customer::create([
            'email' => $request->email,
            'name' => $request->first_name . ' ' . $request->last_name,
            'address' => [
                'line1' => $request->address,
                'postal_code' => $request->zip,
                'city' => $request->city,
                'country' => 'DE',
            ],
        ]);

       $session = Session::create([
            'payment_method_types' => [
                'card',
                'paypal',
                
                'amazon_pay',
            ],
            'line_items' => $lineItems,
            'mode' => 'payment',
            'customer' => $stripeCustomer->id, 
            'metadata' => [
                'booking_id' => $booking->id,
                'user_id' => $user->id,
            ],
            'success_url' => route('checkout.success') . '?session_id={CHECKOUT_SESSION_ID}',
            'cancel_url' => route('checkout.cancel') . '?session_id={CHECKOUT_SESSION_ID}',
        ]);

        $booking->update([
            'stripe_session_id' => $session->id,
            'last_payment_link_sent_at' => now(),
        ]);
    
        $paymentLink = $session->url;

        $bookingItems = BookingItem::where('booking_id', $booking->id)->get();
        $userDetails = BookingUserDetail::where('booking_id', $booking->id)->first();
        $admin = Admin::first();

        $vendorId = $bookingItems->pluck('vendor_id')->filter()->first();
        $vendor = $vendorId ? \App\Models\Vendor::find($vendorId) : null;
    
        $pickupInfo = $vendor
        ? [
            'name' => $vendor->name,
            'street' => $vendor->street,
            'zip_code' => $vendor->zip_code,
            'city' => $vendor->city,
            'state' => $vendor->country 
        ]
        : [
            'name' => $admin->name,
            'street' => $admin->address,
            'zip_code' => $admin->zip_code,
            'city' => $admin->city,
            'state' => $admin->state
        ];
    
        $emailData = [
            'booking' => $booking,
            'customer' => [
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
            ],
            'product_name' => $request->product_name,
            'rental_period' => $request->rental_from . ' to ' . $request->rental_to,
            'total_amount' => number_format($total, 2, ',', '.'),
            'payment_link' => $paymentLink,
            'bookingItems' => $bookingItems,
            'userDetails' => $userDetails,
            'pickupInfo' => $pickupInfo,
            'admin' => $admin
        ];
    
        Mail::send('email.admin-booking-payment-link', $emailData, function ($message) use ($request, $booking) {
            $message->to($request->email)
                    ->subject('Zahlungslink für Ihre Buchung bei Dachboxmiete – 24 Stunden gültig');
        });
        
        Mail::send('email.admin-booking-payment-link', $emailData, function ($message) use ($booking) {
            $message->to('info@dachboxit.de')
                    ->subject('Zahlungslink für Ihre Buchung bei Dachboxmiete – 24 Stunden gültig');
        });
        
        if ($vendor) {
            Mail::send('email.admin-booking-payment-link', $emailData, function ($message) use ($vendor, $booking) {
                $message->to($vendor->email)
                ->subject('Zahlungslink für Ihre Buchung bei Dachboxmiete – 24 Stunden gültig');
            });
        }


        
        // Mail::send('email.admin-booking-payment-link', [
        //     'booking' => $booking,
        //     'customer' => [
        //         'first_name' => $request->first_name,
        //         'last_name' => $request->last_name,
        //     ],
        //     'product_name' => $request->product_name,
        //     'rental_period' => $request->rental_from . ' to ' . $request->rental_to,
        //     'total_amount' => number_format($total, 2, ',', '.'),
        //     'payment_link' => $paymentLink,
        //     'bookingItems' => $bookingItems,
        //     'userDetails' => $userDetails,
        //     'admin' => $admin
        // ], function ($message) use ($request, $booking) {
        //     $message->to($request->email)
        //             ->subject('Zahlungslink für Ihre Buchung bei Dachboxmiete – 24 Stunden gültig');
        // });

        // Mail::send('email.admin-booking-payment-link', [
        //     'booking' => $booking,
        //     'customer' => [
        //         'first_name' => $request->first_name,
        //         'last_name' => $request->last_name,
        //     ],
        //     'product_name' => $request->product_name,
        //     'rental_period' => $request->rental_from . ' to ' . $request->rental_to,
        //     'total_amount' => number_format($total, 2, ',', '.'),
        //     'payment_link' => $paymentLink,
        //     'bookingItems' => $bookingItems,
        //     'userDetails' => $userDetails,
        //     'admin' => $admin
        // ], function ($message) use ($request, $booking) {
        //     $message->to('info@dachboxit.de')
        //             ->subject('Zahlungslink für Ihre Buchung bei Dachboxmiete – 24 Stunden gültig');
        // });

      

        return redirect()->route('admin.booking.index')
        ->with('success', 'Booking created successfully. Payment link has been sent to the customer.');
    }

    public function remove(Request $request)
{
    
        $user = \Illuminate\Support\Facades\Session::get('guest_user');
    

    try {
        $item = CartItem::where('id', $request->item_id)
            ->where('user_id', $user->id)
            ->firstOrFail();

        // delete all children too

        // delete parent item
        $item->delete();

        return response()->json(['success' => true]);
    } catch (\Exception $e) {
        return response()->json(['success' => false], 500);
    }
}


}
