<?php

namespace App\Http\Controllers\Admin\Bundle;

use App\Http\Controllers\Controller;
use App\Models\Vendor;
use Illuminate\Http\Request;
use Exception;
use Illuminate\Support\Facades\Validator;
use PDF;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;


use App\Imports\BundleFileImport;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\BundleFileExport;
use App\Exports\BundleReportExport;

use App\Models\Bundle;

class BundleController extends Controller
{
    

    public function index(Request $request)
    {
        $timeFilter = $request->get('time_filter', 'year');
        $search = $request->get('search');
        $roofType = $request->get('roof_type');
        
        $roofTypes = Bundle::select('rooftype_eng')
                         ->distinct()
                         ->pluck('rooftype_eng')
                         ->filter()
                         ->toArray();

        $bundles = Bundle::query()
            ->when($search, function($query) use ($search) {
                $query->where(function($q) use ($search) {
                    $q->where('rooftype_eng', 'like', "%$search%")
                      ->orWhere('car_make', 'like', "%$search%")
                      ->orWhere('bundle_name', 'like', "%$search%")
                      ->orWhere('model', 'like', "%$search%");
                });
            })
            ->when($roofType, function($query) use ($roofType) {
                $query->where('rooftype_eng', $roofType);
            })
            ->orderBy('created_at', 'asc')
            ->get();

        return view('admin.bundle.index', compact('bundles', 'roofTypes'));
    }

   

    public function uploadCSV(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls,csv'
        ]);

        Bundle::truncate();

        Excel::import(new BundleFileImport, $request->file('file'));

        return back()->with('success', 'Excel Imported Successfully!');
    }


    public function downloadCSV()
    {
        return Excel::download(new BundleFileExport, 'bundle.csv');
    }



    public function downloadReportCSV(Request $request)
    {
        $search = $request->get('search');
        $roofType = $request->get('roof_type');

        
        $bundles = Bundle::query()
            ->when($search, function($query) use ($search) {
                $query->where(function($q) use ($search) {
                    $q->where('rooftype_eng', 'like', "%$search%")
                      ->orWhere('car_make', 'like', "%$search%")
                      ->orWhere('bundle_name', 'like', "%$search%")
                      ->orWhere('model', 'like', "%$search%");
                });
            })
            ->when($roofType, function($query) use ($roofType) {
                $query->where('rooftype_eng', $roofType);
            })
            ->orderBy('created_at', 'desc')
            ->get();
        return Excel::download(new BundleReportExport($bundles), 'bundle.csv');
    }

    public function downloadReportPDF(Request $request)
    {
        set_time_limit(300);
        $search = $request->get('search');
        $roofType = $request->get('roof_type');
        
        $bundles = Bundle::query()
            ->when($search, function($query) use ($search) {
                $query->where(function($q) use ($search) {
                    $q->where('rooftype_eng', 'like', "%$search%")
                      ->orWhere('car_make', 'like', "%$search%")
                      ->orWhere('bundle_name', 'like', "%$search%")
                      ->orWhere('model', 'like', "%$search%");
                });
            })
            ->when($roofType, function($query) use ($roofType) {
                $query->where('rooftype_eng', $roofType);
            })
            ->orderBy('created_at', 'desc')
            ->get();


        $pdf = PDF::loadView('admin.bundle.pdf', compact('bundles'));
        return $pdf->download('bundles.pdf');
    }


    public function show(Bundle $bundle){
        return view('admin.bundle.details',compact('bundle'));
    }

    public function edit(Bundle $bundle){
        return view('admin.bundle.edit',compact('bundle'));
    }

    public function update(Request $request, Bundle $bundle)
    {
        $validator = Validator::make($request->all(), [
            'bundle_name' => 'required|string|max:100',
            'category' => 'required|string|max:100',
            'rooftype_eng' => 'required|string|max:100',
            'rooftype_ger' => 'required|string|max:100',
            'carrier_length' => 'required|string|max:50',
            'foot_sku' => 'required|string|max:50',
            'footkit_sku' => 'required|string|max:50',
            'car_make' => 'required|string|max:100',
            'model' => 'required|string|max:100',
            'type_year' => 'required|string|max:100',
            'recommended_roofbox_size' => 'nullable|string|max:100',
            'description' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $bundle->update($request->all());

        return redirect()->route('admin.bundle.index')->with('success', 'Bundle updated successfully!');
    }

   
}

