<?php

namespace App\Http\Controllers\Admin\Calender;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\WorkingHours;
use App\Models\DateAvailability;
use Carbon\Carbon;
use App\Models\Booking;
use App\Models\BookingItem;
use App\Models\BookingUserDetail;
use App\Models\CartItem;
use Illuminate\Support\Facades\Response;
use App\Models\Vendor;
use PDF;
use Illuminate\Support\Facades\DB;

use App\Models\ActivityLog;


class CalenderController extends Controller
{
    public $filePath = 'admin.calendar';

    public function index()
    {
        $parsedDate = now()->format('Y-m-d');
        // $bookings = Booking::with(['items', 'userDetails'])
        //         ->whereHas('items', function ($query) use ($parsedDate) {
        //             $query->where(function($q) use ($parsedDate) {
        //                 $q->whereDate('pickup_date', $parsedDate)
        //                 ->orWhereDate('dropoff_date', $parsedDate);
        //             });

                   
        //         })
        //         ->orderBy('created_at', 'desc')
        //         ->get()
        //         ->map(function($booking) use ($parsedDate) {
        //             $booking->setRelation('items', $booking->items->filter(function($item) use ($parsedDate) {
        //                 return Carbon::parse($item->pickup_date)->isSameDay($parsedDate) || 
        //                     Carbon::parse($item->dropoff_date)->isSameDay($parsedDate);
        //             }));
        //             return $booking;
        //         })
        //         ->filter(function($booking) {
        //             return $booking->items->isNotEmpty();
        //         });



        $bookings = Booking::with(['items', 'userDetails'])
            ->whereHas('items', function ($query) use ($parsedDate) {
                $query->where(function($q) use ($parsedDate) {
                    $q->whereDate('pickup_date', $parsedDate)
                    ->orWhere(function($q2) use ($parsedDate) {
                        $q2->where(function($q3) use ($parsedDate) {
                            $q3->whereNotNull('new_dropoff_date')
                                ->whereDate('new_dropoff_date', $parsedDate);
                        })
                        ->orWhere(function($q4) use ($parsedDate) {
                            $q4->whereNull('new_dropoff_date')
                                ->whereDate('dropoff_date', $parsedDate);
                        });
                    });
                });
            })
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function($booking) use ($parsedDate) {
                $booking->setRelation('items', $booking->items->filter(function($item) use ($parsedDate) {
                    $dropoff = $item->new_dropoff_date ?? $item->dropoff_date;
                    return Carbon::parse($item->pickup_date)->isSameDay($parsedDate) || 
                        Carbon::parse($dropoff)->isSameDay($parsedDate);
                }));
                return $booking;
            })
            ->filter(function($booking) {
                return $booking->items->isNotEmpty();
            });

        // Get unique product names for filter dropdown
        $uniqueProducts = BookingItem::select('product_name')
            ->distinct()
            ->orderBy('product_name')
            ->pluck('product_name');

        $vendors = Vendor::select('id', 'street', 'zip_code', 'city')
            ->where('status', 'active')
            ->get();

        $activityLogs = ActivityLog::where('action', 'appointment_change')->get();

        $activityCounts = [];
        
        foreach ($activityLogs as $log) {
            if (preg_match('/booking\s+#(\d+)/i', $log->description, $matches)) {
                $bookingId = $matches[1];
                if (!isset($activityCounts[$bookingId])) {
                    $activityCounts[$bookingId] = 0;
                }
                $activityCounts[$bookingId]++;
            }
        }
    

        return view($this->filePath.'.index', compact('bookings', 'uniqueProducts','vendors','activityCounts'));
    }

    public function getAvailabilityByDate($date)
    {
        $adminId = Auth::guard('admin')->id();
        $date = Carbon::parse($date)->format('Y-m-d');

        $availability = DateAvailability::where('admin_id', $adminId)
                        ->where('date', $date)
                        ->get();

        return response()->json($availability);
    }

    public function storeAvailability(Request $request)
    {
        $request->validate([
            'date' => 'required|date',
            'slots' => 'required|array',
            'slots.*.start_time' => 'required|date_format:H:i',
            'slots.*.end_time' => 'required|date_format:H:i|after:slots.*.start_time',
        ]);

        $adminId = Auth::guard('admin')->id();
        $date = Carbon::parse($request->date)->format('Y-m-d');

        DateAvailability::where('admin_id', $adminId)
            ->where('date', $date)
            ->delete();

        foreach ($request->slots as $slot) {
            DateAvailability::create([
                'admin_id' => $adminId,
                'date' => $date,
                'start_time' => $slot['start_time'],
                'end_time' => $slot['end_time'],
                'is_closed' => false
            ]);
        }

        return response()->json(['status' => 'success']);
    }

    public function closeDate(Request $request)
    {
        $request->validate([
            'date' => 'required|date'
        ]);

        $adminId = Auth::guard('admin')->id();
        $date = Carbon::parse($request->date)->format('Y-m-d');

        DateAvailability::where('admin_id', $adminId)
            ->where('date', $date)
            ->delete();

        DateAvailability::create([
            'admin_id' => $adminId,
            'date' => $date,
            'start_time' => '00:00:00',
            'end_time' => '00:00:00',
            'is_closed' => true
        ]);

        return response()->json(['status' => 'success']);
    }

    public function deleteSlot($id)
    {
        try {
            $adminId = Auth::guard('admin')->id();
            
            $slot = DateAvailability::where('admin_id', $adminId)
                    ->where('id', $id)
                    ->firstOrFail();
                    
            $slot->delete();
            
            return response()->json([
                'status' => 'success',
                'message' => 'Slot deleted successfully'
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to delete slot'
            ], 500);
        }
    }

    public function getBookingsByDate($date)
    {
        try {
            $parsedDate = Carbon::parse($date)->toDateString();
            $productFilter = request()->query('product');
            $vendorFilter = request()->query('vendor_id');
            // $bookings = Booking::with(['items', 'userDetails'])
            //     ->whereHas('items', function ($query) use ($parsedDate, $productFilter,$vendorFilter) {
            //         $query->where(function($q) use ($parsedDate) {
            //             $q->whereDate('pickup_date', $parsedDate)
            //             ->orWhereDate('dropoff_date', $parsedDate);
            //         });

            //         if ($productFilter && $productFilter !== 'all') {
            //             $query->where('product_name', 'like', '%'.$productFilter.'%');
            //         }
            //         if ($vendorFilter) {
            //             $query->where('vendor_id', $vendorFilter);
            //         }else{
            //             $query->where('vendor_id', null);
            //         }
            //     })
            //     ->orderBy('created_at', 'desc')
            //     ->get()
            //     ->map(function ($booking) use ($parsedDate, $vendorFilter) {
            //         $booking->setRelation('items', $booking->items->filter(function ($item) use ($parsedDate, $vendorFilter) {
            //             $matchesDate = Carbon::parse($item->pickup_date)->isSameDay($parsedDate)
            //                 || Carbon::parse($item->dropoff_date)->isSameDay($parsedDate);
    
            //                 $matchesVendor = $vendorFilter ? $item->vendor_id == $vendorFilter : $item->vendor_id == null;
    
            //             return $matchesDate && $matchesVendor;
            //         }));
            //         return $booking;
            //     })
            //     ->filter(function ($booking) {
            //         return $booking->items->isNotEmpty(); // Only keep bookings with matched items
            //     });
            $bookings = Booking::with(['items', 'userDetails'])
                ->whereHas('items', function ($query) use ($parsedDate, $productFilter, $vendorFilter) {
                    $query->where(function ($q) use ($parsedDate) {
                        $q->whereDate('pickup_date', $parsedDate)
                        ->orWhere(function ($q2) use ($parsedDate) {
                            $q2->whereNotNull('new_dropoff_date')
                                ->whereDate('new_dropoff_date', $parsedDate);
                        })
                        ->orWhere(function ($q3) use ($parsedDate) {
                            $q3->whereNull('new_dropoff_date')
                                ->whereDate('dropoff_date', $parsedDate);
                        });
                    });
            
                    if ($productFilter && $productFilter !== 'all') {
                        $query->where('product_name', 'like', '%'.$productFilter.'%');
                    }
            
                    if ($vendorFilter) {
                        $query->where('vendor_id', $vendorFilter);
                    } else {
                        $query->where('vendor_id', null);
                    }
                })
                ->orderBy('created_at', 'desc')
                ->get()
                ->map(function ($booking) use ($parsedDate, $vendorFilter) {
                    $booking->setRelation('items', $booking->items->filter(function ($item) use ($parsedDate, $vendorFilter) {
                        
                        $matchesDate = Carbon::parse($item->pickup_date)->isSameDay($parsedDate)
                            || (isset($item->new_dropoff_date) && Carbon::parse($item->new_dropoff_date)->isSameDay($parsedDate))
                            || (!isset($item->new_dropoff_date) && Carbon::parse($item->dropoff_date)->isSameDay($parsedDate));
            
                        $matchesVendor = $vendorFilter ? $item->vendor_id == $vendorFilter : $item->vendor_id == null;
            
                        return $matchesDate && $matchesVendor;
                    }));
                    return $booking;
                })
                ->filter(function ($booking) {
                    return $booking->items->isNotEmpty();
                });

                $activityLogs = ActivityLog::where('action', 'appointment_change')->get();

                $activityCounts = [];
                
                foreach ($activityLogs as $log) {
                    if (preg_match('/booking\s+#(\d+)/i', $log->description, $matches)) {
                        $bookingId = $matches[1];
                        if (!isset($activityCounts[$bookingId])) {
                            $activityCounts[$bookingId] = 0;
                        }
                        $activityCounts[$bookingId]++;
                    }
                }

           

            $html = view('admin.calendar.partials.booking_table', compact('bookings','activityCounts'))->render();

            return response()->json([
                'success' => true,
                'html' => $html
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function getUniqueProducts()
    {
        $products = BookingItem::select('product_name')
            ->distinct()
            ->orderBy('product_name')
            ->pluck('product_name');

        return response()->json($products);
    }

    public function downloadPdf($date)
    {
        $parsedDate = Carbon::parse($date)->toDateString();
        
        $productFilter = request()->query('product');
        $vendorFilter = request()->query('vendor_id');
        $bookings = Booking::with(['items', 'userDetails'])
        ->whereHas('items', function ($query) use ($parsedDate, $productFilter, $vendorFilter) {
            $query->where(function ($q) use ($parsedDate) {
                $q->whereDate('pickup_date', $parsedDate)
                ->orWhere(function ($q2) use ($parsedDate) {
                    $q2->whereNotNull('new_dropoff_date')
                        ->whereDate('new_dropoff_date', $parsedDate);
                })
                ->orWhere(function ($q3) use ($parsedDate) {
                    $q3->whereNull('new_dropoff_date')
                        ->whereDate('dropoff_date', $parsedDate);
                });
            });
    
            if ($productFilter && $productFilter !== 'all') {
                $query->where('product_name', 'like', '%'.$productFilter.'%');
            }
    
            if ($vendorFilter) {
                $query->where('vendor_id', $vendorFilter);
            } else {
                $query->where('vendor_id', null);
            }
        })
        ->orderBy('created_at', 'desc')
        ->get()
        ->map(function ($booking) use ($parsedDate, $vendorFilter) {
            $booking->setRelation('items', $booking->items->filter(function ($item) use ($parsedDate, $vendorFilter) {
                
                $matchesDate = Carbon::parse($item->pickup_date)->isSameDay($parsedDate)
                    || (isset($item->new_dropoff_date) && Carbon::parse($item->new_dropoff_date)->isSameDay($parsedDate))
                    || (!isset($item->new_dropoff_date) && Carbon::parse($item->dropoff_date)->isSameDay($parsedDate));
    
                $matchesVendor = $vendorFilter ? $item->vendor_id == $vendorFilter : $item->vendor_id == null;
    
                return $matchesDate && $matchesVendor;
            }));
            return $booking;
        })
        ->filter(function ($booking) {
            return $booking->items->isNotEmpty();
        });

        $activityLogs = ActivityLog::where('action', 'appointment_change')->get();

        $activityCounts = [];
        
        foreach ($activityLogs as $log) {
            if (preg_match('/booking\s+#(\d+)/i', $log->description, $matches)) {
                $bookingId = $matches[1];
                if (!isset($activityCounts[$bookingId])) {
                    $activityCounts[$bookingId] = 0;
                }
                $activityCounts[$bookingId]++;
            }
        }

        $pdf = PDF::loadView('admin.calendar.pdf', [
            'bookings' => $bookings,
            'date' => $date,
            'activityCounts' =>$activityCounts
        ]);

        return $pdf->download('Calendar Report -'.$date.'.pdf');
    }

    public function downloadCsv($date)
    {
        $parsedDate = Carbon::parse($date)->toDateString();
        $productFilter = request()->query('product');
        $vendorFilter = request()->query('vendor_id');
        $bookings = Booking::with(['items', 'userDetails'])
        ->whereHas('items', function ($query) use ($parsedDate, $productFilter, $vendorFilter) {
            $query->where(function ($q) use ($parsedDate) {
                $q->whereDate('pickup_date', $parsedDate)
                ->orWhere(function ($q2) use ($parsedDate) {
                    $q2->whereNotNull('new_dropoff_date')
                        ->whereDate('new_dropoff_date', $parsedDate);
                })
                ->orWhere(function ($q3) use ($parsedDate) {
                    $q3->whereNull('new_dropoff_date')
                        ->whereDate('dropoff_date', $parsedDate);
                });
            });
    
            if ($productFilter && $productFilter !== 'all') {
                $query->where('product_name', 'like', '%'.$productFilter.'%');
            }
    
            if ($vendorFilter) {
                $query->where('vendor_id', $vendorFilter);
            } else {
                $query->where('vendor_id', null);
            }
        })
        ->orderBy('created_at', 'desc')
        ->get()
        ->map(function ($booking) use ($parsedDate, $vendorFilter) {
            $booking->setRelation('items', $booking->items->filter(function ($item) use ($parsedDate, $vendorFilter) {
                
                $matchesDate = Carbon::parse($item->pickup_date)->isSameDay($parsedDate)
                    || (isset($item->new_dropoff_date) && Carbon::parse($item->new_dropoff_date)->isSameDay($parsedDate))
                    || (!isset($item->new_dropoff_date) && Carbon::parse($item->dropoff_date)->isSameDay($parsedDate));
    
                $matchesVendor = $vendorFilter ? $item->vendor_id == $vendorFilter : $item->vendor_id == null;
    
                return $matchesDate && $matchesVendor;
            }));
            return $booking;
        })
        ->filter(function ($booking) {
            return $booking->items->isNotEmpty();
        });
    
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="Calendar Report'.$date.'.csv"',
        ];

        $activityLogs = ActivityLog::where('action', 'appointment_change')->get();

        $activityCounts = [];
        $activityReasons = [];
    
        foreach ($activityLogs as $log) {
            if (preg_match('/booking\s+#(\d+)/i', $log->description, $matches)) {
                $bookingId = $matches[1];
    
                if (!isset($activityCounts[$bookingId])) {
                    $activityCounts[$bookingId] = 0;
                }
                $activityCounts[$bookingId]++;
    
                $activityReasons[$bookingId][] = $log->details;
            }
        }
    
    
        $callback = function() use ($bookings,$activityCounts, $activityReasons) {
            $file = fopen('php://output', 'w');
            
            // Header row
            fputcsv($file, [
                'Booking ID',
                
                'Client Name',
                'Phone Number',
                'Email',
                'No. Of Times Changed',
                'Reason',
                'Pick-Up Date/Time',
                'Drop-Off Date/Time',
                'Status'
            ]);
    
            // Data rows
            foreach ($bookings as $booking) {
                foreach ($booking->items as $item) {
                    if(!$item->related_item){
                        $reasons = isset($activityReasons[$booking->id])
                        ? implode("\n", $activityReasons[$booking->id])
                        : '-';
    
                        fputcsv($file, [
                            $booking->id,
                        
                            ($booking->userDetails->first_name ?? '') . ' ' . ($booking->userDetails->last_name ?? ''),
                            
                            preg_replace('/^(\+\d{0,2})(\d+)/', '$1 $2', $booking->userDetails->phone) ?? 'N/A',
                            $booking->userDetails->email,
                            $activityCounts[$booking->id] ?? 0,
                            $reasons,
                            $item->pickup_date ? Carbon::parse($item->pickup_date)->format('d.m.Y') . ' ' . 
                                ($item->pickup_time ? Carbon::parse($item->pickup_time)->format('H:i') : '') : '-',
                            $item->dropoff_date ? Carbon::parse($item->dropoff_date)->format('d.m.Y') . ' ' . 
                                ($item->dropoff_time ? Carbon::parse($item->dropoff_time)->format('H:i') : '') : '-',
                            ucfirst($booking->status)
                        ]);
                    }
                    
                }
            }
    
            fclose($file);
        };
    
        return response()->stream($callback, 200, $headers);
    }
}