<?php

namespace App\Http\Controllers\Admin\Payment;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Exports\PaymentsExport;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\Vendor;

use PDF;
class PaymentController extends Controller
{
    public $filePath='admin.payment';


    public function index(Request $request)
    {
        $bookings = Booking::with('userDetails')
            ->orderByDesc('created_at');
        
        // Apply time period filter if requested
        if ($request->has('time_period')) {
            $timePeriod = $request->input('time_period');
            
            switch ($timePeriod) {
                case 'year':
                    $bookings->whereYear('created_at', date('Y'));
                    break;
                case 'month':
                    $bookings->whereMonth('created_at', date('m'))
                            ->whereYear('created_at', date('Y'));
                    break;
                case 'week':
                    $bookings->whereBetween('created_at', [
                        now()->startOfWeek(),
                        now()->endOfWeek()
                    ]);
                    break;
                case 'day':
                    $bookings->whereDate('created_at', date('Y-m-d'));
                    break;
            }
        }
        $vendorId = $request->vendor_id;

        // If frontend sends 'null' as string, normalize it to null
        if ($vendorId === 'null') {
            $vendorId = null;
        }
        
        if (!is_null($vendorId)) {
            $bookings->whereHas('items', function($q) use ($vendorId) {
                $q->where('vendor_id', $vendorId);
            });
        } else {
            $bookings->whereHas('items', function($q) {
                $q->whereNull('vendor_id');
            });
        }
        
        $bookings = $bookings->get();
        $vendors = Vendor::select('id', 'street', 'zip_code', 'city')->where('status','active')->get();
        
        // Calculate analytics
        $totalPayments = $bookings->where('payment_status', 'paid')->sum('total');
        $totalRefunds = $bookings->where('payment_status', 'refund')->sum('refund_amount');
        $totalReconciliations = $bookings->where('payment_status', 'reconciliation')->sum('total');
        
        $paymentMethods = $bookings->where('payment_status', 'paid')
            ->groupBy('payment_method')
            ->map(function ($group) {
                return [
                    'count' => $group->count(),
                    'total' => $group->sum('total')
                ];
            });
        
        $paypalPercentage = $paymentMethods->get('paypal') 
            ? round(($paymentMethods['paypal']['count'] / $bookings->where('payment_status', 'paid')->count()) * 100, 2)
            : 0;
        
            $paidCount = $bookings->where('payment_status', 'paid')->count();

            $stripeCount = $paymentMethods->get('stripe')['count'] ?? 0;
            $cardCount = $paymentMethods->get('card')['count'] ?? 0;
            
            $creditCardCount = $stripeCount + $cardCount;
            
            $creditCardPercentage = $paidCount > 0 
                ? round(($creditCardCount / $paidCount) * 100, 2) 
                : 0;
        
        return view($this->filePath.'.index', compact(
            'bookings',
            'paypalPercentage',
            'creditCardPercentage',
            'totalRefunds',
            'totalReconciliations',
            'vendors'
        ));
    }

    public function downloadPdf(Request $request)
    {
        $bookings = $this->getFilteredBookings($request);
        
        $data = [
            'bookings' => $bookings,
            'title' => 'Payment Analytics Report'
        ];

        $pdf = PDF::loadView('admin.payment.pdf', $data)
            ->setPaper('a4', 'landscape');
        return $pdf->download('payment-analytics-report.pdf');
    }

    public function downloadCsv(Request $request)
    {
        $bookings = $this->getFilteredBookings($request);
        
        return Excel::download(new PaymentsExport($bookings), 
            'payments-analytics-'.now()->format('d.m.Y').'.csv', 
            \Maatwebsite\Excel\Excel::CSV, [
                'Content-Type' => 'text/csv',
            ]);
    }

    private function getFilteredBookings(Request $request)
    {
        $bookings = Booking::with('userDetails');
        
        // Apply time period filter if requested
        if ($request->has('time_period')) {
            $timePeriod = $request->input('time_period');
            
            switch ($timePeriod) {
                case 'year':
                    $bookings->whereYear('created_at', date('Y'));
                    break;
                case 'month':
                    $bookings->whereMonth('created_at', date('m'))
                            ->whereYear('created_at', date('Y'));
                    break;
                case 'week':
                    $bookings->whereBetween('created_at', [
                        now()->startOfWeek(),
                        now()->endOfWeek()
                    ]);
                    break;
                case 'day':
                    $bookings->whereDate('created_at', date('Y-m-d'));
                    break;
            }
        }
       
        $vendorId = $request->vendor_id;

        // If frontend sends 'null' as string, normalize it to null
        if ($vendorId === 'null') {
            $vendorId = null;
        }
        
        if (!is_null($vendorId)) {
            $bookings->whereHas('items', function($q) use ($vendorId) {
                $q->where('vendor_id', $vendorId);
            });
        } else {
            $bookings->whereHas('items', function($q) {
                $q->whereNull('vendor_id');
            });
        }
        
        // Apply status filter if requested
        if ($request->has('status') && $request->status !== 'all') {
            $bookings->where('payment_status', $request->status);
        }
        
        // Apply search filter if requested
        if ($request->has('search')) {
            $searchQuery = $request->input('search');
            $bookings->whereHas('userDetails', function ($query) use ($searchQuery) {
                $query->where('first_name', 'like', "%{$searchQuery}%")
                    ->orWhere('last_name', 'like', "%{$searchQuery}%")
                    ->orWhere('email', 'like', "%{$searchQuery}%")
                    ->orWhere('address', 'like', "%{$searchQuery}%");
            });
        }
        
        return $bookings->orderByDesc('created_at')->get();
    }

    public function viewPaymentHistory(Request $request)
    {
        $booking = Booking::with(['userDetails','items'])->where('id', $request->id)->first();
        return view($this->filePath.'.view_payment_history', compact('booking'));
    }
  

    public function updatePaymentStatus(Request $request, Booking $booking)
    {
        $validStatuses = ['paid', 'unpaid', 'refund', 'reconciliation'];

        $rules = [
            'payment_status' => ['required', 'string', Rule::in($validStatuses)],
        ];

        if ($request->payment_status === 'refund') {
            $rules['refund_amount'] = ['required', 'numeric', 'min:0'];
        }

        $validated = $request->validate($rules);

        $updateData = ['payment_status' => $request->payment_status];


        $previousStatus = $booking->status;
        if ($request->payment_status == 'paid' && $previousStatus == 'cancelled') {
        
            foreach ($booking->items as $item) {
                $pickupDate = date('Y-m-d', strtotime($item->pickup_date));
                $dropoffDate = date('Y-m-d', strtotime($item->dropoff_date));
                $pickupDateTime = $pickupDate . ' ' . $item->pickup_time;
                $dropoffDateTime = $dropoffDate . ' ' . $item->dropoff_time;
        
                $newDropoffDateTime = null;
                if ($item->new_dropoff_date && $item->new_dropoff_time) {
                    $newDropoffDate = date('Y-m-d', strtotime($item->new_dropoff_date));
                    $newDropoffDateTime = $newDropoffDate . ' ' . $item->new_dropoff_time;
                }
        
    
                $timesToCheck = array_filter([$pickupDateTime, $dropoffDateTime, $newDropoffDateTime]);
    
                $conflict = \App\Models\BookingItem::whereHas('booking', function ($q) {
                    $q->where('status', 'confirmed');
                    if (!empty($booking->vendor_id)) {
                        $q->where('vendor_id', $booking->vendor_id);
                    }
                })
    
               
                ->where('id', '!=', $item->id)
                ->where(function ($q) use ($timesToCheck) {
                    foreach ($timesToCheck as $datetime) {
                        $date = date('Y-m-d', strtotime($datetime));
                        $time = date('H:i:s', strtotime($datetime));
    
                        $q->orWhere(function ($sub) use ($date, $time) {
                            $sub->where(function ($x) use ($date, $time) {
                                $x->where('pickup_date', $date)->where('pickup_time', $time);
                            })
                            ->orWhere(function ($x) use ($date, $time) {
                                $x->where('dropoff_date', $date)->where('dropoff_time', $time);
                            })
                            ->orWhere(function ($x) use ($date, $time) {
                                $x->where('new_dropoff_date', $date)->where('new_dropoff_time', $time);
                            });
                        });
                    }
                })
                ->first();
                if ($conflict) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Cannot confirm this booking — another confirmed booking (ID: ' . $conflict->booking_id . ') already uses one or more of these date/time slots.'
                    ], 409);
                }
            }
        }

        if ($request->payment_status === 'refund') {
            $updateData['refund_amount'] = $request->refund_amount;
        } else {
            $updateData['refund_amount'] = null; // clear any previous refund
        }


        if($request->payment_status == 'paid'){
            $booking->update(['status' => 'confirmed' ] );
        }else if($request->payment_status == 'unpaid'){
            $booking->update(['status' => 'cancelled'] );
        }else if($request->payment_status == 'refund'){
            $booking->update(['status' => 'cancelled'] );
        }else if($request->payment_status == 'reconciliation'){
            $booking->update(['status' => 'pending'] );
        }

        $booking->update($updateData);

        return response()->json([
            'success' => true,
            'message' => 'Payment status updated successfully'
        ]);
    }

    public function searchPaymentUser(Request $request){
        
            $validated = $request->validate([
                'search' => 'nullable|string|max:255',
            ]);
            $searchQuery = $request->input('search', '');
                $bookings = Booking::with('userDetails');
                if ($searchQuery) {
                    $bookings = $bookings->whereHas('userDetails', function ($query) use ($searchQuery) {
                        $query->where('first_name', 'like', "%{$searchQuery}%")
                              ->orWhere('last_name', 'like', "%{$searchQuery}%")
                              ->orWhere('email', 'like', "%{$searchQuery}%")
                              ->orWhere('address', 'like', "%{$searchQuery}%");
                    });
                }
            $bookings = $bookings->orderByDesc('created_at')->get(); 
            $count = $bookings->count();
            return response()->json([
                'html' => view('admin.payment.search_payment_user', compact('bookings', 'count'))->render()
         ]);
        
    }
}
