<?php

namespace App\Http\Controllers\Admin\Product;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ProductBrand;
use App\Models\Category;
use App\Models\Size;
use App\Models\Color;
use App\Models\Variation;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;
use App\Models\ProductPrice;
use App\Models\Product;
use App\Models\ProductVariation;
use App\Models\ProductVariationOption;
use App\Models\RelatedProductVariation;
use App\Models\ProductImage;
use App\Models\ProductModel;
use App\Models\ProductOtherInfo;
use Illuminate\Support\Facades\Storage;
use App\Exports\ProductsExport;
use Maatwebsite\Excel\Facades\Excel;
use PDF;
use League\Csv\Reader;
use Illuminate\Support\Facades\DB;
use League\Csv\Writer;
use App\Models\Bundle;
use App\Models\BookingItem;


class ProductController extends Controller
{
    public $filePath = 'admin.product';


    // public function index(Request $request)
    // {
    //     $products = Product::with([
    //         'category',
    //         'variations.options.color',
    //         'variations.size',
    //         'variations.variation',
    //     ]);

    //     if ($request->has('time_period')) {
    //         $timePeriod = $request->input('time_period');
            
    //         switch ($timePeriod) {
    //             case 'year':
    //                 $products->whereYear('created_at', date('Y'));
    //                 break;
    //             case 'month':
    //                 $products->whereMonth('created_at', date('m'))
    //                         ->whereYear('created_at', date('Y'));
    //                 break;
    //             case 'week':
    //                 $products->whereBetween('created_at', [
    //                     now()->startOfWeek(),
    //                     now()->endOfWeek()
    //                 ]);
    //                 break;
    //             case 'day':
    //                 $products->whereDate('created_at', date('Y-m-d'));
    //                 break;
                    
    //         }
    //     }

    //     $products = $products->get();

    //     $totalProducts =$products->flatMap(function ($product) {
    //         return $product->variations->where('status', 'in_stock');
    //     })->count();
    //     $inactiveProducts = $products->flatMap(function ($product) {
    //         return $product->variations->where('status', '!=', 'in_stock');
    //     })->count();
    //     $totalProductCategories = Category::count();

    //     $bundleProducts = Bundle::count();

    //     $availableProducts = $products->sum('quantity'); 
        
    //     // Group by category name
    //     $groupedByCategory = $products->groupBy(fn ($product) => $product->category->name ?? 'Uncategorized');
    
    //     return view($this->filePath . '.index', compact(
    //         'groupedByCategory',
    //         'totalProducts',
    //         'inactiveProducts',
    //         'totalProductCategories',
    //         'availableProducts',
    //         'bundleProducts'
    //     ));
    // }
    public function index(Request $request)
    {
        $products = Product::with([
            'category', 
            'variations.options.color', 
            'variations.size', 
            'variations.variation',
        ]);

        if ($request->has('time_period')) {
            $timePeriod = $request->input('time_period');
            switch ($timePeriod) {
                case 'year':
                    $products->whereYear('created_at', date('Y'));
                    break;
                case 'month':
                    $products->whereMonth('created_at', date('m'))
                        ->whereYear('created_at', date('Y'));
                    break;
                case 'week':
                    $products->whereBetween('created_at', [
                        now()->startOfWeek(),
                        now()->endOfWeek()
                    ]);
                    break;
                case 'day':
                    $products->whereDate('created_at', date('Y-m-d'));
                    break;
            }
        }

        $products = $products->get();
        
        // Get all rented items count
        $rentedCounts = $this->getRentedItemsCount();
        
        foreach ($products as $product) {
            if ($product->variations->isEmpty()) {
                $rentedCount = $rentedCounts['product'][$product->id] ?? 0;
                $product->total_quantity = $product->quantity - $rentedCount;
            } else {
                foreach ($product->variations as $variation) {
                    if (!$variation->options->isEmpty()) {
                        foreach ($variation->options as $option) {
                            $rentedCount = $rentedCounts['option'][$option->id] ?? 0;
                            $option->total_quantity = $option->quantity - $rentedCount;
                        }
                    } else {
                        $rentedCount = $rentedCounts['variation'][$variation->id] ?? 0;
                        $variation->total_quantity = $variation->quantity - $rentedCount;
                    }
                }
            }
        }
        $totalProducts = $products->flatMap(function ($product) {
            return $product->variations->where('status', 'in_stock');
        })->count();

        $inactiveProducts = $products->flatMap(function ($product) {
            return $product->variations->where('status', '!=', 'in_stock');
        })->count();

        $totalProductCategories = Category::count();
        $bundleProducts = Bundle::count();
        $availableProducts = $products->sum('quantity');

        // Group by category name
        $groupedByCategory = $products->groupBy(fn ($product) => $product->category->name ?? 'Uncategorized');

        // $order = [
        //     'Roof Box',
        //     'Rearbox',
        //     'Roof Bike Carrier',
        //     'Clutch Bike Rack',
        //     'Roof Rack',
        //     'Traverse',
        //     'Foot',
        //     'Footkit',
        // ];

        $order = [
            'Dachbox',
            'Heckbox',
            'Dachfahrradträger',
            'Kupplungsfahrradträger',
            'Dachträger',
            'Traverse',
            'Fuß',
            'Fußkit',
        ];

        $groupedByCategory = $groupedByCategory->sortBy(function ($products, $categoryName) use ($order) {
            $index = array_search($categoryName, $order);
            return $index !== false ? $index : PHP_INT_MAX; // categories not in list go to end
        });

        return view($this->filePath . '.index', compact(
            'groupedByCategory',
            'totalProducts',
            'inactiveProducts',
            'totalProductCategories',
            'availableProducts',
            'bundleProducts',
            'rentedCounts'
        ));
    }

    private function getRentedItemsCount()
    {
        $today = now()->format('Y-m-d');
        
        $rentedItems = BookingItem::where('damage', '!=', 1)
            ->where('already_booked', 0)
            ->where('vendor_id', null)
            ->whereHas('booking', function ($q) {
                $q->where('status', 'confirmed');
            })
            ->where(function($query) use ($today) {
                $query->where(function($q) use ($today) {
                    $q->where('pickup_date', '<=', $today)
                    ->where(function($q2) use ($today) {
                        $q2->where(function($q3) use ($today) {
                            $q3->whereNotNull('new_dropoff_date')
                                ->where('new_dropoff_date', '>=', $today);
                        })->orWhere(function($q4) use ($today) {
                            $q4->whereNull('new_dropoff_date')
                                ->where('dropoff_date', '>=', $today);
                        });
                    });
                });
            })
            ->get();

            



        $rentedCounts = [
            'product' => [],
            'variation' => [],
            'option' => []
        ];

        foreach ($rentedItems as $item) {
            // Count by product_id (for simple products like roof racks)
            if ($item->product_id) {
                if (!isset($rentedCounts['product'][$item->product_id])) {
                    $rentedCounts['product'][$item->product_id] = 0;
                }
                $rentedCounts['product'][$item->product_id]++;
            }
            

           
            if ($item->product_id && $item->size) {
                // Find the variation for this product with matching size
                $product = Product::find($item->product_id);
                if ($product && $product->variations) {
                    foreach ($product->variations as $variation) {
                        if (($variation->size && $variation->size->name == $item->size)  || ($variation->variation && $variation->variation->name == $item->size)) {
                           
                            if (!isset($rentedCounts['variation'][$variation->id])) {
                                $rentedCounts['variation'][$variation->id] = 0;
                            }
                            $rentedCounts['variation'][$variation->id]++;
                          

                            $name = strtolower($item->product_name);

                            if (in_array($name, ['carrier', 'foot', 'footkit', 'traverse', 'fuß', 'fußkit'])) {
                                $item->color = 'Schwarz';
                            }
                            // If there are options, find the matching one by color
                            if ($item->color && $variation->options) {
                                foreach ($variation->options as $option) {
                                    if ($option->color && $option->color->name == $item->color) {
                                        if (!isset($rentedCounts['option'][$option->id])) {
                                            $rentedCounts['option'][$option->id] = 0;
                                        }
                                        $rentedCounts['option'][$option->id]++;
                                        break;
                                    }
                                }
                            }
                            break;
                        }
                    }
                }
            }
        }

        return $rentedCounts;
    }



    // public function downloadPdf()
    // {
    //     $products = Product::with(['category', 'variations.options'])->get();
        
    //     $data = [
    //         'products' => $products,
    //         'title' => 'Product Analytics Report'
    //     ];
        
    //     $pdf = PDF::loadView('admin.product.pdf', $data);
    //     return $pdf->download('product-analytics-report.pdf');
    // }


    public function downloadPdf()
    {
        $products = Product::with(['category', 'variations.options.color', 'variations.size', 'variations.variation'])->get();

        $rentedCounts = $this->getRentedItemsCount();

        foreach ($products as $product) {
            if ($product->variations->isEmpty()) {
                $rentedCount = $rentedCounts['product'][$product->id] ?? 0;
                $product->total_quantity = $product->quantity - $rentedCount;
            } else {
                foreach ($product->variations as $variation) {
                    if (!$variation->options->isEmpty()) {
                        foreach ($variation->options as $option) {
                            $rentedCount = $rentedCounts['option'][$option->id] ?? 0;
                            $option->total_quantity = $option->quantity - $rentedCount;
                        }
                    } else {
                        $rentedCount = $rentedCounts['variation'][$variation->id] ?? 0;
                        $variation->total_quantity = $variation->quantity - $rentedCount;
                    }
                }
            }
        }

        $data = [
            'products' => $products,
            'title' => 'Product Analytics Report'
        ];

        $pdf = PDF::loadView('admin.product.pdf', $data);
        return $pdf->download('product-analytics-report.pdf');
    }

    public function downloadCsv()
    {
        return Excel::download(new ProductsExport(), 'products-analytics-'.now()->format('Y-m-d').'.csv', \Maatwebsite\Excel\Excel::CSV, [
            'Content-Type' => 'text/csv',
        ]);
    }




    public function create()
    {
        $brands = ProductBrand::all();
        $categories = Category::all();
        $sizes = Size::all();
        $colors = Color::all();
        $variations = Variation::all();
        $productVariations = ProductVariation::with(['product.mainPhoto', 'options'])->get();
        $products = Product::all();

       
        return view($this->filePath . '.add-product', compact('brands', 'categories', 'sizes', 'colors','variations' ,'productVariations','products'));
    }

    public function store(Request $request)
    {
      
        
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category_id' => 'required|exists:categories,id',
            'brand_id' => 'required|exists:product_brands,id',
            'keywords' => 'nullable|string',
            'variations' => 'sometimes|array',
            'variations.*.size_id' => 'sometimes|integer',
            'variations.*.colors' => 'sometimes|array',
            'variations.*.colors.*.color_id' => 'sometimes|exists:colors,id',
            'variations.*.colors.*.quantity' => 'sometimes|integer|min:0',
            'variations.*.colors.*.sku' => 'sometimes|string|unique:product_variation_options,sku',
            'variations.*.status' => 'sometimes|in:in_stock,out_of_stock,pre_order',
            'variations.*.weight' => 'nullable|numeric',
            'variations.*.status' => 'sometimes|in:in_stock,out_of_stock,pre_order',
            'variations.*.length' => 'nullable|numeric',
            'variations.*.width' => 'nullable|numeric',
            'variations.*.height' => 'nullable|numeric',
            'variations.*.volume' => 'nullable|numeric',
            'variations.*.max_load' => 'nullable|numeric',
            'weight' => 'nullable|numeric',
            'status' => 'nullable|in:in_stock,out_of_stock,pre_order',
            'length' => 'nullable|numeric',
            'width' => 'nullable|numeric',
            'height' => 'nullable|numeric',
            'volume' => 'nullable|numeric',
            'max_load' => 'nullable|numeric',
            'other_info' => 'nullable|array',
            'other_info.*.description' => 'nullable|string',
            'models' => 'nullable|array',
            'models.*.title' => 'nullable|string|max:255',
            'models.*.description' => 'nullable|string',
            'gallery_images' => 'nullable|array',
            'gallery_images.*' => 'image|mimes:jpeg,png,jpg,webp|max:2048',
            'main_image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'info_image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'home_image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'railing_type' => 'nullable|boolean',
        ]);


        if ($validator->fails()) {
            // dd($validator->errors()->toArray(),$request->all());
          
        }

        
    
        
        $validated = $validator->validated();
    
        $product = Product::create([
            'name' => $validated['name'],
            'description' => $validated['description'],
            'category_id' => $validated['category_id'],
            'brand_id' => $validated['brand_id'],
            'keywords' => $validated['keywords'],
            'railing_type' => $validated['railing_type'] ?? false,
            // Store product attributes directly for simple products
            'weight' => $validated['weight'] ?? null,
            'status' => $validated['status'] ?? null,
            'length' => $validated['length'] ?? null,
            'width' => $validated['width'] ?? null,
            'height' => $validated['height'] ?? null,
            'volume' => $validated['volume'] ?? null,
            'max_load' => $validated['max_load'] ?? null,
        ]);

        if (!empty($validated['variations'])) {
            $category = Category::findOrFail($validated['category_id']);
            $sizeIds = is_array($category->size_ids) ? $category->size_ids : json_decode($category->size_ids, true);
            $variationIds = is_array($category->variation_ids) ? $category->variation_ids : json_decode($category->variation_ids, true);
    
            foreach ($validated['variations'] as $variationData) {
                $id = $variationData['size_id']; 
    
                $isSize = in_array($id, $sizeIds ?? []);
                $isVariation = in_array($id, $variationIds ?? []);
               
                $variation = ProductVariation::create([
                    'product_id' => $product->id,
                    'size_id' => $isSize ? (int) $id : null,
                    'variation_id' => $isVariation ? (int)$id : null,
                    'weight' => $variationData['weight'] ?? null,
                    'status' => $variationData['status'] ?? null,
                    'length' => $variationData['length'] ?? null,
                    'width' => $variationData['width'] ?? null,
                    'height' => $variationData['height'] ?? null,
                    'volume' => $variationData['volume'] ?? null,
                    'max_load' => $variationData['max_load'] ?? null,
                ]);
    
                if (!empty($variationData['colors'])) {
                    foreach ($variationData['colors'] as $colorData) {
                        if (empty($colorData['color_id'])) {
                            continue;
                        }
    
                        ProductVariationOption::create([
                            'variation_id' => $variation->id,
                            'color_id' => $colorData['color_id'],
                            'quantity' => $colorData['quantity'],
                            'sku' => $colorData['sku'],
                        ]);
                    }
                }
            }
        }
        if ($request->has('related_products')) {
            foreach ($request->input('related_products') as $optionId) {
                RelatedProductVariation::create([
                    'product_id' => $product->id,
                    'variation_option_id' => $optionId,
                ]);
            }
        }
      

        if (!empty($validated['other_info'])) {
            foreach ($validated['other_info'] as $index => $infoData) {
               
        
                if ($request->hasFile("other_info.$index.image")) {
                    $imagePath = $request->file("other_info.$index.image")->store('products/other_info', 'public');
        
                    ProductOtherInfo::create([
                        'product_id' => $product->id,
                        'description' => $infoData['description'],
                        'image_path' => $imagePath,
                    ]);
                }
            }
        }
        if (!empty($validated['models'])) {
            foreach ($validated['models'] as $modelData) {
                ProductModel::create([
                    'product_id' => $product->id,
                    'title' => $modelData['title'],
                    'description' => $modelData['description'],
                ]);
            }
        }

        if ($request->hasFile('gallery_images')) {
            foreach ($request->file('gallery_images') as $image) {
                $path = $image->store('products/gallery', 'public');
                ProductImage::create([
                    'product_id' => $product->id,
                    'image_path' => $path,
                    'type' => 'gallery',
                ]);
            }
        }

        if ($request->hasFile('main_image')) {
            $path = $request->file('main_image')->store('products/main', 'public');
            ProductImage::create([
                'product_id' => $product->id,
                'image_path' => $path,
                'type' => 'main',
            ]);
        }

        if ($request->hasFile('info_image')) {
            $path = $request->file('info_image')->store('products/info', 'public');
            ProductImage::create([
                'product_id' => $product->id,
                'image_path' => $path,
                'type' => 'info',
            ]);
        }
        if($request->hasFile('home_image')){
            $path = $request->file('home_image')->store('products/home','public');
            ProductImage::create([
                'product_id' => $product->id,
                'image_path' => $path,
                'type' => 'home',
            ]);
        }

        return redirect()->route('admin.products.index')->with('success', 'Product created successfully');
    }

    // public function show($id)
    // {

    //     return view($this->filePath . '.product_detail');
    // }

    public function show($id)
    {
        $product = Product::with([
            'category',
            'brand',
            'images' ,
            'otherInfo',
            'models'
        ])->findOrFail($id);

        $productPrice = ProductPrice::where('product_name', $product->name)->first();

        $relatedOptions = ProductVariationOption::with(['variation.product.mainPhoto'])
        ->whereHas('variation', function ($q) use ($product) {
            $q->where('product_id', '!=', $product->id);
        })->get();
        $selectedOptionIds = RelatedProductVariation::where('product_id',  $product->id)    
            ->pluck('variation_option_id')
            ->toArray();

      

        return view($this->filePath . '.product_detail_without_variations', compact('product','relatedOptions','selectedOptionIds','productPrice'));
    }



    public function showVariant($productId, $variantId ,$optionId = null)
    {
        $variant = ProductVariation::with([
            'product.category',
            'product.brand',
            'options.color',
            'size',
            'variation',
            'product.images',
            'product.otherInfo',
            'product.models',
            
        ])->where('id', $variantId)
        ->where('product_id', $productId)
        ->firstOrFail();

        
        $product = $variant->product;
        $productPrice = ProductPrice::where('product_name', $product->name)->first();
        $relatedProducts = Product::with('homePhoto')
        ->whereIn('id', RelatedProductVariation::where('product_id', $product->id)
            ->pluck('variation_option_id')
        )
        ->get();
    
       

        

        $mainOption = null;
        if ($optionId) {
            $mainOption = $variant->options->firstWhere('id', $optionId);
        }
       
        return view($this->filePath . '.product_detail', [
            'product' => $variant->product,
            'variant' => $variant,
            'mainOption' => $mainOption,
            'relatedProducts' => $relatedProducts,
            'productPrice' => $productPrice

            
        ]);
    }

    public function quickEdit($id)
{
    // Use eager loading strategically to avoid memory issues
    $product = Product::with([
        'images' => function($query) {
            $query->whereIn('type', ['main', 'info', 'home', 'gallery']);
        },
        'otherInfo',
        'models',
        'category',
        'brand'
        // NOTE: We're NOT loading variations here
    ])->findOrFail($id);

    $categories = Category::all();
    $brands = ProductBrand::all();
    
    // Get related products (limit to avoid performance issues)
    $products = Product::with('mainPhoto')
        ->where('id', '!=', $id)
        ->limit(100)
        ->get();
    
    $selectedOptionIds = RelatedProductVariation::where('product_id', $id)
        ->pluck('variation_option_id')
        ->toArray();

    return view($this->filePath . '.quick-edit', compact(
        'product', 
        'categories', 
        'brands', 
        'products',
        'selectedOptionIds'
    ));
}

/**
 * Update product without touching variations
 */
public function quickUpdate(Request $request, $id)
{
    $product = Product::findOrFail($id);

    $rules = [
        'name' => 'required|string|max:255',
        'description' => 'nullable|string',
        'category_id' => 'required|exists:categories,id',
        'brand_id' => 'required|exists:product_brands,id',
        'keywords' => 'nullable|string',
        'railing_type' => 'nullable|boolean',
        'other_info' => 'nullable|array',
        'other_info.*.id' => 'nullable|exists:product_other_infos,id',
        'other_info.*.description' => 'nullable|string',
        'other_info.*.image' => 'nullable|image|mimes:jpeg,png,jpg,webp,svg|max:2048',
        'other_info.*.existing_image' => 'nullable',
        'models' => 'nullable|array',
        'models.*.title' => 'nullable|string|max:255',
        'models.*.description' => 'nullable|string',
        'gallery_images' => 'nullable|array',
        'gallery_images.*' => 'image|mimes:jpeg,png,jpg,webp|max:2048',
        'main_image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
        'info_image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
        'home_image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
    ];

    // Only include these fields for non-variant products
    if ($product->variations->isEmpty()) {
        $rules['weight'] = 'nullable|numeric';
        $rules['status'] = 'required|in:in_stock,out_of_stock,pre_order';
        $rules['length'] = 'nullable|numeric';
        $rules['width'] = 'nullable|numeric';
        $rules['height'] = 'nullable|numeric';
        $rules['volume'] = 'nullable|numeric';
        $rules['max_load'] = 'nullable|numeric';
    }

    $validator = Validator::make($request->all(), $rules);

    if ($validator->fails()) {
        \Log::error('Quick update validation failed', [
            'errors' => $validator->errors()->toArray()
        ]);
        
        return back()
            ->withErrors($validator)
            ->withInput();
    }

    $validated = $validator->validated();

    try {
        DB::beginTransaction();

        // Update basic product info
        $updateData = [
            'name' => $validated['name'],
            'description' => $validated['description'],
            'category_id' => $validated['category_id'],
            'brand_id' => $validated['brand_id'],
            'keywords' => $validated['keywords'] ?? null,
            'railing_type' => $validated['railing_type'] ?? false,
        ];

        // Only update these fields for non-variant products
        if ($product->variations->isEmpty()) {
            $updateData['weight'] = $validated['weight'] ?? null;
            $updateData['status'] = $validated['status'] ?? 'in_stock';
            $updateData['length'] = $validated['length'] ?? null;
            $updateData['width'] = $validated['width'] ?? null;
            $updateData['height'] = $validated['height'] ?? null;
            $updateData['volume'] = $validated['volume'] ?? null;
            $updateData['max_load'] = $validated['max_load'] ?? null;
        }

        $product->update($updateData);

        // Handle removed images
        if ($request->has('removed_images')) {
            foreach ($request->removed_images as $imageId) {
                $image = ProductImage::find($imageId);
                if ($image) {
                    Storage::delete('public/' . $image->image_path);
                    $image->delete();
                }
            }
        }

        // Handle other info
        $existingInfoIds = [];
        if (!empty($validated['other_info'])) {
            foreach ($validated['other_info'] as $infoData) {
                $imagePath = $infoData['existing_image'] ?? null;

                if (isset($infoData['image'])) {
                    if ($imagePath) {
                        Storage::delete('public/' . $imagePath);
                    }
                    $imagePath = $infoData['image']->store('products/other_info', 'public');
                }

                $info = $product->otherInfo()->updateOrCreate(
                    ['id' => $infoData['id'] ?? null],
                    [
                        'description' => $infoData['description'],
                        'image_path' => $imagePath,
                    ]
                );
                $existingInfoIds[] = $info->id;
            }
        }
        $product->otherInfo()->whereNotIn('id', $existingInfoIds)->delete();

        // Handle models
        $existingModelIds = [];
        if (!empty($validated['models'])) {
            foreach ($validated['models'] as $modelData) {
                $model = $product->models()->updateOrCreate(
                    ['id' => $modelData['id'] ?? null],
                    [
                        'title' => $modelData['title'],
                        'description' => $modelData['description'],
                    ]
                );
                $existingModelIds[] = $model->id;
            }
        }
        $product->models()->whereNotIn('id', $existingModelIds)->delete();

        // Handle gallery images
        if ($request->hasFile('gallery_images')) {
            foreach ($request->file('gallery_images') as $image) {
                $path = $image->store('products/gallery', 'public');
                ProductImage::create([
                    'product_id' => $product->id,
                    'image_path' => $path,
                    'type' => 'gallery',
                ]);
            }
        }

        // Handle related products
        RelatedProductVariation::where('product_id', $product->id)->delete();
        if ($request->has('related_products')) {
            $relatedOptions = $request->input('related_products');
            $insertData = [];
            foreach ($relatedOptions as $optionId) {
                $insertData[] = [
                    'product_id' => $product->id,
                    'variation_option_id' => $optionId,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }
            RelatedProductVariation::insert($insertData);
        }

        // Handle main image
        if ($request->hasFile('main_image')) {
            $oldMainImage = $product->images()->where('type', 'main')->first();
            if ($oldMainImage) {
                Storage::delete('public/' . $oldMainImage->image_path);
                $oldMainImage->delete();
            }
            $path = $request->file('main_image')->store('products/main', 'public');
            ProductImage::create([
                'product_id' => $product->id,
                'image_path' => $path,
                'type' => 'main',
            ]);
        }

        // Handle info image
        if ($request->hasFile('info_image')) {
            $oldInfoImage = $product->images()->where('type', 'info')->first();
            if ($oldInfoImage) {
                Storage::delete('public/' . $oldInfoImage->image_path);
                $oldInfoImage->delete();
            }
            $path = $request->file('info_image')->store('products/info', 'public');
            ProductImage::create([
                'product_id' => $product->id,
                'image_path' => $path,
                'type' => 'info',
            ]);
        }

        // Handle home image
        if ($request->hasFile('home_image')) {
            $oldHomeImage = $product->images()->where('type', 'home')->first();
            if ($oldHomeImage) {
                Storage::delete('public/' . $oldHomeImage->image_path);
                $oldHomeImage->delete();
            }
            $path = $request->file('home_image')->store('products/home', 'public');
            ProductImage::create([
                'product_id' => $product->id,
                'image_path' => $path,
                'type' => 'home',
            ]);
        }

        DB::commit();

        return redirect()->route('admin.products.index')
            ->with('success', 'Product updated successfully (variations unchanged)');

    } catch (\Exception $e) {
        DB::rollBack();
        \Log::error('Quick update failed: ' . $e->getMessage());
        
        return back()
            ->with('error', 'Failed to update product: ' . $e->getMessage())
            ->withInput();
    }
}

    

    public function edit($id)
{
    $product = Product::with([
        'category',
        'brand',
        'variations.options.color',
        'variations',
        'images',
        'models',
        'otherInfo'
    ])->findOrFail($id);

    $categories = Category::all();
    $brands = ProductBrand::all();

    $category = $product->category;
    

    $sizeIds = is_array($category->size_ids) ? $category->size_ids : json_decode($category->size_ids, true);
    $variationIds = is_array($category->variation_ids) ? $category->variation_ids : json_decode($category->variation_ids, true);
    $colorIds = is_array($category->color_ids) ? $category->color_ids : json_decode($category->color_ids, true);

    $sizes = $sizeIds ? Size::whereIn('id', $sizeIds)->get() : collect();
    $variations = $variationIds ? Variation::whereIn('id', $variationIds)->get() : collect();
    $colors = $colorIds ? Color::whereIn('id', $colorIds)->get() : collect();

    $relatedOptions = ProductVariationOption::with(['variation.product.mainPhoto'])
    ->whereHas('variation', function ($q) use ($product) {
        $q->where('product_id', '!=', $product->id);
    })->get();

    $products = Product::with('mainPhoto')
    ->where('id', '!=', $product->id)
    ->get();

    $selectedOptionIds = RelatedProductVariation::where('product_id', $product->id)
        ->pluck('variation_option_id')
        ->toArray();
    return view($this->filePath . '.edit_product_detail', compact(
        'product', 'categories', 'brands', 'sizes', 'variations', 'colors',
        'relatedOptions', 'selectedOptionIds' , 'products'
    ));
}

    public function update(Request $request, $id)
    {
      
        $product = Product::with(['variations', 'images', 'models', 'otherInfo'])->findOrFail($id);

        $rules = [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category_id' => 'required|exists:categories,id',
            'brand_id' => 'required|exists:product_brands,id',
            'keywords' => 'nullable|string',
            'variations' => 'sometimes|array',
            'variations.*.id' => 'nullable',
            'variations.*.size_id' => 'sometimes|integer',
            'variations.*.colors' => 'sometimes|array',
            'variations.*.weight' => 'nullable|numeric',
            'variations.*.status' => 'sometimes|in:in_stock,out_of_stock,pre_order',
            'variations.*.length' => 'nullable|numeric',
            'variations.*.width' => 'nullable|numeric',
            'variations.*.height' => 'nullable|numeric',
            'variations.*.volume' => 'nullable|numeric',
            'variations.*.max_load' => 'nullable|numeric',
            'other_info' => 'nullable|array',
            'other_info.*.id' => 'nullable|exists:product_other_infos,id',
            'other_info.*.description' => 'nullable|string',
            'other_info.*.image' => 'nullable|image|mimes:jpeg,png,jpg,webp,svg|max:2048',
            'other_info.*.existing_image' => 'nullable',
            'models' => 'nullable|array',
            'models.*.title' => 'nullable|string|max:255',
            'models.*.description' => 'nullable|string',
            'gallery_images' => 'nullable|array',
            'gallery_images.*' => 'image|mimes:jpeg,png,jpg,webp|max:2048',
            'main_image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'info_image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'home_image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:2048',
            'variations.*.colors.*.color_id' => 'sometimes|exists:colors,id',
            'variations.*.colors.*.quantity' => 'sometimes|numeric',
            'variations.*.colors.*.sku' => 'sometimes|string',
            'variations.*.colors.*.id' => 'nullable',
            'weight' => 'nullable|numeric',
            'status' => 'sometimes|in:in_stock,out_of_stock,pre_order',
            'length' => 'nullable|numeric',
            'width' => 'nullable|numeric',
            'height' => 'nullable|numeric',
            'volume' => 'nullable|numeric',
            'max_load' => 'nullable|numeric',
            'railing_type' => 'nullable|boolean',
        ];
    
        $validator = Validator::make($request->all(), $rules);
    
        if($request->input('variations', [])){

        $validator->after(function ($validator) use ($request, $product) {
            foreach ($request->input('variations', []) as $i => $variation) {
                $colors = is_array($variation['colors'] ?? []) ? array_values($variation['colors']) : [];
                
                foreach ($colors as $j => $color) {
                    $colorId = $color['color_id'] ?? null;
                    $optionId = $color['id'] ?? null;
                    $sku = $color['sku'] ?? null;
                    
                    if (empty($sku)) {
                        $validator->errors()->add("variations.$i.colors.$j.sku", 'The SKU is required.');
                        continue;
                    }
        
                    if ($optionId && is_numeric($optionId)) {
                        $existingOption = ProductVariationOption::find($optionId);
                        if ($existingOption && $existingOption->sku === $sku) {
                            continue;
                        }
                    }
        
                    $existingSku = ProductVariationOption::where('sku', $sku)
                        ->when($optionId, function($query) use ($optionId) {
                            return $query->where('id', '!=', $optionId);
                        })
                        ->exists();

                    if ($existingSku) {
                        $validator->errors()->add("variations.$i.colors.$j.sku", 'The SKU has already been taken.');
                    }
                }
            }
        });
    }
        

        if ($validator->fails()) {
            // dd($validator->errors()->toArray()); // Shows which fields failed and why
        
            \Log::error('Product update validation failed', [
                'errors' => $validator->errors()->toArray()
            ]);
        
            return back()
                ->withErrors($validator)
                ->withInput();
        }
        
        $validated = $validator->validated();
        
        
        $product->update([
            'name' => $validated['name'],
            'description' => $validated['description'],
            'category_id' => $validated['category_id'],
            'brand_id' => $validated['brand_id'],
            'keywords' => $validated['keywords'],
            'railing_type' => $validated['railing_type'] ?? false,
            'weight' => $validated['weight'] ?? null,
            'status' => $validated['status'] ?? null,
            'length' => $validated['length'] ?? null,
            'width' => $validated['width'] ?? null,
            'height' => $validated['height'] ?? null,
            'volume' => $validated['volume'] ?? null,
            'max_load' => $validated['max_load'] ?? null,
        ]);
    
        if ($request->has('removed_images')) {
            foreach ($request->removed_images as $imageId) {
                $image = ProductImage::find($imageId);
                if ($image) {
                    Storage::delete('public/' . $image->image_path);
                    $image->delete();
                }
            }
        }

        $existingVariationIds = [];
        $category = Category::findOrFail($validated['category_id']);

        $sizeIds = is_array($category->size_ids) ? $category->size_ids : json_decode($category->size_ids, true);
        $variationIds = is_array($category->variation_ids) ? $category->variation_ids : json_decode($category->variation_ids, true);
        if (!empty($validated['variations'])) {
        foreach ($validated['variations'] as $variationData) {
            $colors = isset($variationData['colors']) && is_array($variationData['colors']) 
                ? array_values($variationData['colors']) 
                : [];

            if (isset($variationData['id']) && str_starts_with($variationData['id'], 'new-')) {
                $variationData['id'] = null;
            }

            
            $id = $variationData['size_id']; 
            $isSize = in_array($id, $sizeIds ?? []);
            $isVariation = in_array($id, $variationIds ?? []);
           
            $variation = $product->variations()->updateOrCreate(
                ['id' => $variationData['id'] ?? null],
                [
                    'size_id' => $isSize ? (int) $id : null,
                    'variation_id' => $isVariation ? (int)$id : null,
                    'weight' => $variationData['weight'],
                    'status' => $variationData['status'],
                    'length' => $variationData['length'],
                    'width' => $variationData['width'],
                    'height' => $variationData['height'],
                    'volume' => $variationData['volume'],
                    'max_load' => $variationData['max_load'],
                ]
            );

            $existingVariationIds[] = $variation->id;

            $existingOptionIds = [];
            foreach ($colors as $colorData) {
                if (empty($colorData['color_id'])) continue;

                if (isset($colorData['id']) && str_starts_with($colorData['id'], 'new-')) {
                    $colorData['id'] = null;
                }

               

                $option = $variation->options()->updateOrCreate(
                    ['id' => $colorData['id'] ?? null],
                    [
                        'color_id' => $colorData['color_id'],
                        'quantity' => $colorData['quantity'],
                        'sku' => $colorData['sku'],
                    ]
                );
                $existingOptionIds[] = $option->id;
            }


            $variation->options()->whereNotIn('id', $existingOptionIds)->delete();
        }

        }
       
        
        $product->variations()->whereNotIn('id', $existingVariationIds)->delete();
        $existingInfoIds = [];
       
        if (!empty($validated['other_info'])) {
            foreach ($validated['other_info'] as $infoData) {
                $imagePath = $infoData['existing_image'] ?? null;
    
                if (isset($infoData['image'])) {
                    if ($imagePath) {
                        Storage::delete('public/' . $imagePath);
                    }
                    $imagePath = $infoData['image']->store('products/other_info', 'public');
                }
    
                $info = $product->otherInfo()->updateOrCreate(
                    ['id' => $infoData['id'] ?? null],
                    [
                        'description' => $infoData['description'],
                        'image_path' => $imagePath,
                    ]
                );
                $existingInfoIds[] = $info->id;
            }
        }
        $product->otherInfo()->whereNotIn('id', $existingInfoIds)->delete();
    
        $existingModelIds = [];
        if (!empty($validated['models'])) {
            foreach ($validated['models'] as $modelData) {
                $model = $product->models()->updateOrCreate(
                    ['id' => $modelData['id'] ?? null],
                    [
                        'title' => $modelData['title'],
                        'description' => $modelData['description'],
                    ]
                );
                $existingModelIds[] = $model->id;
            }
        }
        $product->models()->whereNotIn('id', $existingModelIds)->delete();
    
        if ($request->hasFile('gallery_images')) {
            foreach ($request->file('gallery_images') as $image) {
                $path = $image->store('products/gallery', 'public');
                ProductImage::create([
                    'product_id' => $product->id,
                    'image_path' => $path,
                    'type' => 'gallery',
                ]);
            }
        }


        RelatedProductVariation::where('product_id', $product->id)->delete();
        if ($request->has('related_products')) {
            $relatedOptions = $request->input('related_products');
    
            $insertData = [];
            foreach ($relatedOptions as $optionId) {
                $insertData[] = [
                    'product_id' => $product->id,
                    'variation_option_id' => $optionId,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }
    
            RelatedProductVariation::insert($insertData);
        }

    
        if ($request->hasFile('main_image')) {
            $oldMainImage = $product->images()->where('type', 'main')->first();
            if ($oldMainImage) {
                Storage::delete('public/' . $oldMainImage->image_path);
                $oldMainImage->delete();
            }
    
            $path = $request->file('main_image')->store('products/main', 'public');
            ProductImage::create([
                'product_id' => $product->id,
                'image_path' => $path,
                'type' => 'main',
            ]);
        }
    
        if ($request->hasFile('info_image')) {
            $oldInfoImage = $product->images()->where('type', 'info')->first();
            if ($oldInfoImage) {
                Storage::delete('public/' . $oldInfoImage->image_path);
                $oldInfoImage->delete();
            }
    
            $path = $request->file('info_image')->store('products/info', 'public');
            ProductImage::create([
                'product_id' => $product->id,
                'image_path' => $path,
                'type' => 'info',
            ]);
        }
        if ($request->hasFile('home_image')) {
            $oldInfoImage = $product->images()->where('type', 'home')->first();
            if ($oldInfoImage) {
                Storage::delete('public/' . $oldInfoImage->image_path);
                $oldInfoImage->delete();
            }
    
            $path = $request->file('home_image')->store('products/info', 'public');
            ProductImage::create([
                'product_id' => $product->id,
                'image_path' => $path,
                'type' => 'home',
            ]);
        }
    
        return redirect()->route('admin.products.index')->with('success', 'Product updated successfully');
    }

    public function destroy($id)
    {
        $product = Product::with([
            'images',
            'variations.options',
            'variations',
            'otherInfos',
            'models',
            'related_product_variations',
        ])->findOrFail($id);
    
        // Delete product images from storage and DB
        foreach ($product->images as $image) {
            if ($image->image_path && Storage::exists('public/' . $image->image_path)) {
                Storage::delete('public/' . $image->image_path);
            }
            $image->delete();
        }
    
        // Delete other info images
        foreach ($product->otherInfos as $info) {
            if ($info->image_path && Storage::exists('public/' . $info->image_path)) {
                Storage::delete('public/' . $info->image_path);
            }
            $info->delete();
        }
    
        // Delete product models
        foreach ($product->models as $model) {
            $model->delete();
        }
    
        // Delete related variation references
        RelatedProductVariation::where('product_id', $product->id)->delete();
    
        // Delete variation options and variations
        foreach ($product->variations as $variation) {
            foreach ($variation->options as $option) {
                $option->delete();
            }
            $variation->delete();
        }
    
        // Finally delete the product itself
        $product->delete();
    
        return redirect()->route('admin.products.index')->with('success', 'Product and all related data deleted successfully.');
    }

    public function updateVariantStatus(Request $request, $id)
    {
        $variant = ProductVariation::findOrFail($id);
        $variant->status = $request->status === 'in_stock' ? 'in_stock' : 'out_of_stock';
        $variant->save();
    
        return response()->json(['success' => true]);
    }
    
    public function updateProductStatus(Request $request, $id)
    {
        $product = Product::findOrFail($id);
        $product->status = $request->status === 'in_stock' ? 'in_stock' : 'out_of_stock';
        $product->save();
    
        return response()->json(['success' => true]);
    }
    public function exportFull()
    {
        $products = Product::with([
            'category', 
            'brand', 
            'variations.options.color', 
            'variations.size',
            'variations.variation',
            'models', 
            'otherInfos', 
            'images'
        ])->get();
        
        $csv = Writer::createFromString('');
        $csv->insertOne([
            'ID', 'Name', 'Category', 'Brand', 'Keywords', 'Railing Type',
            'Variation ID', 'Variation Type', 'Variation Name', 'Color', 'SKU', 'Total Quantity',
            'Variation Weight', 'Variation Status', 'Variation Length', 'Variation Width', 
            'Variation Height', 'Variation Volume', 'Variation Max Load',
            'Created At', 'Updated At'
        ]);
    
        foreach ($products as $product) {
            // Skip products without variations
            if ($product->variations->isEmpty()) {
                continue;
            }
    
            $modelTitles = $product->models->pluck('title')->implode('|');
            $modelDescriptions = $product->models->pluck('description')->implode('|');
            $otherInfoDescriptions = $product->otherInfos->pluck('description')->implode('|');
    
            foreach ($product->variations as $variation) {
                foreach ($variation->options as $option) {
                    $csv->insertOne([
                        $product->id,
                        $product->name,
                        $product->category->name,
                        $product->brand->name,
                        $product->keywords,
                        $product->railing_type ? 'Yes' : 'No',
                        $variation->id,
                        $variation->size_id ? 'Size' : 'Variation',
                        $variation->size_id ? $variation->size->name : $variation->variation->name,
                        $option->color->name,
                        $option->sku,
                        $option->quantity,
                        $variation->weight,
                        $variation->status,
                        $variation->length,
                        $variation->width,
                        $variation->height,
                        $variation->volume,
                        $variation->max_load,
                      
                        $product->created_at->format('d.m.y'),
                        $product->updated_at->format('d.m.y')
                    ]);
                }
            }
        }
    
        return response((string) $csv, 200, [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="products_export_'.date('Y-m-d').'.csv"',
        ]);
    }

public function exportTemplate()
{
    $csv = Writer::createFromString('');
    $csv->insertOne([
        'Name*', 'Description*', 'Category*', 'Brand*', 'Keywords', 'Railing Type (Yes/No)',
        'Weight', 'Status (in_stock/out_of_stock/pre_order)', 'Length', 'Width', 'Height', 'Volume', 'Max Load',
        'Variation Type (Size/Variation)', 'Variation Name*', 'Color*', 'SKU*', 'Total Quantity*',
        'Variation Weight', 'Variation Status', 'Variation Length', 'Variation Width', 
        'Variation Height', 'Variation Volume', 'Variation Max Load',
        'Model Title', 'Model Description',
        'Other Info Description', 'Other Info Image Path',
        'Main Image Path', 'Info Image Path', 'Home Image Path', 'Gallery Image Paths (separate with |)'
    ]);

    return response((string) $csv, 200, [
        'Content-Type' => 'text/csv',
        'Content-Disposition' => 'attachment; filename="product_import_template.csv"',
    ]);
}

public function import(Request $request)
{
    $request->validate([
        'csv_file' => 'required|file|mimes:csv,txt'
    ]);

    $file = $request->file('csv_file');
    $csv = Reader::createFromPath($file->getPathname(), 'r');
    $csv->setHeaderOffset(0);
    
    $header = $csv->getHeader();
    $records = $csv->getRecords();
    
    // Check which image columns exist in the CSV
    $hasMainImage = in_array('Main Image', $header);
    $hasInfoImage = in_array('Info Image', $header);
    $hasHomeImage = in_array('Home Image', $header);
    $hasGalleryImages = in_array('Gallery Images', $header);
    $hasOtherInfoImages = in_array('Other Info Images', $header);


    
    $products = [];
    $currentProduct = null;

    $csvProductNames = [];
    $csvVariationIds = [];

    
    $parseDecimal = function($value) {
        return $value === '' || $value === null ? null : (float)$value;
    };
    
    foreach ($records as $index => $record) {
        // Skip rows where ALL fields are empty
        if (empty(array_filter($record))) continue;
    
        // Initialize product if "Name" is set and new
        $name = trim($record['Name'] ?? '');
        if ($name !== '') {
            if ($currentProduct && $currentProduct['name'] !== $name) {
                $products[] = $currentProduct;
            }
    
            if (!$currentProduct || $currentProduct['name'] !== $name) {
                $currentProduct = [
                    'name' => $name,
                    'category' => trim($record['Category'] ?? ''),
                    'brand' => trim($record['Brand'] ?? ''),
                    'keywords' => $record['Keywords'] ?? '',
                    'railing_type' => strtolower($record['Railing Type'] ?? 'no') === 'yes',
                    'variations' => [],
                    'models' => [],
                    'other_infos' => [],
                    'main_image' => $hasMainImage ? ($record['Main Image'] ?? null) : null,
                    'info_image' => $hasInfoImage ? ($record['Info Image'] ?? null) : null,
                    'home_image' => $hasHomeImage ? ($record['Home Image'] ?? null) : null,
                    'gallery_images' => $hasGalleryImages ?
                        (!empty($record['Gallery Images']) ? explode('|', $record['Gallery Images']) : []) : [],
                    'images_processed' => false,
                    'has_main_image' => $hasMainImage,
                    'has_info_image' => $hasInfoImage,
                    'has_home_image' => $hasHomeImage,
                    'has_gallery_images' => $hasGalleryImages,
                    'has_other_info_images' => $hasOtherInfoImages
                ];
            }
        }
    
        // Add variation if any
        if (!empty($record['Variation Name'])) {
            $variation = [
                'id' => $record['Variation ID'] ?? null,
                'type' => $record['Variation Type'] ?? 'Size',
                'name' => $record['Variation Name'],
                'color' => $record['Color'],
                'sku' => $record['SKU'],
                'quantity' => (int)($record['Total Quantity'] ?? 0),
                'weight' => $parseDecimal($record['Variation Weight'] ?? null),
                'status' => $record['Variation Status'] ?? 'in_stock',
                'length' => $parseDecimal($record['Variation Length'] ?? null),
                'width' => $parseDecimal($record['Variation Width'] ?? null),
                'height' => $parseDecimal($record['Variation Height'] ?? null),
                'volume' => $parseDecimal($record['Variation Volume'] ?? null),
                'max_load' => $parseDecimal($record['Variation Max Load'] ?? null),
            ];
            $currentProduct['variations'][] = $variation;
        }
    }
    
    // Add the last product
    if ($currentProduct && !in_array($currentProduct, $products)) {
        $products[] = $currentProduct;
    }

    // Process the products
    $imported = 0;
    $updated = 0;
    $errors = [];
    foreach ($products as $index => $productData) {
        try {
            DB::beginTransaction();

            if (empty($productData['name'])) {
                throw new \Exception("Product name is required (Row " . ($index + 2) . ")");
            }
            
            $category = Category::firstOrCreate(['name' => $productData['category']]);


            $product = Product::where('name', $productData['name'])->first();

            
            if ($product) {
                $product->update([
                    'keywords' => $productData['keywords'],
                ]);
            } else {
                $brand = ProductBrand::firstOrCreate(['name' => $productData['brand'] ?? '']);
                $product = Product::create([
                    'name' => $productData['name'],
                    'category_id' => $category->id,
                    'brand_id' => $brand->id,
                    'keywords' => $productData['keywords'],
                    'railing_type' => $productData['railing_type'],
                ]);
            }

            
            foreach ($productData['variations'] as $variationData) {
                $variation = ProductVariation::where('id', $variationData['id'])->first();
    
                if ($variation) {
                    // ✅ Update only variation status
                    $variation->update([
                        'status' => $variationData['status'],
                    ]);
    
                    // Update option quantity
                    $option = ProductVariationOption::where('sku', $variationData['sku'])->first();
                    if ($option) {
                        $option->update(['quantity' => $variationData['quantity']]);
                    }
                } else {
                    // New variation → allow all fields
                    if ($variationData['type'] === 'Size') {
                        $size = Size::firstOrCreate(['name' => $variationData['name']]);
                        $variation = ProductVariation::create([
                            'product_id' => $product->id,
                            'size_id' => $size->id,
                            'weight' => $variationData['weight'],
                            'length' => $variationData['length'],
                            'width' => $variationData['width'],
                            'height' => $variationData['height'],
                            'volume' => $variationData['volume'],
                            'max_load' => $variationData['max_load'],
                            'status' => $variationData['status'],
                        ]);
                    } else {
                        $variationType = Variation::firstOrCreate(['name' => $variationData['name']]);
                        $variation = ProductVariation::create([
                            'product_id' => $product->id,
                            'variation_id' => $variationType->id,
                            'weight' => $variationData['weight'],
                            'length' => $variationData['length'],
                            'width' => $variationData['width'],
                            'height' => $variationData['height'],
                            'volume' => $variationData['volume'],
                            'max_load' => $variationData['max_load'],
                            'status' => $variationData['status'],
                        ]);
                    }
    
                    $color = Color::firstOrCreate(['name' => $variationData['color']]);
                    ProductVariationOption::create([
                        'variation_id' => $variation->id,
                        'color_id' => $color->id,
                        'sku' => $variationData['sku'],
                        'quantity' => $variationData['quantity'],
                    ]);
                }
            }
          
            // Product::whereNotIn('name', $csvProductNames)->delete();

            // Delete variations not in CSV
            // ProductVariation::whereNotIn('id', $csvVariationIds)->delete();
    
           
            DB::commit();
            $imported++;
        } catch (\Exception $e) {
            DB::rollBack();
            $errors[] = "Row " . ($index + 2) . ": " . $e->getMessage();
        }
    }
    
    if (!empty($errors)) {
        dd($errors);
        return redirect()->back()
            ->with('error', 'Some products failed to import')
            ->with('error_details', $errors);
    }
    
    return redirect()->back()->with('success', "Successfully processed {$imported} products (updated existing records where needed)");
}
}
