<?php

namespace App\Http\Controllers\Admin\ReportAnalytics;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use App\Models\Booking;
use App\Exports\DailyTasksExport;
use Maatwebsite\Excel\Facades\Excel;
use PDF;
use App\Models\User;
use App\Exports\RevenueReportExport;


class ReportAnalyticsController extends Controller
{
    public $filePath = 'admin.report';

    public function index()
    {
        return view($this->filePath.'.index');
    }

    public function dashboardCards(Request $request)
    {
        $type = $request->query('type', 'year');
        
        $revenueQuery = DB::table('bookings')
            ->whereIn('status', ['confirmed', 'completed','cancelled'])
            ->whereExists(function ($query) {
                $query->select(DB::raw(1))
                    ->from('booking_items')
                    ->whereColumn('booking_items.booking_id', 'bookings.id')
                    ->whereNull('booking_items.vendor_id');
            });
        
        $refundQuery = Booking::query()
            ->where('payment_status', 'refund')
            ->where('vendor_id',null)
            ->whereIn('status', ['confirmed', 'completed', 'cancelled']);
            
        
        $conversionQuery = DB::table('bookings')->where('vendor_id',null);
        $this->applyTimeFilter($conversionQuery, $type);
        $this->applyTimeFilter($refundQuery, $type);
        $totalBookings = $conversionQuery->count();
        $confirmedBookings = $conversionQuery->clone()->where('status', 'confirmed')->count();
        $conversionRate = $totalBookings > 0 ? round(($confirmedBookings / $totalBookings) * 100) : 0;

        $guestUsers = User::where('is_guest',1)->whereIn('id', function ($query) {
            $query->select('user_id')
                ->from('bookings')
                ->where('status', 'confirmed')
                ->where('payment_status', 'paid');
        })->count();
        
        
        // Cancellation Rate
        $cancelledBookings = DB::table('bookings')
            ->where('vendor_id',null)
            ->where('status', 'cancelled');
        $this->applyTimeFilter($cancelledBookings, $type);
        $cancelledCount = $cancelledBookings->count();
        $cancellationRate = $totalBookings > 0 ? round(($cancelledCount / $totalBookings) * 100) : 0;
        
        $totalCustomers = DB::table('users')
        ->where('is_guest', '0')
        ->where('is_deleted', '0')
        ->where('status', 1)
        ->whereIn('id', function ($query) {
            $query->select('user_id')
                ->from('bookings')
                ->where('vendor_id',null)
                ->where('status', 'confirmed')
                ->where('payment_status', 'paid');
        })
        ->count();
        
        $refundAmount = $refundQuery->sum('refund_amount');
        $this->applyTimeFilter($revenueQuery, $type);
        $rentalRevenue = $revenueQuery->sum('subtotal') - $refundAmount;
        
        return response()->json([
            'rental_revenue' => number_format($rentalRevenue, 2, '.', ''),
            'conversion_rate' => $guestUsers,
            'cancelation_rate' => $cancellationRate,
            'total_customers' => $totalCustomers
        ]);
    }

    public function revenueChartData(Request $request)
    {
        $type = $request->query('type', 'year');
        $query = DB::table('bookings')
            ->where('status', 'confirmed')
            ->where('vendor_id',null)
            ->where('payment_status', 'paid');

        $labels = [];
        $revenueData = [];

        if ($type === 'year') {
            $query->selectRaw('MONTH(created_at) as label, SUM(subtotal) as revenue')
                ->whereYear('created_at', Carbon::now()->year)
                ->groupBy(DB::raw('MONTH(created_at)'))
                ->orderBy('label');

            $labels = ["Jan", "Feb", "Mar", "Apr", "May", "June", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"];
            $revenueData = array_fill(1, 12, 0);
        } elseif ($type === 'month') {
            $query->selectRaw('DAY(created_at) as label, SUM(subtotal) as revenue')
                ->whereMonth('created_at', Carbon::now()->month)
                ->whereYear('created_at', Carbon::now()->year)
                ->groupBy(DB::raw('DAY(created_at)'))
                ->orderBy('label');

            $daysInMonth = Carbon::now()->daysInMonth;
            $labels = range(1, $daysInMonth);
            $revenueData = array_fill(1, $daysInMonth, 0);
        } elseif ($type === 'week') {
            $query->selectRaw('DATE(created_at) as label, SUM(subtotal) as revenue')
                ->whereBetween('created_at', [
                    Carbon::now()->startOfWeek(),
                    Carbon::now()->endOfWeek()
                ])
                ->groupBy(DB::raw('DATE(created_at)'))
                ->orderBy('label');

            $labels = [];
            $revenueData = [];
            for ($date = Carbon::now()->startOfWeek(); $date <= Carbon::now()->endOfWeek(); $date->addDay()) {
                $labels[] = $date->format('D');
                $revenueData[$date->toDateString()] = 0;
            }
        } elseif ($type === 'day') {
            $query->selectRaw('HOUR(created_at) as label, SUM(subtotal) as revenue')
                ->whereDate('created_at', Carbon::today())
                ->groupBy(DB::raw('HOUR(created_at)'))
                ->orderBy('label');

            $labels = range(0, 23);
            $revenueData = array_fill(0, 24, 0);
        }

        foreach ($query->get() as $row) {
            $label = is_numeric($row->label) ? (int)$row->label : $row->label;
            $revenueData[$label] = (float)$row->revenue;
        }

        return response()->json([
            'labels' => array_values($labels),
            'data' => array_values($revenueData),
        ]);
    }

    public function revenueBreakdown2(Request $request)
    {
        $type = $request->query('type', 'year');
        $query = DB::table('bookings')
            ->where('status', 'confirmed')
            ->where('vendor_id',null)
            ->where('payment_status', 'paid');
        
        $this->applyTimeFilter($query, $type);
        
        $data = $query->selectRaw('SUM(subtotal) as rental, SUM(tax) as tax')->first();
        
        return response()->json([
            'rental' => round($data->rental ?? 0, 2),
            'tax' => round($data->tax ?? 0, 2),
        ]);
    }

    public function revenueBreakdown(Request $request)
    {
        $type = $request->query('type', 'year');

        $revenueQuery = DB::table('bookings')
            ->whereIn('status', ['confirmed', 'completed', 'cancelled'])
            ->whereExists(function ($query) {
                $query->select(DB::raw(1))
                    ->from('booking_items')
                    ->whereColumn('booking_items.booking_id', 'bookings.id')
                    ->whereNull('booking_items.vendor_id');
            });

        $refundQuery = Booking::query()
            ->where('payment_status', 'refund')
            ->where('vendor_id', null)
            ->whereIn('status', ['confirmed', 'completed', 'cancelled']);

        $this->applyTimeFilter($revenueQuery, $type);
        $this->applyTimeFilter($refundQuery, $type);

        $refundAmount = $refundQuery->sum('refund_amount');
        $data = $revenueQuery->selectRaw('SUM(subtotal) as rental, SUM(tax) as tax')->first();

        $rental = ($data->rental ?? 0) - $refundAmount;
        $tax = $data->tax ?? 0;

        return response()->json([
            'rental' => round($rental, 2),
            'tax' => round($tax, 2),
        ]);
    }


    public function cancellationReport(Request $request)
    {
        $type = $request->query('type', 'year');
        $query = DB::table('bookings')
            ->where('vendor_id',null)
            ->where('status', 'cancelled');
        
        if ($type === 'year') {
            $query->selectRaw('MONTH(created_at) as month, COUNT(*) as count')
                ->whereYear('created_at', now()->year)
                ->groupBy(DB::raw('MONTH(created_at)'))
                ->orderBy('month');
                
            $monthlyCancellations = array_fill(1, 12, 0);
            foreach ($query->get() as $row) {
                $monthlyCancellations[(int)$row->month] = $row->count;
            }
            
            return response()->json([
                'labels' => ["Jan", "Feb", "Mar", "Apr", "May", "June", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"],
                'data' => array_values($monthlyCancellations),
            ]);
        } elseif ($type === 'month') {
            $query->selectRaw('DAY(created_at) as day, COUNT(*) as count')
                ->whereMonth('created_at', now()->month)
                ->whereYear('created_at', now()->year)
                ->groupBy(DB::raw('DAY(created_at)'))
                ->orderBy('day');
                
            $daysInMonth = now()->daysInMonth;
            $dailyCancellations = array_fill(1, $daysInMonth, 0);
            foreach ($query->get() as $row) {
                $dailyCancellations[(int)$row->day] = $row->count;
            }
            
            return response()->json([
                'labels' => range(1, $daysInMonth),
                'data' => array_values($dailyCancellations),
            ]);
        } elseif ($type === 'week') {
            $query->selectRaw('DAYOFWEEK(created_at) as day, COUNT(*) as count')
                ->whereBetween('created_at', [
                    now()->startOfWeek(),
                    now()->endOfWeek()
                ])
                ->groupBy(DB::raw('DAYOFWEEK(created_at)'))
                ->orderBy('day');
                
            $weeklyCancellations = array_fill(1, 7, 0);
            foreach ($query->get() as $row) {
                $weeklyCancellations[(int)$row->day] = $row->count;
            }
            
            return response()->json([
                'labels' => ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'],
                'data' => array_values($weeklyCancellations),
            ]);
        } else { // day
            $query->selectRaw('HOUR(created_at) as hour, COUNT(*) as count')
                ->whereDate('created_at', now()->toDateString())
                ->groupBy(DB::raw('HOUR(created_at)'))
                ->orderBy('hour');
                
            $hourlyCancellations = array_fill(0, 24, 0);
            foreach ($query->get() as $row) {
                $hourlyCancellations[(int)$row->hour] = $row->count;
            }
            
            return response()->json([
                'labels' => range(0, 23),
                'data' => array_values($hourlyCancellations),
            ]);
        }
    }

    public function paymentMethodBreakdown(Request $request)
    {
        $type = $request->query('type', 'year');
        $query = DB::table('bookings')
            ->where('payment_status', 'paid')
            ->where('vendor_id',null)
            ->whereNotNull('payment_method');
        
        $this->applyTimeFilter($query, $type);
        
        $rawData = $query->selectRaw('LOWER(payment_method) as method, COUNT(*) as count')
            ->groupBy('method')
            ->get()
            ->pluck('count', 'method');

            $methods = ['card' => 0, 'paypal' => 0, 'cash pay' => 0, 'others' => 0];

            foreach ($rawData as $method => $count) {
                $normalized = match (true) {
                    str_contains($method, 'stripe'),
                    str_contains($method, 'card')     => 'card',
                    str_contains($method, 'paypal')   => 'paypal',
                    str_contains($method, 'cash')     => 'cash pay',
                    default                           => 'others',
                };
            
                $methods[$normalized] += $count;
            }

        return response()->json([
            'labels' => array_map('ucwords', array_keys($methods)),
            'data' => array_values($methods),
            'total' => array_sum($methods),
        ]);
    }

    public function topProductsChart(Request $request)
    {
        $type = $request->query('type', 'year');
        $query = DB::table('booking_items')
            ->join('bookings', 'booking_items.booking_id', '=', 'bookings.id')
            ->whereIn('booking_items.status', ['confirmed', 'completed'])
            ->where('bookings.payment_status', 'paid')
            ->whereNull('booking_items.vendor_id');;
        
        $this->applyTimeFilter($query, $type, 'bookings.created_at');
        
        $data = $query->select('product_name', DB::raw('COUNT(*) as count'))
            ->groupBy('product_name')
            ->orderByDesc('count')
            ->limit(5)
            ->get();

        return response()->json([
            'labels' => $data->pluck('product_name')->toArray(),
            'data' => $data->pluck('count')->toArray(),
        ]);
    }

    private function applyTimeFilter($query, $type, $column = 'created_at')
    {
        switch ($type) {
            case 'year':
                $query->whereYear($column, Carbon::now()->year);
                break;
            case 'month':
                $query->whereMonth($column, Carbon::now()->month)
                      ->whereYear($column, Carbon::now()->year);
                break;
            case 'week':
                $query->whereBetween($column, [
                    Carbon::now()->startOfWeek(),
                    Carbon::now()->endOfWeek()
                ]);
                break;
            case 'day':
                $query->whereDate($column, Carbon::today());
                break;
        }
    }


    public function downloadDailyTasksPdf(Request $request)
    {
        $fromDate = $request->input('from_date');
        $toDate   = $request->input('to_date');
        
        $query = Booking::with(['userDetails', 'items'])
            ->where('status', 'confirmed')
            ->whereNull('vendor_id')
            ->orderBy('created_at');

        // include pickup, dropoff, and new_dropoff in filter
        $query->whereHas('items', function($q) use ($fromDate, $toDate) {
            if ($fromDate && $toDate) {
                $q->whereBetween('pickup_date', [$fromDate, $toDate])
                ->orWhereBetween('dropoff_date', [$fromDate, $toDate])
                ->orWhereBetween('new_dropoff_date', [$fromDate, $toDate]);
            }
        });

        $bookings = $query->get();

        $data = [
            'bookings' => $bookings,
            'fromDate' => $fromDate,
            'toDate'   => $toDate,
            'title'    => 'Daily Tasks Report'
        ];

        $pdf = \PDF::loadView('admin.report.daily_tasks_pdf', $data)
            ->setPaper('a3', 'landscape');

        return $pdf->download('daily-tasks-report-'.now()->format('Y-m-d').'.pdf');
    }


    public function downloadDailyTasksCsv(Request $request)
    {
        $fromDate = $request->input('from_date');
        $toDate = $request->input('to_date');

        return Excel::download(
            new DailyTasksExport($fromDate, $toDate), 
            'daily-tasks-report-'.now()->format('Y-m-d').'.csv', 
            \Maatwebsite\Excel\Excel::CSV,
            ['Content-Type' => 'text/csv']
        );
    }
    public function downloadRevenueReportPdf(Request $request)
    {
        $fromDate = $request->input('from_date');
        $toDate = $request->input('to_date');

        $bookings = Booking::with(['items', 'userDetails'])
            ->where('status', 'confirmed')
            ->where('vendor_id',null)
            ->when($fromDate, fn($q) => $q->whereDate('created_at', '>=', $fromDate))
            ->when($toDate, fn($q) => $q->whereDate('created_at', '<=', $toDate))
            ->get();

        $data = [
            'bookings' => $bookings,
            'fromDate' => $fromDate,
            'toDate' => $toDate,
        ];

        $pdf = PDF::loadView('admin.report.revenue_pdf', $data)
            ->setPaper('a3', 'landscape');

        return $pdf->download('revenue-report-' . now()->format('Y-m-d') . '.pdf');
    }

    public function downloadRevenueReportCsv(Request $request)
    {
        return Excel::download(
            new RevenueReportExport($request->from_date, $request->to_date),
            'revenue-report-' . now()->format('Y-m-d') . '.csv'
        );
    }
}