<?php

namespace App\Http\Controllers\Admin\SPO;

use App\Http\Controllers\Controller;
use App\Models\Vendor;
use Illuminate\Http\Request;
use Exception;
use Illuminate\Support\Facades\Validator;
use PDF;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class SPOController extends Controller
{
    public function index(Request $request)
    {
        $timeFilter = $request->get('time_filter');
        $search = $request->get('search');
        $statusFilter = $request->get('status_filter', 'all'); 
        
        $analytics = $this->getVendorAnalytics($timeFilter);
        
        $vendors = Vendor::query()
            ->when($search, function($query) use ($search) {
                $query->where(function($q) use ($search) {
                    $q->where('first_name', 'like', "%$search%")
                    ->orWhere('last_name', 'like', "%$search%")
                    ->orWhere('email', 'like', "%$search%")
                    ->orWhere('number', 'like', "%$search%");
                });
            })
            ->when($statusFilter !== 'all', function($query) use ($statusFilter) {
                if (in_array($statusFilter, ['active', 'inactive', 'closed'])) {
                    $query->where('status', $statusFilter);
                }
            })
            ->when($timeFilter === 'day', function($query) {
                $query->whereDate('created_at', now());
            })
            ->when($timeFilter === 'week', function($query) {
                $query->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()]);
            })
            ->when($timeFilter === 'month', function($query) {
                $query->whereMonth('created_at', now()->month)
                    ->whereYear('created_at', now()->year);
            })
            ->when($timeFilter === 'year', function($query) {
                $query->whereYear('created_at', now()->year);
            })
            ->orderBy('created_at', 'desc')
            ->get();
            
            return view('admin.spo.index', compact('analytics', 'vendors', 'timeFilter', 'search', 'statusFilter'));
        }

    
        protected function getVendorAnalytics($timeFilter)
        {
            $query = Vendor::query();
        
            // Apply time filter
            if ($timeFilter === 'day') {
                $query->whereDate('created_at', now());
            } elseif ($timeFilter === 'week') {
                $query->whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()]);
            } elseif ($timeFilter === 'month') {
                $query->whereMonth('created_at', now()->month)
                      ->whereYear('created_at', now()->year);
            } elseif ($timeFilter === 'year') {
                $query->whereYear('created_at', now()->year);
            }
            
            $totalVendors = (clone $query)->count();
            $fulltimer = (clone $query)
            ->where('business_model', 'Fulltimer')
            ->where('status', 'active')
            ->count();
        
        $sidebusiness = (clone $query)
            ->where('business_model', 'Side Business')
            ->where('status', 'active')
            ->count();
        
            $active = (clone $query)->where('status','active')->count();
        
            return [
                'total_users' => $totalVendors,
                'fulltimer' => $fulltimer,
                'sidebusiness' => $sidebusiness,
                'active' => $active,
            ];
        }
        
    
    public function create()
    {
        return view('admin.spo.create');
    }
    
    public function store(Request $request)
{
    try {
        $validated = $request->validate([
            'title' => 'required|string',
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email|unique:vendors,email',
            'password' => 'required|string|min:8|confirmed',
            'profile_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'country_code' => 'required|string',
            'number' => 'required|string',
            'street' => 'required|string',
            'zip_code' => 'required|string',
            'city' => 'required|string',
            'country' => 'required|string',
            'business_type' => 'required|string',
            'business_model' => 'required|string',
            'contact_person' => 'required|string',
            'size_of_location' => 'required|string',
            'tax_number' => 'required|string',
            'tax_id' => 'required|string|size:11',
            'bank_name' => 'required|string',
            'account_owner' => 'required|string',
            'account_number' => 'required|string',
            'bic_number' => 'required|string',
            'iban_number' => 'required|string',
            'days_closed' => 'nullable|array',
            'holidays' => 'nullable|array',
        ]);
    } catch (ValidationException $e) {
        return redirect()->back()->withErrors($e->errors())->withInput();
    }
    DB::beginTransaction();

    try {
        // Handle profile image upload
        if ($request->hasFile('profile_image')) {
            $path = $request->file('profile_image')->store('vendor_images', 'public');
            $validated['profile_image'] = $path;
        }

        // Create vendor
        $validated['password'] = Hash::make($validated['password']);
        $validated['status'] = 'inactive'; // Active by default
        $vendor = Vendor::create($validated);

        // Save working hours
        $days = ['mo', 'di', 'mi', 'do', 'fr', 'sa', 'so'];
        foreach ($days as $day) {
            $isClosed = in_array($day, $request->days_closed ?? []);
            
            $vendor->workingHours()->create([
                'day' => $day,
                'open_time' => $isClosed ? null : $request->input("{$day}_from"),
                'close_time' => $isClosed ? null : $request->input("{$day}_to"),
                'is_closed' => $isClosed
            ]);
        }

        // Save holidays
        if ($request->holidays) {
            foreach ($request->holidays as $holiday) {
                $vendor->holidays()->create([
                    'name' => $holiday['name'],
                    'date' => $holiday['date']
                ]);
            }
        }

        DB::commit();

        return redirect()->route('admin.spo.index')->with('success', 'SPO created successfully!');
    } catch (\Exception $e) {
        DB::rollBack();
        dd($e->getMessage());
        return back()->withInput()->with('error', 'Error creating SPO: ' . $e->getMessage());
    }
}
    
    public function show(Vendor $vendor)
    {
        
        return view('admin.spo.details', compact('vendor'));
    }
    
    public function edit(Vendor $vendor)
    {
        return view('admin.spo.edit', compact('vendor'));
    }
    
    public function update(Request $request, Vendor $vendor)
    {
        DB::beginTransaction();
    
        try {
            // Validate the request data
            $validated = $request->validate([
                'title' => 'required|string',
                'first_name' => 'required|string|max:255',
                'last_name' => 'required|string|max:255',
                'email' => 'required|email|unique:vendors,email,' . $vendor->id,
                'password' => 'nullable|string|min:8|confirmed',
                'profile_image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
                'country_code' => 'required|string',
                'number' => 'required|string',
                'street' => 'required|string',
                'zip_code' => 'required|string',
                'city' => 'required|string',
                'new_in_business'=> 'nullable',
                'country' => 'required|string',
                'business_type' => 'required|string',
                'business_model' => 'required|string',
                'contact_person' => 'required|string',
                'size_of_location' => 'required|string',
                'tax_number' => 'required|string',
                'tax_id' => 'required|string|size:11',
                'bank_name' => 'required|string',
                'account_owner' => 'required|string',
                'account_number' => 'required|string',
                'bic_number' => 'required|string',
                'iban_number' => 'required|string',
                'days_closed' => 'nullable|array',
                'holidays' => 'nullable|array',
                'holidays.*.id' => 'nullable|integer',
                'holidays.*.name' => 'required|string',
                'holidays.*.date' => 'required|date',
                'mo_from' => 'nullable|date_format:H:i',
                'mo_to' => 'nullable|date_format:H:i|after:mo_from',
                'di_from' => 'nullable|date_format:H:i',
                'di_to' => 'nullable|date_format:H:i|after:di_from',
                'mi_from' => 'nullable|date_format:H:i',
                'mi_to' => 'nullable|date_format:H:i|after:mi_from',
                'do_from' => 'nullable|date_format:H:i',
                'do_to' => 'nullable|date_format:H:i|after:do_from',
                'fr_from' => 'nullable|date_format:H:i',
                'fr_to' => 'nullable|date_format:H:i|after:fr_from',
                'sa_from' => 'nullable|date_format:H:i',
                'sa_to' => 'nullable|date_format:H:i|after:sa_from',
                'so_from' => 'nullable|date_format:H:i',
                'so_to' => 'nullable|date_format:H:i|after:so_from',
            ]);
    
            if ($request->hasFile('profile_image')) {
                // Delete old image if it exists
                if ($vendor->profile_image) {
                    Storage::disk('public')->delete($vendor->profile_image);
                }
                
                $path = $request->file('profile_image')->store('vendor_images', 'public');
                $validated['profile_image'] = $path;
            }
    
            if ($request->filled('password')) {
                $validated['password'] = Hash::make($validated['password']);
            } else {
                unset($validated['password']);
            }
            $vendor->update($validated);
    
            $days = ['mo', 'di', 'mi', 'do', 'fr', 'sa', 'so'];
            foreach ($days as $day) {
                $isClosed = in_array($day, $request->days_closed ?? []);
                
                $workingHour = $vendor->workingHours()->where('day', $day)->first();
                
                if ($workingHour) {
                    $workingHour->update([
                        'open_time' => $isClosed ? null : $request->input("{$day}_from"),
                        'close_time' => $isClosed ? null : $request->input("{$day}_to"),
                        'is_closed' => $isClosed
                    ]);
                } else {
                    $vendor->workingHours()->create([
                        'day' => $day,
                        'open_time' => $isClosed ? null : $request->input("{$day}_from"),
                        'close_time' => $isClosed ? null : $request->input("{$day}_to"),
                        'is_closed' => $isClosed
                    ]);
                }
            }
    
            if ($request->holidays) {
                $existingHolidayIds = $vendor->holidays->pluck('id')->toArray();
                $updatedHolidayIds = [];
                foreach ($request->holidays as $holidayData) {
                    if (isset($holidayData['id']) && $holidayData['new'] == 0) {
                        $holiday = $vendor->holidays()->find($holidayData['id']);
                        if ($holiday) {
                            $holiday->update([
                                'name' => $holidayData['name'],
                                'date' => $holidayData['date']
                            ]);
                            $updatedHolidayIds[] = $holiday->id;
                        }
                    } else {
                        $newHoliday = $vendor->holidays()->create([
                            'name' => $holidayData['name'],
                            'date' => $holidayData['date']
                        ]);
                        $updatedHolidayIds[] = $newHoliday->id;
                    }
                }
                
                // Delete holidays that were removed
                $holidaysToDelete = array_diff($existingHolidayIds, $updatedHolidayIds);
                if (!empty($holidaysToDelete)) {
                    $vendor->holidays()->whereIn('id', $holidaysToDelete)->delete();
                }
            } else {
                // If no holidays submitted, delete all existing ones
                $vendor->holidays()->delete();
            }
    
            DB::commit();
    
            return redirect()->route('admin.spo.index', $vendor)->with('success', 'SPO updated successfully!');
        } catch (ValidationException $e) {
            DB::rollBack();
            return back()->withErrors($e->errors())->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()->with('error', 'Error updating SPO: ' . $e->getMessage());
        }
    }
    
    public function status(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:vendors,id',
            'new_status' => 'required|in:Active,Blocked,Closed'
        ]);
        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => 'Invalid request']);
        }
        
        try {
            $vendor = Vendor::find($request->user_id);
            if($request->new_status == 'Blocked'){
                $request->new_status = 'inactive';
            }else if($request->new_status == 'Active'){
                $request->new_status = 'active';
            }else{
                $request->new_status = 'closed';
            }
            $vendor->status = $request->new_status ;
            $vendor->save();
            
            return redirect()->route('admin.spo.index')->with('success', 'Status Updated Successfully');
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()]);
        }
    }
    
    public function destroy(Request $request)
    {
        $requestData = $request->all();

        if (isset($requestData['user_id']) && is_string($requestData['user_id'])) {
            $requestData['user_id'] = json_decode($requestData['user_id'], true)['id'] ?? null;
        }
        
        // Now validate
        $validator = Validator::make($requestData, [
            'user_id' => 'required|exists:vendors,id'
        ]);

        
        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => 'Invalid request']);
        }
        
        try {
            Vendor::find($requestData['user_id'])->delete();
            return response()->json(['success' => true]);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Error deleting vendor']);
        }
    }
    
    public function downloadCSV(Request $request)
    {
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="vendors_' . date('d.m.Y') . '.csv"',
        ];
    
        $vendors = Vendor::query()
            ->when($request->search, function($query) use ($request) {
                $query->where(function($q) use ($request) {
                    $q->where('first_name', 'like', "%{$request->search}%")
                      ->orWhere('last_name', 'like', "%{$request->search}%")
                      ->orWhere('email', 'like', "%{$request->search}%");
                });
            })
            ->when($request->status_filter && $request->status_filter !== 'all', function($query) use ($request) {
                $query->where('status', $request->status_filter);
            })
            ->orderBy('created_at', 'desc')
            ->get();
    
        $callback = function() use ($vendors) {
            $file = fopen('php://output', 'w');
            
            // Header row
            fputcsv($file, [
                'ID', 'Title', 'First Name', 'Last Name', 'Email', 'Phone',
                'Street', 'ZIP Code', 'City', 'Country', 'Status','New in Business',
                'Business Model', 'Size of Location', 'Business Type',
                'Contract Type', 'Tax Number', 'Tax ID', 'Bank Name',
                'Account Owner', 'Account Number', 'BIC Number', 'IBAN Number'
                , 'SPO ID', 'Created At'
            ]);
            
            // Data rows
            foreach ($vendors as $vendor) {
                fputcsv($file, [
                    $vendor->id,
                    ucfirst($vendor->title),
                    $vendor->first_name,
                    $vendor->last_name,
                    $vendor->email,
                    $vendor->country_code . ' ' . $vendor->number,
                    $vendor->street,
                    $vendor->zip_code,
                    $vendor->city,
                    $vendor->country,
                    $vendor->status == 'active' ? 'Active' : 'Blocked',
                    $vendor->new_in_business ? 'Yes' : 'No',
                    $vendor->business_model ? __('messages.' . $vendor->business_model) : 'N/A',
                    $vendor->size_of_location ? __('messages.' . $vendor->size_of_location) : 'N/A',
                     $vendor->business_type ? __('messages.' . $vendor->business_type) : 'N/A',
                    $vendor->contact_person ? __('messages.' . $vendor->contact_person) : 'N/A',
                    $vendor->tax_number,
                    $vendor->tax_id,
                    $vendor->bank_name,
                    $vendor->account_owner,
                    $vendor->account_number,
                    $vendor->bic_number,
                    $vendor->iban_number,
                   
                    'SPO'.$vendor->id,
                    $vendor->created_at->format('d.m.Y H:i'),
                ]);
            }
            
            fclose($file);
        };
    
        return response()->stream($callback, 200, $headers);
    }
    
    public function downloadPDF(Request $request)
    {
        $vendors = Vendor::query()
            ->when($request->search, function($query) use ($request) {
                $query->where(function($q) use ($request) {
                    $q->where('first_name', 'like', "%{$request->search}%")
                      ->orWhere('last_name', 'like', "%{$request->search}%")
                      ->orWhere('email', 'like', "%{$request->search}%");
                });
            })
            ->when($request->status_filter && $request->status_filter !== 'all', function($query) use ($request) {
                $query->where('status', $request->status_filter);
            })
            ->orderBy('created_at', 'desc')
            ->get();
    
        $pdf = PDF::loadView('admin.spo.pdf', compact('vendors'));
        return $pdf->download('vendors_' . date('d.m.Y') . '.pdf');
    }

}