<?php

namespace App\Http\Controllers\Admin\SpoProduct;

use App\Http\Controllers\Controller;
use App\Models\Vendor;
use Illuminate\Http\Request;
use Exception;
use Illuminate\Support\Facades\Validator;
use PDF;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\SPOProductsExport;
use App\Models\BookingItem;
use App\Models\Category;
use App\Models\Bundle;
use App\Models\Product;
use App\Models\VendorProductQuantity;
use App\Models\ProductPrice;
use App\Models\ProductVariationOption;
use App\Models\RelatedProductVariation;
use App\Models\ProductVariation;
use App\Models\User;
use League\Csv\Reader;
use Illuminate\Support\Facades\Response;
use App\Models\Booking;
use Illuminate\Support\Str;

use Illuminate\Support\Facades\Auth;

class SpoProductController extends Controller
{

    // public function index(){
    //     $products = Product::with(['category'])->get();
    //     return view('admin.spoProduct.index',compact('products'));
    // }
    public function index(Request $request)
    {

        $status = $request->get('status');

        // $vendors = Vendor::where('status', 'active')
        // ->orderBy('created_at', $sortBy)
        // ->paginate(10);
        $query = Vendor::query();
        if ($status) {
            $query->where('status', $status);
        }
    
        $vendors = $query->orderBy('created_at', 'desc')->paginate(10);


        $products = Product::with(['category'])->get();

        $quantities = VendorProductQuantity::select(
                'vendor_id',
                'product_variations.product_id',
                DB::raw('SUM(vendors_product_quantities.quantity) as total_quantity')
            )
            ->join('product_variation_options', 'vendors_product_quantities.product_variation_option_id', '=', 'product_variation_options.id')
            ->join('product_variations', 'product_variation_options.variation_id', '=', 'product_variations.id')
            ->groupBy('vendor_id', 'product_variations.product_id')
            ->get()
            ->keyBy(fn ($item) => $item->vendor_id . '-' . $item->product_id);

        


        return view('admin.spoProduct.index', compact('vendors', 'products', 'quantities'));
    }

    public function detail(Vendor $vendor,Request $request){

        $rentedCounts = $this->getRentedItemsCount($vendor);
       

        $products = $vendor->products()->with([
            'category',
            'variations.options.color',
            'variations.size',
            'variations.variation',
        ]);

        if ($request->has('time_period')) {
            $timePeriod = $request->input('time_period');
            switch ($timePeriod) {
                case 'year':
                    $products->whereYear('products.created_at', date('Y'));
                    break;
                case 'month':
                    $products->whereMonth('products.created_at', date('m'))
                             ->whereYear('products.created_at', date('Y'));
                    break;
                case 'week':
                    $products->whereBetween('products.created_at', [
                        now()->startOfWeek(),
                        now()->endOfWeek()
                    ]);
                    break;
                case 'day':
                    $products->whereDate('products.created_at', date('Y-m-d'));
                    break;
            }
        }

        $products = $products->get();

        // Inject vendor-level quantity
        foreach ($products as $product) {
            foreach ($product->variations as $variation) {
                foreach ($variation->options as $option) {
                    $vendorQty = $option->vendorQuantities()
                        ->where('vendor_id', $vendor->id)
                        ->first();
                    $option->quantity = $vendorQty?->quantity ?? 0;
                }
            }
        }

        $disabledVariants = $vendor->disabledVariants()->pluck('product_variation_id')->toArray();


        // Inject vendor-level quantity
        // foreach ($products as $product) {
        //     foreach ($product->variations as $variation) {
        //         $variation->is_disabled = in_array($variation->id, $disabledVariants);
        //         $variation->status  = $variation->is_disabled ? 'out_of_stock' : 'in_stock';
        //         foreach ($variation->options as $option) {
        //             $vendorQty = $option->vendorQuantities()
        //                 ->where('vendor_id', $vendor->id)
        //                 ->first();
        //             $option->quantity = $vendorQty?->quantity ?? 0;
                
        //         }
        //     }
        // }

        foreach ($products as $product) {
            if ($product->variations->isEmpty()) {
                $rentedCount = $rentedCounts['product'][$product->id] ?? 0;
                $product->total_quantity = $product->quantity - $rentedCount;
            } else {
                foreach ($product->variations as $variation) {
                    $variation->is_disabled = in_array($variation->id, $disabledVariants);
                    $variation->status  = $variation->is_disabled ? 'out_of_stock' : 'in_stock';
                    if (!$variation->options->isEmpty()) {
                        foreach ($variation->options as $option) {
                            $vendorQty = $option->vendorQuantities()
                                ->where('vendor_id', $vendor->id)
                                ->first();
                            $option->quantity = $vendorQty?->quantity ?? 0;
                            $rentedCount = $rentedCounts['option'][$option->id] ?? 0;
                            $option->total_quantity = $option->quantity - $rentedCount;
                        }
                    } else {
                        $rentedCount = $rentedCounts['variation'][$variation->id] ?? 0;
                        $variation->total_quantity = $variation->quantity - $rentedCount;
                    }
                }
            }
        }
        $disabledVariants = $vendor->disabledVariants()->pluck('product_variation_id')->count();
        $totalProductCategories = Category::count();

        $bundleProducts = Bundle::count();

        $totalProducts = $vendor->products()->count();
        $activeRentals = 0;
        $returnedRentals = 0;
        $availableProducts = $products->sum('quantity');

        $groupedByCategory = $products->groupBy(fn ($product) => $product->category->name ?? 'Uncategorized');

        // $order = [
        //     'Roof Box',
        //     'Rearbox',
        //     'Roof Bike Carrier',
        //     'Clutch Bike Rack',
        //     'Roof Rack',
        //     'Carrier',
        //     'Foot',
        //     'Footkit',
        // ];

        $order = [
            'Dachbox',
            'Heckbox',
            'Dachfahrradträger',
            'Kupplungsfahrradträger',
            'Dachträger',
            'Traverse',
            'Fuß',
            'Fußkit',
        ];


        $groupedByCategory = $groupedByCategory->sortBy(function ($products, $categoryName) use ($order) {
            $index = array_search($categoryName, $order);
            return $index !== false ? $index : PHP_INT_MAX; // categories not in list go to end
        });

        $analytics = $this->calculateBookingAnalytics($request , $vendor);


       

        return view('admin.spoProduct.detail', compact(
            'groupedByCategory',
            'totalProducts',
            'activeRentals',
            'returnedRentals',
            'availableProducts',
            'vendor',
            'rentedCounts',
            'analytics'
        ));
    }

    private function calculateBookingAnalytics($request,Vendor $vendor)
    {
        $timeFilter = $request->input('time_filter', 'year');
        
        // Base query for confirmed/completed paid bookings
        $confirmedQuery = Booking::query()
            ->whereIn('status', ['confirmed', 'completed','cancelled']);

        
        $refundQuery = Booking::query()
            ->where('payment_status', 'refund')
            ->where('vendor_id',$vendor->id)
            ->whereIn('status', ['confirmed', 'completed', 'cancelled']);
        
        
        // Apply vendor filter if present
        if ($vendor) {
            $confirmedQuery->whereHas('items', function($q) use ($vendor) {
                $q->where('vendor_id', $vendor->id);
            });
        } else {
            $confirmedQuery->whereHas('items', function($q) use ($vendor) {
                $q->where('vendor_id', null);
            });
        }
        
        // Apply search filter if present
        if ($request->has('search')) {
            $searchQuery = $request->input('search');
            $confirmedQuery->whereHas('userDetails', function($q) use ($searchQuery) {
                $q->where('first_name', 'like', "%{$searchQuery}%")
                ->orWhere('last_name', 'like', "%{$searchQuery}%")
                ->orWhere('email', 'like', "%{$searchQuery}%")
                ->orWhere('number', 'like', "%{$searchQuery}%");
            });
        }
        
        // Query for cancellations
        $cancelledQuery = Booking::query()->where('status', 'cancelled');
        
        // Apply vendor filter to cancellations if present
        if ($vendor) {
            $cancelledQuery->whereHas('items', function($q) use ($vendor) {
                $q->where('vendor_id', $vendor->id);
            });
        } else {
            $cancelledQuery->whereHas('items', function($q) use ($request) {
                $q->where('vendor_id', null);
            });
        }
        
        // Apply search filter to cancellations if present
        if ($request->has('search')) {
            $searchQuery = $request->input('search');
            $cancelledQuery->whereHas('userDetails', function($q) use ($searchQuery) {
                $q->where('first_name', 'like', "%{$searchQuery}%")
                ->orWhere('last_name', 'like', "%{$searchQuery}%")
                ->orWhere('email', 'like', "%{$searchQuery}%")
                ->orWhere('number', 'like', "%{$searchQuery}%");
            });
        }
        $refundAmount = $refundQuery->sum('refund_amount');
        // Calculate metrics
        $netRevenue = $confirmedQuery->sum('subtotal') - $refundAmount;
        $totalBookings = $confirmedQuery->count();
        $cancellations = $cancelledQuery->count();
        $totalRevenue = $confirmedQuery->sum('total') - $refundAmount;
        
        return [
            'net_revenue' => number_format($netRevenue, 2, ',', '.'),
            'total_bookings' => $totalBookings,
            'cancellations' => $cancellations,
            'total_revenue' => number_format($totalRevenue, 2, ',', '.')
        ];
    }


   

    private function getRentedItemsCount(Vendor $vendor)
    {
        $today = now()->format('Y-m-d');
        $rentedItems = BookingItem::where('damage', '!=', 1)
            ->where('already_booked', 0)
            ->where('vendor_id', $vendor->id)
            ->whereHas('booking', function ($q) {
                $q->where('status', 'confirmed');
            })
            ->where(function($query) use ($today) {
                $query->where(function($q) use ($today) {
                    $q->where('pickup_date', '<=', $today)
                    ->where(function($q2) use ($today) {
                        $q2->where(function($q3) use ($today) {
                            $q3->whereNotNull('new_dropoff_date')
                                ->where('new_dropoff_date', '>=', $today);
                        })->orWhere(function($q4) use ($today) {
                            $q4->whereNull('new_dropoff_date')
                                ->where('dropoff_date', '>=', $today);
                        });
                    });
                });
            })
            ->get();



        $rentedCounts = [
            'product' => [],
            'variation' => [],
            'option' => []
        ];

        foreach ($rentedItems as $item) {
            // Count by product_id (for simple products like roof racks)
            if ($item->product_id) {
                if (!isset($rentedCounts['product'][$item->product_id])) {
                    $rentedCounts['product'][$item->product_id] = 0;
                }
                $rentedCounts['product'][$item->product_id]++;
            }
            
          
           
            if ($item->product_id && $item->size) {
                $product = Product::find($item->product_id);
                if ($product && $product->variations) {
                    foreach ($product->variations as $variation) {
                        if (($variation->size && $variation->size->name == $item->size)  || ($variation->variation && $variation->variation->name == $item->size)) {
                           
                            if (!isset($rentedCounts['variation'][$variation->id])) {
                                $rentedCounts['variation'][$variation->id] = 0;
                            }
                            $rentedCounts['variation'][$variation->id]++;
                            $name = strtolower($item->product_name);

                            if (in_array($name, ['carrier', 'foot', 'footkit', 'traverse', 'fuß', 'fußkit'])) {
                                $item->color = 'Schwarz';
                            }
                            // If there are options, find the matching one by color
                            if ($item->color && $variation->options) {
                                foreach ($variation->options as $option) {
                                    if ($option->color && $option->color->name == $item->color) {
                                        if (!isset($rentedCounts['option'][$option->id])) {
                                            $rentedCounts['option'][$option->id] = 0;
                                        }
                                        $rentedCounts['option'][$option->id]++;
                                        break;
                                    }
                                }
                            }
                            break;
                        }
                    }
                }
            }
        }
        // dd($rentedCounts);
        return $rentedCounts;
    }

    public function showVariant($productId, $variantId ,$optionId = null,$vendorId)
    {
        $variant = ProductVariation::with([
            'product.category',
            'product.brand',
            'options.color',
            'size',
            'variation',
            'product.images',
            'product.otherInfo',
            'product.models',
            
        ])->where('id', $variantId)
        ->where('product_id', $productId)
        ->firstOrFail();

        
        $product = $variant->product;
        $productPrice = ProductPrice::where('product_name', $product->name)->first();

        $relatedProducts = Product::with('homePhoto')
        ->whereIn('id', RelatedProductVariation::where('product_id', $product->id)
            ->pluck('variation_option_id')
        )
        ->get();
        

        

        $mainOption = null;
        if ($optionId) {
            $mainOption = $variant->options->firstWhere('id', $optionId);
            $vendorQty = $mainOption->vendorQuantities()
                        ->where('vendor_id',$vendorId)
                        ->first();
            $mainOption->quantity = $vendorQty?->quantity ?? 0;
        }
        
        $vendor = Vendor::where('id', $vendorId)->first();
        return view( 'admin.spoProduct.productDetail', [
            'product' => $variant->product,
            'variant' => $variant,
            'mainOption' => $mainOption,
            'relatedProducts' => $relatedProducts,
            'productPrice' => $productPrice,
            'vendor' => $vendor

            
        ]);
    }

    public function updateQuantity(Request $request)
    {
        $request->validate([
            'option_id' => 'required|exists:product_variation_options,id',
            'quantity' => 'required|integer|min:0',
            'vendor_id' => 'required'
        ]);
        $vendorId = $request->vendor_id;
        VendorProductQuantity::updateOrCreate(
            [
                'vendor_id' => $vendorId,
                'product_variation_option_id' => $request->option_id
            ],
            [
                'quantity' => $request->quantity
            ]
        );

        return redirect()->back()->with('success', 'Quantity updated successfully.');
    }

    


    public function downloadCSV(Request $request)
    {
        $fileName = 'spo_products_' . now()->format('d.m.Y') . '.csv';

        $vendors = Vendor::query()
            ->where('status', 'active')
            ->when($request->search, function ($q) use ($request) {
                $q->where(function ($q2) use ($request) {
                    $q2->where('first_name', 'like', "%{$request->search}%")
                    ->orWhere('last_name', 'like', "%{$request->search}%")
                    ->orWhere('email', 'like', "%{$request->search}%");
                });
            })
            ->orderBy('created_at', $request->sort_by ?? 'desc')
            ->get();

        $products = Product::all();

        $quantities = VendorProductQuantity::select(
                'vendor_id',
                'product_variations.product_id',
                DB::raw('SUM(vendors_product_quantities.quantity) as total_quantity')
            )
            ->join('product_variation_options', 'vendors_product_quantities.product_variation_option_id', '=', 'product_variation_options.id')
            ->join('product_variations', 'product_variation_options.variation_id', '=', 'product_variations.id')
            ->groupBy('vendor_id', 'product_variations.product_id')
            ->get()
            ->keyBy(fn ($item) => $item->vendor_id . '-' . $item->product_id);

        $headers = [
            "Content-type" => "text/csv",
            "Content-Disposition" => "attachment; filename=$fileName",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        ];

        $callback = function () use ($vendors, $products, $quantities) {
            $handle = fopen('php://output', 'w');

            // Header row
            $header = ['SPO ID', 'SPO Name', 'City'];
            foreach ($products as $product) {
                $header[] = $product->name;
            }
            fputcsv($handle, $header);

            // Rows
            foreach ($vendors as $vendor) {
                $row = [
                    '#' . $vendor->id,
                    $vendor->first_name . ' ' . $vendor->last_name,
                    $vendor->city
                ];

                foreach ($products as $product) {
                    $key = $vendor->id . '-' . $product->id;
                    $row[] = $quantities[$key]->total_quantity ?? 0;
                }

                fputcsv($handle, $row);
            }

            fclose($handle);
        };

        return response()->stream($callback, 200, $headers);
    }


    public function downloadPDF(Request $request)
    {
        
        $vendors = Vendor::query()
            ->where('status', 'active')
            ->when($request->search, function ($q) use ($request) {
                $q->where(function ($q2) use ($request) {
                    $q2->where('first_name', 'like', "%{$request->search}%")
                       ->orWhere('last_name', 'like', "%{$request->search}%")
                       ->orWhere('email', 'like', "%{$request->search}%");
                });
            })
            ->orderBy('created_at', $request->sort_by ?? 'desc')
            ->get();
    
        $products = Product::all();
    
        $quantities = VendorProductQuantity::select(
                'vendor_id',
                'product_variations.product_id',
                DB::raw('SUM(vendors_product_quantities.quantity) as total_quantity')
            )
            ->join('product_variation_options', 'vendors_product_quantities.product_variation_option_id', '=', 'product_variation_options.id')
            ->join('product_variations', 'product_variation_options.variation_id', '=', 'product_variations.id')
            ->groupBy('vendor_id', 'product_variations.product_id')
            ->get()
            ->keyBy(fn ($item) => $item->vendor_id . '-' . $item->product_id);
    
        $pdf = PDF::loadView('admin.spoProduct.pdf', compact('vendors', 'products', 'quantities'));
        return $pdf->download('spo_products_' . now()->format('Y_m_d_H_i_s') . '.pdf');
    }

    public function detailDownloadPdf(Vendor $vendor , Request $request){


        $rentedCounts = $this->getRentedItemsCount($vendor);

        $products = $vendor->products()->with([
            'category',
            'variations.options.color',
            'variations.size',
            'variations.variation',
        ]);
        $products = $products->get();

         foreach ($products as $product) {
        if ($product->variations->isEmpty()) {
            $rentedCount = $rentedCounts['product'][$product->id] ?? 0;
            $product->total_quantity = ($product->quantity ?? 0) - $rentedCount;
        } else {
            foreach ($product->variations as $variation) {
                if (!$variation->options->isEmpty()) {
                    foreach ($variation->options as $option) {
                        $vendorQty = $option->vendorQuantities()
                            ->where('vendor_id', $vendor->id)
                            ->first();
                        $option->quantity = $vendorQty?->quantity ?? 0;
                        $rentedCount = $rentedCounts['option'][$option->id] ?? 0;
                        $option->total_quantity = $option->quantity - $rentedCount;
                    }
                } else {
                    $rentedCount = $rentedCounts['variation'][$variation->id] ?? 0;
                    $variation->total_quantity = ($variation->quantity ?? 0) - $rentedCount;
                }
            }
        }
    }

        $data = [
            'products' => $products,
            'title' => 'Product Analytics Report',
            'vendor' => $vendor,
        ];
        
        $pdf = PDF::loadView('admin.spoProduct.detailpdf', $data);
        return $pdf->download('product-analytics-report.pdf');

    }
    public function detailDownloadCSV($vendorId)
    {
        $vendor = Vendor::findOrFail($vendorId);

        return Excel::download(
            new SPOProductsExport($vendor), 
            'vendor-'.$vendor->id.'-products-'.now()->format('Y-m-d').'.csv'
        );
    }

    public function getAvailableProducts(Request $request)
    {
        $vendor = Vendor::where('id',$request->vendor_id)->first();
        
        $allProducts = Product::with(['homePhoto'])
            ->get();
        
        $enabledProductIds = $vendor->products()->pluck('products.id')->toArray();
        $products = $allProducts->map(function($product) use ($enabledProductIds) {
            return [
                'id' => $product->id,
                'name' => $product->name,
                'image' => $product->homePhoto ? asset('storage/' . $product->homePhoto->image_path) : asset('frontend/images/placeholder.png'),
                'selected' => in_array($product->id, $enabledProductIds)
            ];
        });
     
        return response()->json([
            'products' => $products
        ]);
    }

    public function updateSelectedProducts(Request $request)
    {
        $vendor = Vendor::where('id',$request->vendor_id)->first();
        $productIds = $request->input('product_ids', []);
    
        // Sync products
        $vendor->products()->sync($productIds);
    
        // --- 1. Get all variation options for selected products ---
        $selectedOptionIds = ProductVariationOption::whereHas('variation', function ($q) use ($productIds) {
            $q->whereIn('product_id', $productIds);
        })->pluck('id','sku');
    
        // --- 2. Remove quantities for products that are no longer selected ---
        VendorProductQuantity::where('vendor_id', $vendor->id)
            ->whereNotIn('product_variation_option_id', $selectedOptionIds->values())
            ->delete();
    
        // --- 3. Ensure quantities exist for newly selected products ---
        foreach ($selectedOptionIds as $sku => $optionId) {
            VendorProductQuantity::firstOrCreate(
                [
                    'vendor_id' => $vendor->id,
                    'product_variation_option_id' => $optionId,
                ],
                [
                    'quantity' => 0,
                    'sku' => $sku
                ]
            );
        }
    
        return response()->json(['success' => true]);
    }

    public function downloadQuantityTemplate(Request $request)
    {
        $vendorId = $request->query('vendor_id');
        $vendor = Vendor::where('id',$vendorId)->first();
        $vendorName = Str::slug($vendor->first_name . '_' . $vendor->last_name, '_');
        
        // Get all products and variations available to this vendor
        $vendorProducts = VendorProductQuantity::with(['variationOption.variation.product', 'variationOption.color'])
            ->where('vendor_id', $vendorId)
            ->get();
        
            $fileName = "quantity_template_{$vendorName}_{$vendorId}_" . date('Y-m-d') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename=' . $fileName,
        ];
        
        $callback = function() use ($vendorProducts) {
            $file = fopen('php://output', 'w');
            
            // CSV header
            fputcsv($file, [
                'product_id', 
                'product_name', 
                'variation_id', 
                'variation_name', 
                'color', 
                'size', 
                'total_quantity',
                'new_quantity'
            ]);
            
            foreach ($vendorProducts as $vendorProduct) {
                $option = $vendorProduct->variationOption;
                $variation = $option->variation;
                $product = $variation->product;
                
                fputcsv($file, [
                    $product->id,
                    $product->name,
                    $option->id,
                    $variation->size ? $variation->size->name : ($variation->variation ? $variation->variation->name : 'Default'),
                    $option->color ? $option->color->name : 'Default',
                    $variation->size ? $variation->size->name : 'N/A',
                    $vendorProduct->quantity,
                    0 // Placeholder for new quantity
                ]);
            }
            
            fclose($file);
        };
        
        return Response::stream($callback, 200, $headers);
    }

    public function destroy(Request $request)
{
    try {
        $request->validate([
            'vendor_id' => 'required|exists:vendors,id',
            'product_id' => 'required|exists:products,id',
        ]);

        $vendorId  = $request->input('vendor_id');
        $productId = $request->input('product_id');

        $vendor = Vendor::findOrFail($vendorId);

        // 1. Detach product from vendor pivot
        $vendor->products()->detach($productId);

        $selectedOptionIds = ProductVariationOption::whereHas('variation', function ($q) use ($productId) {
                $q->where('product_id', $productId);
            })
            ->pluck('id');

        VendorProductQuantity::where('vendor_id', $vendorId)
            ->whereIn('product_variation_option_id', $selectedOptionIds)
            ->delete();

        return redirect()->back()->with('success', 'Product deleted successfully.');
    } catch (\Exception $e) {
        return redirect()->back()->with('error', 'Error deleting product: ' . $e->getMessage());
    }
}


    
    public function importQuantities(Request $request)
    {
        $request->validate([
            'csv_file' => 'required|file|mimes:csv,txt'
        ]);
        
        $vendorId = $request->vendor_id;
        $file = $request->file('csv_file');
        
        try {
            $csv = Reader::createFromPath($file->getPathname(), 'r');
            $csv->setHeaderOffset(0);
            $records = $csv->getRecords();
            
            $updated = 0;
            $errors = [];
            
            foreach ($records as $index => $record) {
                // Skip empty rows
                if (empty(array_filter($record))) {
                    continue;
                }
                
                // Validate required fields
                if (empty($record['variation_id']) || !isset($record['new_quantity'])) {
                    $errors[] = "Row " . ($index + 2) . ": Missing variation_id or new_quantity";
                    continue;
                }
                
                // Validate quantity is a non-negative integer
                if (!is_numeric($record['new_quantity']) || $record['new_quantity'] < 0) {
                    $errors[] = "Row " . ($index + 2) . ": Quantity must be a non-negative number";
                    continue;
                }
                
                $variationId = $record['variation_id'];
                $quantity = (int) $record['new_quantity'];
                
                // Update or create the quantity record
                VendorProductQuantity::updateOrCreate(
                    [
                        'vendor_id' => $vendorId,
                        'product_variation_option_id' => $variationId
                    ],
                    [
                        'quantity' => $quantity
                    ]
                );
                
                $updated++;
            }
            
            if (!empty($errors)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Some quantities failed to update',
                    'errors' => $errors
                ], 422);
            }
            
            return response()->json([
                'success' => true,
                'message' => "Successfully updated {$updated} product quantities"
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error processing CSV file: ' . $e->getMessage()
            ], 500);
        }
    }

    
   


}