<?php

namespace App\Http\Controllers\Admin\User;

use App\Http\Controllers\Controller;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Auth;
use App\Exports\UsersExport;
class UserController extends Controller
{
    public function index(Request $request)
    {
        $query = User::where('id', '!=', 0);
        
        if ($request->has('search')) {
            $searchQuery = $request->input('search');
            $query->where(function($q) use ($searchQuery) {
                $q->where('first_name', 'like', "%{$searchQuery}%")
                  ->orWhere('last_name', 'like', "%{$searchQuery}%")
                  ->orWhere('email', 'like', "%{$searchQuery}%")
                  ->orWhere('street', 'like', "%{$searchQuery}%");
            });
        }
        
        $sort = $request->input('sort_by', 'desc');
        $query->orderBy('id', $sort);
        
        if ($request->has('time_filter') && $request->time_filter != 'year') {
            $timeFilter = $request->input('time_filter');
            switch ($timeFilter) {
                case 'week':
                    $query->where('created_at', '>=', now()->subWeek());
                    break;
                case 'month':
                    $query->where('created_at', '>=', now()->subMonth());
                    break;
                case 'day':
                    $query->whereDate('created_at', today());
                    
                    break;
            }
        } else {
            $query->where('created_at', '>=', now()->subYear());
        }
        $query->withCount([
            'bookings as confirmed_completed_bookings_count' => function ($q) {
                $q->whereIn('status', ['confirmed', 'completed']);
            },
            'bookings as cancelled_bookings_count' => function ($q) {
                $q->where('status', 'cancelled');
            },
        ]);

        
        $users = $query->get();
        $count = $users->count();
        
        $analytics = $this->calculateUserAnalytics($request->input('time_filter', 'year'));
        
        return view('admin.user.index', compact('users', 'count', 'analytics'));
    }
    
    protected function calculateUserAnalytics($timeRange = 'year')
    {
        $query = User::where('is_guest', '!=', 1);
        
        switch ($timeRange) {
            case 'week':
                $query->where('created_at', '>=', now()->subWeek());
                break;
            case 'month':
                $query->where('created_at', '>=', now()->subMonth());
                break;
            case 'year':
                $query->where('created_at', '>=', now()->subYear());
                break;
            default:
                $query->whereDate('created_at', today());
                break;
        }
        
        $totalUsers = User::where('is_guest', '!=', 1)->count();
           
        
        $lastYearCustomers = DB::table('bookings')
            ->select('user_id')
            ->whereBetween('created_at', [now()->subYears(2)->startOfYear(), now()->subYears(1)->endOfYear()])
            ->distinct()
            ->pluck('user_id')
            ->toArray();
        
        $thisYearCustomers = DB::table('bookings')
            ->select('user_id')
            ->whereBetween('created_at', [now()->subYear()->startOfYear(), now()->subYear()->endOfYear()])
            ->distinct()
            ->pluck('user_id')
            ->toArray();
        
        $returningCustomerCount = count(array_intersect($lastYearCustomers, $thisYearCustomers));
        $previousYearCustomerCount = count($lastYearCustomers);
        
        $returningPercentage = $previousYearCustomerCount > 0
            ? round(($returningCustomerCount / $previousYearCustomerCount) * 100)
            : 0;
        
        $currentYearStart = now()->startOfYear();
        $currentYearEnd = now()->endOfYear();
        
        $topCustomers = DB::table('bookings')
            ->select('user_id')
            ->whereBetween('created_at', [$currentYearStart, $currentYearEnd])
            ->groupBy('user_id')
            ->havingRaw('COUNT(*) > 1')
            ->pluck('user_id')
            ->toArray();
        
        $currentYearCustomers = DB::table('bookings')
            ->select('user_id')
            ->whereBetween('created_at', [$currentYearStart, $currentYearEnd])
            ->distinct()
            ->pluck('user_id')
            ->toArray();
        
        $topPercentage = count($currentYearCustomers) > 0
            ? round((count($topCustomers) / count($currentYearCustomers)) * 100)
            : 0;
        
        $lostCustomers = array_diff($lastYearCustomers, $thisYearCustomers);

        $lostPercentage = $previousYearCustomerCount > 0
            ? round((count($lostCustomers) / $previousYearCustomerCount) * 100)
            : 0;
        
        return [
            'total_users' => $totalUsers,
            'returning_percentage' => $returningPercentage,
            'top_percentage' => $topPercentage,
            'lost_percentage' => $lostPercentage,
            'time_range' => $timeRange
        ];
    }
    
    public function downloadPDF()
    {
        $users = User::where('is_guest', '!=', 1)
            ->with(['bookings' => function ($q) {
                $q->select('id', 'user_id', 'status');
            }])->get();
        $pdf = \PDF::loadView('admin.user.pdf', compact('users'))
              ->setPaper('a4', 'landscape');
        return $pdf->download('users_report.pdf');
    }
    
    
    public function downloadCSV(Request $request)
    {
        return \Excel::download(new UsersExport($request), 'users_report.csv');
    }
    

    public function search(Request $request)
    {
        $users = User::where('is_guest', '!=', 1);
        
        if ($request->has('search')) {
            $searchQuery = $request->input('search');
            $users->where(function($q) use ($searchQuery) {
                $q->where('first_name', 'like', "%{$searchQuery}%")
                  ->orWhere('last_name', 'like', "%{$searchQuery}%")
                  ->orWhere('email', 'like', "%{$searchQuery}%")
                  ->orWhere('street', 'like', "%{$searchQuery}%");
            });
        }
        
        $users = $users->get();
        
        return response()->json([
            'html' => view('admin.user.search_results', compact('users'))->render()
        ]);
    }
    
    public function addUserView()
    {
        $country_code = json_decode(File::get(public_path('country-codes.json')), true);;
        return view('admin.user.add_user', compact('country_code'));
    }

    public function addUser(Request $request)
    {
        // dd($request->all());
        try {
            $validator = Validator::make($request->all(), [
                'title' => 'required',
                'first_name' => 'required|string|max:50',
                'last_name' => 'required|string|max:50',
                'email' => 'required|email|unique:users,email',
                'password' => 'required|min:10|confirmed',
                'password_confirmation' => 'required',
                'street' => 'required|string|max:100',
                'zip_code' => 'required|string|max:20',
                'city' => 'required|string|max:50',
                'region' => 'required|string|max:50',
                'country' => 'required|string|max:50',
                'phone_number' => 'required|string|max:20',
                'country_code' => 'required|string|max:5',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            $user = User::create([
                'title' => $request->title,
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'email' => strtolower($request->email),
                'password' => Hash::make($request->password),
                'street' => $request->street,
                'zip_code' => $request->zip_code,
                'city' => $request->city,
                'region' => $request->region,
                'country' => $request->country,
                'number' => $request->phone_number,
                'country_code' => $request->country_code,
                'email_verified_at' => Carbon::now(),
            ]);

            return redirect()->route('admin.user.index')->with('success', 'User registered successfully.');
        } catch (\Exception $e) {
            Log::error('Registration Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong! Please try again later.');
        }
    }

    public function viewProfile(Request $request, User $user)
    {

        return view('admin.user.profile', compact('user'));
    }

    public function editUser(Request $request, User $user)
    {
        $country_code = json_decode(File::get(public_path('country-codes.json')), true);;

        return view('admin.user.edit', compact('user', 'country_code'));
    }

    public function updateUser(Request $request, User $user)
    {
        // dd($request->all());    
        try {
            $validator = Validator::make($request->all(), [
                'title' => 'required',
                'first_name' => 'required|string|max:50',
                'last_name' => 'required|string|max:50',
                'street' => 'required|string|max:100',
                'zip_code' => 'required|string|max:20',
                'email' => 'required',
                'city' => 'required|string|max:50',
                'region' => 'required|string|max:50',
                'country' => 'required|string|max:50',
                'country_code' => 'required|string',
                'number' => 'required|string|regex:/^\+?[0-9]*$/',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }


            $user->title = $request->title;
            $user->first_name = $request->first_name;
            $user->last_name = $request->last_name;
            $user->email = $request->email;
            $user->street = $request->street;
            $user->zip_code = $request->zip_code;
            $user->city = $request->city;
            $user->region = $request->region;
            $user->country = $request->country;
            $user->country_code = $request->country_code;
            $user->number = $request->number;


            $user->save();
            return redirect()->route('admin.user.index')->with('success', 'User updated successfully!');
        } catch (\Exception $e) {
            // dd($e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong! Please try again later.');
        }
    }

    public function deleteUser(Request $request)
    {
        try {
        $user = User::findOrFail($request->input('user_id'));
            $user->delete();

            return redirect()->route('admin.user.index')->with('success', 'User deleted successfully!');
        } catch (\Exception $e) {
            dd($e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong! Please try again later.');
        }
    }

    public function updateStatus(Request $request)
    {
        $user = User::findOrFail($request->input('user_id'));
        $user->status = $request->input('new_status') === 'Active' ? 1 : 0;
        $user->save();
        // return response()->json(['status' => 'success', 'message' => 'User status updated successfully!']);

        return redirect()->back()->with('success', 'User status updated successfully!');
    }



   
}
