<?php

namespace App\Http\Controllers\Admin\Voucher;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\Voucher;
use App\Models\Vendor;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;

class VoucherController extends Controller
{
    public $filePath='admin.voucher';

    public function index()
    {
        $vouchers = Voucher::orderBy('created_at','desc')->get();
        return view($this->filePath.'.index',compact('vouchers'));
    }
    public function createVoucher()
    {
        $products = Product::orderBy('name')->get(['id', 'name']);
        $spos = Vendor::where('status','active')->get(['id','first_name','last_name']);
        return view($this->filePath.'.create_voucher',compact('products','spos'));
    }

    public function storeVoucher(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:50|unique:vouchers',
            'min_order_value' => 'required|numeric|min:1',
            'max_discount' => 'required|numeric|min:1',
            'voucher_value' => 'required|numeric|min:1',
            'start_date' => 'required|date',
            'expiration_date' => 'required|date|after:start_date',
            'applied_to' => 'nullable|array',
            'applied_to_spo' => 'nullable|array',
            'customer_type' => 'required|in:all,new,existing',
            'combine_with_other_vouchers' => 'required|boolean',
           'redemption_limit' => 'nullable|integer|min:1|max:50',
            'type' => 'required',
        ]);
        $validated['applied_to'] = $request->has('applied_to')
        ? json_encode($request->input('applied_to'))
        : null;


        $validated['applied_to_spo'] = $request->has('applied_to_spo')
        ? json_encode($request->input('applied_to_spo'))
        : null;

        $validated['combine_with_other_vouchers'] = $request->combination_with_other == 'YES' ? 1 : 0;
        Voucher::create($validated);

        return redirect()->route('admin.voucher.index')->with('success', 'Voucher created successfully!');
    }
    public function viewVoucher($id)
    {
        $voucher = Voucher::findOrFail($id);
        $productIds = json_decode($voucher->applied_to, true) ?? [];
        $spoIds = json_decode($voucher->applied_to_spo,true) ?? [];

        // Fetch product titles by IDs
        $products = Product::whereIn('id', $productIds)->pluck('name')->toArray();
        $spos = Vendor::whereIn('id',$spoIds)->pluck('first_name')->toArray();
        
        return view($this->filePath.'.view_voucher', compact('voucher','products','spos'));
    }
    public function editVoucher($id)
    {
        $voucher = Voucher::findOrFail($id);
        $allProducts = Product::all(['id', 'name']);
        $spos = Vendor::where('status','active')->get(['id','first_name','last_name']);
        // dd($allProducts);
        $selectedProductIds = json_decode($voucher->applied_to ?? '[]');
        $selectedSpoIds = json_decode($voucher->applied_to_spo ?? '[]');
        // dd($selectedProductIds);
        return view($this->filePath . '.edit_voucher', compact('voucher','selectedProductIds','allProducts','selectedSpoIds','spos'));
    }

    public function updateVoucher(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:255|unique:vouchers,code,' . $id,
            'min_order_value' => 'required|numeric|min:1',
            'max_discount' => 'required|numeric|min:1',
            'voucher_value' => 'required|numeric|min:1',
            'start_date' => 'required|date',
            'expiration_date' => 'required|date|after_or_equal:start_date',
            'applied_to' => 'nullable|array',
            'applied_to_spo' => 'nullable|array',
            'applied_to.*' => 'integer|exists:products,id', 
            'applied_to_spo.*' => 'integer|exists:vendors,id',
            'customer_type' => 'required|string',
            'combination_with_other' => 'required|in:YES,NO',  // adjust if needed
            'redemption_limit' => 'nullable|integer|min:1|max:50',
            'type' => 'required',
        ]);
    
        $voucher = Voucher::findOrFail($id);
    
        // Save data
        $voucher->name = $request->name;
        $voucher->code = $request->code;
        $voucher->min_order_value = $request->min_order_value;
        $voucher->max_discount = $request->max_discount;
        $voucher->voucher_value = $request->voucher_value;
        $voucher->start_date = $request->start_date;
        $voucher->expiration_date = $request->expiration_date;
        $voucher->applied_to = json_encode($request->applied_to ?? []);
        $voucher->applied_to_spo = json_encode($request->applied_to_spo ?? []);
        $voucher->customer_type = $request->customer_type;
        $voucher->combine_with_other_vouchers = $request->combination_with_other == 'YES' ? 1 : 0;
        $voucher->redemption_limit = $request->redemption_limit;
        $voucher->type = $request->type;
    
        $voucher->save();
    
        return redirect()->route('admin.voucher.index')->with('success', 'Voucher updated successfully.');
    }

    public function destroy(Request $request)
    {
        $voucher_id = $request->voucher_id;
        $voucher = Voucher::findOrFail($voucher_id);
        $voucher->delete();
    
        return redirect()->route('admin.voucher.index')
            ->with('success', 'Voucher deleted successfully.');
    }
    public function checkCode(Request $request)
{
    $query = Voucher::where('code', $request->code);
    
    // If exclude_id is provided (edit case), exclude that voucher from the check
    if ($request->has('exclude_id')) {
        $query->where('id', '!=', $request->exclude_id);
    }
    
    $exists = $query->exists();
    
    return response()->json(['exists' => $exists]);
}

public function searchVoucher(Request $request){
        
    $validated = $request->validate([
        'search' => 'nullable|string|max:255',
    ]);
    $searchQuery = $request->input('search', '');
        $vouchers = Voucher::where('name', 'like', "%{$searchQuery}%")->orWhere('code', 'like', "%{$searchQuery}%");
       
    $vouchers = $vouchers->orderBy('created_at', 'desc')->get(); 
    $count = $vouchers->count();
    return response()->json([
        'html' => view('admin.voucher.search_voucher', compact('vouchers', 'count'))->render()
 ]);

}
public function exportCSV()
{
    $vouchers = \App\Models\Voucher::all();
    $filename = 'vouchers.csv';
    $tempFile = tempnam(sys_get_temp_dir(), $filename);
    $file = fopen($tempFile, 'w');

    // CSV headings
    fputcsv($file, [
        'ID',
        'Name',
        'Code',
        'Start Date',
        'Expiration Date',
        'Min Order Value',
        'Max Discount',
        'Voucher Value',
        'Voucher Type',
        'No. Of Times Redeemed',
        'Applied To (Products)',
        'Applied To (Vendors)',
    ]);

    foreach ($vouchers as $voucher) {
        // Decode product & vendor IDs
        $productIds = json_decode($voucher->applied_to, true) ?? [];
        $vendorIds  = json_decode($voucher->applied_to_spo, true) ?? [];

        // Fetch names
        $productNames = \App\Models\Product::whereIn('id', $productIds)->pluck('name')->toArray();
        $vendorNames  = \App\Models\Vendor::whereIn('id', $vendorIds)
            ->get()
            ->map(function ($vendor) {
                return trim($vendor->first_name . ' ' . $vendor->last_name);
            })
            ->toArray();

        // Prepare lists
        $productList = !empty($productNames) ? implode(', ', $productNames) : '-';
        $vendorList  = !empty($vendorNames) ? implode(', ', $vendorNames) : '-';


        // Write row to CSV
        fputcsv($file, [
            $voucher->id,
            $voucher->name,
            $voucher->code,
            $voucher->start_date?->format('d.m.Y'),
            $voucher->expiration_date?->format('d.m.Y'),
            number_format($voucher->min_order_value, 2, ',', '.') . '€',
            number_format($voucher->max_discount, 2, ',', '.') . '€',
            number_format($voucher->voucher_value, 2, ',', '.') . '€',
            $voucher->type == '1' ? 'Fixed' : 'Percentage',
            $voucher->times_redeemed,
            $productList,
            $vendorList,
        ]);
    }

    fclose($file);

    return response()->download($tempFile, $filename, [
        'Content-Type' => 'text/csv',
    ])->deleteFileAfterSend(true);
}

public function exportPDF()
{
    $vouchers = Voucher::all();
    $pdf = Pdf::loadView('admin.voucher.voucher-pdf', compact('vouchers'));
    return $pdf->download('vouchers.pdf');
}

}
