<?php

namespace App\Http\Controllers\Frontend\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Mail\UserForgotPassword;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\URL;
use App\Mail\VerifyEmail;
use Illuminate\Support\Facades\File;


class AuthController extends Controller
{
    public function index()
    {
        return view('frontend.auth.login');
    }

    public function login(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'email' => 'required|email',
                'password' => 'required',
            ], [
                'email.required' => 'Please enter email address.',
                'email.email' => 'Please enter valid email address.',
                'password.required' => 'Please enter password.',
               
            ]);

            if ($validator->fails()) {
                return redirect()->back()
                    ->withErrors($validator)
                    ->withInput($request->only('email', 'remember'));
            }

            $user = User::where('email', strtolower($request->email))->first();

            if (!$user) {
                return redirect()->back()
                    ->withErrors(['email' => 'This email is not registered.'])
                    ->withInput($request->only('email', 'remember'));
            }

            if (!$user->email_verified_at) {
                $token = sha1($user->email);
                $user->email_verification_token = $token;
                $user->save();

                $verificationUrl = URL::temporarySignedRoute(
                    'verify.email',
                    Carbon::now()->addMinutes(60),
                    ['id' => $user->id, 'token' => $token]
                );

                Mail::to($user->email)->send(new VerifyEmail($verificationUrl , $user->first_name));
                return redirect()->back()
                    ->with('error', 'Please verify your email before logging in. A new verification link has been sent to your email.')
                    ->withInput($request->only('email', 'remember'));
            }

            if (!Auth::attempt(['email' => strtolower($request->email), 'password' => $request->password])) {
                return redirect()->back()
                    ->withErrors(['password' => 'Please enter correct password.'])
                    ->withInput($request->only('email', 'remember'));
            }

            // Check if user is active after successful authentication
            if (Auth::user()->status != 1) {
                Auth::logout();
                return redirect()->back()
                    ->with('error', 'Your account is inactive. Please contact support.')
                    ->withInput($request->only('email', 'remember'));
            }

            $timestamp = session('rental_form_timestamp');
            $fromRentNow = session('from_rent_now');
            
            if ($fromRentNow && $timestamp) {
                $minutesSince = now()->diffInMinutes(Carbon::parse($timestamp));
                
                if ($minutesSince <= 10) { 
                    session()->forget(['from_rent_now', 'rental_form_timestamp']);
                    return redirect()->route('cart.process.session');
                }
            }

            session()->forget(['from_rent_now', 'rental_form_timestamp']);

            return redirect()->route('home')
                ->with('success', 'Login successful!');

        } catch (\Exception $e) {
            Log::error('Login Error: ' . $e->getMessage());
            return redirect()->back()
                ->with('error', 'Something went wrong! Please try again later.')
                ->withInput($request->only('email', 'remember'));
        }
    }

    public function logout(Request $request)
    {
        try {
            Auth::logout();
            return redirect()->route('home')->with('success', 'Logout successful!');
        } catch (Exception $e) {
            return back()->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }

    public function forgotView()
    {
        return view('frontend.auth.forgot-password');
    }

    public function forgotPassword(Request $request)
    {
        try {
            $request->validate(['email' => 'required|email']);
            $user = User::where('email', strtolower($request->email))->first();
            if (!$user) {
                return back()->withErrors(['email' => 'Email not found!']);
            }
            if ($user->status != 1) {
                return redirect()->back()->with('error', 'Account is inactive.');
            }
            if (!$user->email_verified_at) {
                $token = sha1($user->email);
                $user->email_verification_token = $token;
                $user->save();
    
                $verificationUrl = URL::temporarySignedRoute(
                    'verify.email',
                    Carbon::now()->addMinutes(60),
                    ['id' => $user->id, 'token' => $token]
                );
    
                Mail::to($user->email)->send(new VerifyEmail($verificationUrl, $user->first_name));
                return redirect()->back()->with('error', 'Please verify your email before logging in.');
            }
            DB::table('password_reset_tokens')->where('email',  $user->email)->delete();
            
            $existingToken = DB::table('password_reset_tokens')->where('email', $user->email)->first();
            $token = Str::random(60);
            
            if ($existingToken && Carbon::parse($existingToken->created_at)->diffInMinutes(Carbon::now()) <= 15) {
                $token = $existingToken->token;
            } else {
                DB::table('password_reset_tokens')->updateOrInsert(
                    ['email' => $user->email],
                    ['token' => $token, 'created_at' => now()]
                );
            }
            
            
            try {
                Mail::to($user->email)->send(new UserForgotPassword($token, $user->first_name, $user->last_name));
                return back()->with('success', 'Password reset link sent successfully. Please check your email');
            } catch (\Exception $e) {
                Log::error('Email sending failed: ' . $e->getMessage());
                return back()->with('error', 'An error occurred: ' . $e->getMessage());
            }
        } catch (Exception $e) {
            return back()->withErrors(['email' => 'An error occurred: ' . $e->getMessage()]);
        }
    }

    public function resetPasswordView(Request $request)
    {
     $usertoken = $request->query('token');
        return view('frontend.auth.reset-password',compact('usertoken'));
    }

    public function resetPassword(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'password' => 'required|min:8',
                'token' => 'required',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            $resetRecord = DB::table('password_reset_tokens')->where('token', $request->token)->first();
            if (!$resetRecord) {
                return redirect()->back()->with('error','Invalid token.');
            }

            $user = User::where('email', $resetRecord->email)->first();
            if (!$user) {
                return redirect()->back()->with('error','Opps something went wrong.');
            }

            $user->update(['password' => Hash::make($request->password)]);
            DB::table('password_reset_tokens')->where('email', $resetRecord->email)->delete();

            return redirect()->route('login')->with('success', 'Password reset successfully.');
        } catch (Exception $e) {
            return back()->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }

    public function registerView()
    {
        $country_code = json_decode(File::get(public_path('country-codes.json')), true);
        return view('frontend.auth.register',compact('country_code'));
    }

    public function register(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'title' => 'required',
                'first_name' => 'required|string|max:50|min:2',
                'last_name' => 'required|string|max:50|min:2',
                'email' => 'required|email',
                'password' => [
                    'required',
                    'min:8',
                    'confirmed',
                    'regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[^A-Za-z0-9])([^\s]){8,}$/'
                ],
                'street' => 'required|string|max:100',
                'zip_code' => 'required|string|max:20',
                'phone_number' => 'required|numeric|digits_between:8,15',
                'phone_code' => 'required',
                'city' => 'required|string|max:50',
                'region' => 'required|string|max:50',
            ], [
                'title.required' => 'Please select a title.',
                'first_name.required' => 'Please enter your first name.',
                'first_name.min' => 'First name must be at least 2 characters.',
                'first_name.max' => 'First name must be less than 50 characters.',
                'last_name.required' => 'Please enter your surname.',
                'last_name.min' => 'Surname must be at least 2 characters.',
                'last_name.max' => 'Surname must be less than 50 characters.',
                'email.required' => 'Please enter email address.',
                'email.email' => 'Please enter valid email address.',
                'password.required' => 'Please enter password.',
                'password.min' => 'Password must be at least 8 characters.',
                'password.confirmed' => 'Password and confirm password must be the same.',
                'password.regex' => 'Password must be at least 8 characters long with uppercase, lowercase, number, special character, and no spaces.',
                'street.required' => 'Please enter your street and house number.',
                'zip_code.required' => 'Please enter your zip code.',
                'city.required' => 'Please enter your city.',
                'region.required' => 'Please enter your region or state.',
                'phone_code.required' => 'Please select phone code.',
                'phone_number.required' => 'Please enter your phone number.',
                'phone_number.numeric' => 'Please enter valid phone number.',
                'phone_number.digits_between' => 'Phone number should be between 8 to 15 digits.',
            ]);
    
            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            $user = User::where('email', strtolower($request->email))->first();

            if ($user && !$user->is_guest) {
                return redirect()->back()
                    ->withErrors(['email' => 'This email is already registered. Please login.'])
                    ->withInput();
            }

            
    
            DB::beginTransaction();

            try {
                if ($user && $user->is_guest) {
                    // Convert guest → real user
                    $user->update([
                        'title' => $request->title,
                        'first_name' => $request->first_name,
                        'last_name' => $request->last_name,
                        'password' => Hash::make($request->password),
                        'street' => $request->street,
                        'zip_code' => $request->zip_code,
                        'city' => $request->city,
                        'region' => $request->region,
                        'number' => $request->phone_number,
                        'country_code' => $request->phone_code,
                        'is_guest' => 0,
                        'email_verified_at' => null,
                    ]);
                } else {
                    // Brand new user
                    $user = User::create([
                        'title' => $request->title,
                        'first_name' => $request->first_name,
                        'last_name' => $request->last_name,
                        'email' => strtolower($request->email),
                        'password' => Hash::make($request->password),
                        'street' => $request->street,
                        'zip_code' => $request->zip_code,
                        'city' => $request->city,
                        'region' => $request->region,
                        'number' => $request->phone_number,
                        'country_code' => $request->phone_code,
                        'is_guest' => 0,
                    ]);
                }

                DB::commit();
            } catch (\Exception $e) {
                DB::rollBack();
                throw $e;
            }
    
            // Send email verification
            $token = sha1($user->email);
            $user->email_verification_token = $token;
            $user->save();
    
            $verificationUrl = URL::temporarySignedRoute(
                'verify.email',
                Carbon::now()->addMinutes(60),
                ['id' => $user->id, 'token' => $token]
            );
    
            Mail::to($user->email)->send(new VerifyEmail($verificationUrl, $user->first_name));
    
            return redirect()->route('login')->with('success', 'Registration successful! Please check your email for verification.');
        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Registration Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong! Please try again later.');
        }
    }

    public function verify(Request $request)
    {
        try {
            $user = User::where('id', $request->id)
                ->where('email_verification_token', $request->token)
                ->first();

            if (!$user) {
                return redirect()->route('login')->with('error', 'Invalid or expired verification link.');
            }

            $user->email_verified_at = now();
            $user->email_verification_token = null;
            $user->save();

            return redirect()->route('login')->with('success', 'Email verified! You can now log in.');
        } catch (\Exception $e) {
            Log::error('Registration Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong! Please try again later.');
        }
    }
}
