<?php

namespace App\Http\Controllers\Frontend\Booking;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Models\Booking;
use App\Models\BookingItem;
use App\Models\BookingUserDetail;
use App\Models\CartItem;
use Carbon\Carbon;
use App\Models\Notification;
use App\Mail\BookingCancellationMail;
use Illuminate\Support\Facades\Mail;


use App\Models\Vendor;
class BookingController extends Controller
{
    public $file_path = 'frontend.user.';

    public function index()
    {
        try {
            $user_id = Auth::id();
            
            $bookings = Booking::with(['items', 'userDetails', 'items.product.homePhoto'])
                ->where('user_id', $user_id)
                ->whereIn('status', ['confirmed', 'completed', 'cancelled'])
                // ->where('status','confirmed')
                ->orderBy('created_at', 'desc')
                ->get();
            
            $recentBookings = [];
            $pastBookings = [];

            $vendors = Vendor::get()->keyBy('id');
            $adminAddress = 'Esperantostr. 10, 70197 Stuttgart';

            // Loop through each booking and attach vendor address to each item
            foreach ($bookings as $booking) {
                foreach ($booking->items as $item) {
                    if ($item->vendor_id && isset($vendors[$item->vendor_id])) {
                        $vendor = $vendors[$item->vendor_id];
                        $item->vendor_address = trim("{$vendor->street} {$vendor->zip_code} {$vendor->city}");
                    } else {
                        $item->vendor_address = $adminAddress;
                    }
                }
            }

            // Sort bookings into recent or past
            // foreach ($bookings as $booking) {
            //     $latestItem = $booking->items->sortByDesc('dropoff_date')->first();
                
            //     if ($latestItem) {
            //         $dropoffDate = Carbon::parse($latestItem->dropoff_date);
                    
            //         if ($dropoffDate->isFuture()) {
            //             $recentBookings[] = $booking;
            //         } else {
            //             $pastBookings[] = $booking;
            //         }
            //     }
            // }

            foreach ($bookings as $booking) {
                $latestItem = $booking->items->sortByDesc(function ($item) {
                    return $item->new_dropoff_date ?? $item->dropoff_date;
                })->first();
            
                if ($latestItem) {
                    $dropoffDate = $latestItem->new_dropoff_date ?? $latestItem->dropoff_date;
                    $dropoffTime = $latestItem->dropoff_time ?? null;
            
                    // Parse date/datetime safely
                    $dropoffDateTime = Carbon::parse($dropoffDate);
            
                    // If it's just a date (no time part), add time
                    if ($dropoffTime && $dropoffDateTime->format('H:i:s') === '00:00:00') {
                        $dropoffDateTime->setTimeFromTimeString($dropoffTime);
                    }
            
                    $now = Carbon::now();
            
                    if ($dropoffDateTime->greaterThan($now)) {
                        $recentBookings[] = $booking;
                    } else {
                        $pastBookings[] = $booking;
                    }
                }
            }

            return view($this->file_path . 'booking.index', [
                'recentBookings' => $recentBookings,
                'pastBookings' => $pastBookings
            ]);
            
        } catch (\Exception $e) {
            dd($e->getMessage());
            Log::error('Booking Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong! Please try again later.');
        }
    }

    public function getBookingDetails($id)
    {
        try {
            $booking = Booking::with(['items', 'userDetails','items.product.homePhoto'])
                ->where('id', $id)
                ->where('user_id', Auth::id())
                ->firstOrFail();

            $adminAddress = 'Esperantostr. 10, 70197 Stuttgart';
            $vendors = Vendor::get()->keyBy('id');

            // Assign vendor address to each booking item
            foreach ($booking->items as $item) {
                if ($item->vendor_id && isset($vendors[$item->vendor_id])) {
                    $vendor = $vendors[$item->vendor_id];
                    $item->vendor_address = trim("{$vendor->street} {$vendor->zip_code} {$vendor->city}");
                } else {
                    $item->vendor_address = $adminAddress;
                }
            }

            $html = view($this->file_path . 'booking.partials.booking_details_modal', compact('booking'))->render();

            return response()->json([
                'success' => true,
                'html' => $html
            ]);

        } catch (\Exception $e) {
            Log::error('Booking Details Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Booking not found'
            ], 404);
        }
    }

    

    public function cancelBooking(Request $request)
    {
        try {
            $request->validate([
                'booking_id' => 'required|exists:bookings,id',
                'cancel_reason' => 'required|string',
            ]);
            
            $booking = Booking::where('id', $request->booking_id)
                ->where('user_id', Auth::id())
                ->firstOrFail();
                
            $firstItem = $booking->items->sortBy('pickup_date')->first();
            $pickupDate = Carbon::parse($firstItem->pickup_date);
            
            if ($pickupDate->diffInHours(Carbon::now()) <= 48) {
                return response()->json([
                    'success' => false,
                    'message' => __('messages.cancel_not_allowed_48')
                ]);
            }
            
            $booking->update([
                'status' => 'cancelled',
                'cancellation_reason' => $request->cancel_reason === 'Others' 
                    ? $request->cancel_reason_text 
                    : $request->cancel_reason,
                'cancelled_at' => now()
            ]);

            $booking->update(['payment_status'=> 'unpaid'] );
            
            $details = $booking->userDetails;
            if($firstItem->vendor_id){
                Notification::create([
                    'type' => 'booking_cancellation',
                    'status' => 'Booking Cancelled - User Account',
                    'user_id' => auth()->id(),
                    'customer_name' => $details?->first_name . ' ' . $details?->last_name,
                    'vendor_id' => $firstItem->vendor_id,
                    'customer_email' => $details?->email,
                    'customer_phone' => $details?->phone,
                    'data' => [
                        'booking_ID' => '#' . $booking->id,
                        'customer_reason' => $booking->cancellation_reason,
                    ],
                    'is_read' => false,
                ]);
            }else{
                Notification::create([
                    'type' => 'booking_cancellation',
                    'status' => 'Booking Cancelled - User Account',
                    'user_id' => auth()->id(),
                    'customer_name' => $details?->first_name . ' ' . $details?->last_name,
                    'customer_email' => $details?->email,
                    'customer_phone' => $details?->phone,
                    'data' => [
                        'booking_ID' => '#' . $booking->id,
                        'customer_reason' => $booking->cancellation_reason,
                    ],
                    'is_read' => false,
                ]);
            }

            $vendor = null;
            $vendorId = $firstItem->vendor_id;
            if ($vendorId) {
                $vendor = \App\Models\Vendor::find($vendorId);
            }

            $bookingItems = BookingItem::where('booking_id', $booking->id)->get();


            Mail::to($details->email)->send(new BookingCancellationMail(
                $booking,
                $details,
                $bookingItems,
                $vendor
            ));
    
            Mail::to('info@dachboxit.de')->send(new BookingCancellationMail(
                $booking,
                $details,
                $bookingItems,
                $vendor
            ));
    
            if ($vendor) {
                Mail::to($vendor->email)->send(new BookingCancellationMail(
                    $booking,
                    $details,
                    $bookingItems,
                    $vendor
                ));
            }

           

            
            
            return response()->json([
                'success' => true,
                'message' =>  __("messages.booking_cancelled_text") 
            ]);
            
        } catch (\Exception $e) {
            Log::error('Cancel Booking Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    
}