<?php

namespace App\Http\Controllers\Frontend\Cart;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Voucher;
use Carbon\Carbon;
use Hash;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Models\WorkingHours;
use App\Models\CartItem;
use App\Models\Product;
use App\Models\BookingItem;
use Illuminate\Validation\Rule;
use Session;
use App\Models\Notification;
use App\Models\Vendor;



class CartController extends Controller
{

    public $file_path = 'frontend.cart.';

    public function index()
    {
        // dd(Session::get('guest_user'));
        if (Auth::user()) {
            $user_id = Auth::id();
        } elseif (Session::has('guest_user') && isset(Session::get('guest_user')->id)) {
            $user_id = Session::get('guest_user')->id;
        } else {
            return redirect()->route('login');
        }
        try {
            $marketingAgreement = User::where('id', $user_id)->value('marketing_aggrement');
            $cartItems = CartItem::where('user_id', $user_id)
            ->where('status', 'pending')
            ->with('product')
            ->get();
        
        $taxRate = 0.19;
        $onePlus = 1 + $taxRate;
        $insuranceFlatRate = 25.00;
        
        // --- totals (gross) ---
        $grossBeforeDiscount = $cartItems->sum('calculated_rent');
        $discount            = (float) $cartItems->sum('discount'); // discount applies to total gross
        $grossTotal          = max($grossBeforeDiscount - $discount, 0.0); // this is your TOTAL
        
        // insurance gross (flat 25 per insured item) – NOT reduced by discount
        $insuranceCount = $cartItems->where('is_insured', true)->count();
        $insuranceGross = $insuranceCount * $insuranceFlatRate;
        
        // --- nets & tax ---
        $netSubtotal         = $grossTotal / $onePlus;              // “Subtotal” (net, tax-free)
        $insuranceNetTotal   = $insuranceGross / $onePlus;          // “Net Price Insurance”
        $rentalNet           = max($netSubtotal - $insuranceNetTotal, 0.0); // “Net Price” (rent only)
        $tax                 = $grossTotal - $netSubtotal;          // total VAT (includes insurance VAT)
        
        // optional: set item-level rent net/tax (discount spread only over the rental part)
        $totalRentalGrossBeforeDiscount = $cartItems->sum(function ($it) use ($insuranceFlatRate) {
            return $it->calculated_rent - ($it->is_insured ? $insuranceFlatRate : 0);
        });
        
        $cartItems->each(function ($item) use ($onePlus, $insuranceFlatRate, $discount, $totalRentalGrossBeforeDiscount) {
            $baseRentalGross = $item->calculated_rent - ($item->is_insured ? $insuranceFlatRate : 0);
            $share = $totalRentalGrossBeforeDiscount > 0 ? ($baseRentalGross / $totalRentalGrossBeforeDiscount) : 0;
            $itemRentalGrossAfterDiscount = max($baseRentalGross - ($discount * $share), 0.0);
        
            $item->net_price  = $itemRentalGrossAfterDiscount / $onePlus;         // rent net (per item)
            $item->tax_amount = $itemRentalGrossAfterDiscount - $item->net_price; // rent VAT (per item)
            // (Insurance per-item net/tax available if you need them:
            // $item->insurance_net = $item->is_insured ? ($insuranceFlatRate / $onePlus) : 0;
            // $item->insurance_tax = $item->is_insured ? ($insuranceFlatRate - $item->insurance_net) : 0;
        });
        
        // keep your address logic
        $adminAddress = 'Esperantostr. 10, 70197 Stuttgart';
        $vendors = Vendor::get()->keyBy('id');
        $cartItems->each(function ($item) use ($vendors, $adminAddress) {
            if ($item->vendor_id && isset($vendors[$item->vendor_id])) {
                $vendor = $vendors[$item->vendor_id];
                $item->vendor_address = trim("{$vendor->street} {$vendor->zip_code} {$vendor->city}");
            } else {
                $item->vendor_address = $adminAddress;
            }
        });
        
        // expose to the view
        return view($this->file_path . 'index', [
            'cartItems'      => $cartItems,
            'subtotal'       => $netSubtotal,        
            'discount'       => $discount,
            'insuranceTotal' => $insuranceNetTotal,  
            'tax'            => $tax,               
            'netPrice'       => $rentalNet,          
            'total'          => $grossTotal,   
            'marketingAgreement' => $marketingAgreement   
        ]);
        } catch (\Exception $e) {
            Log::error('Cart Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong! Please try again later.');
        }
    }

    // public function remove(Request $request)
    // {

    //     if (Auth::user()) {
    //         $user = Auth::user();
    //     } else {
    //         $user = \Illuminate\Support\Facades\Session::get('guest_user');
    //     }
    //     try {
    //         $item = CartItem::where('id', $request->item_id)
    //             ->where('user_id', $user->id)
    //             ->firstOrFail();

    //         $item->delete();

    //         return response()->json(['success' => true]);
    //     } catch (\Exception $e) {
    //         Log::error('Remove Cart Item Error: ' . $e->getMessage());
    //         return response()->json(['success' => false], 500);
    //     }
    // }

    public function remove(Request $request)
    {
        if (Auth::user()) {
            $user = Auth::user();
        } else {
            $user = \Illuminate\Support\Facades\Session::get('guest_user');
        }

        try {
            $item = CartItem::where('id', $request->item_id)
                ->where('user_id', $user->id)
                ->firstOrFail();

            // delete all children too

            // delete parent item
            $item->delete();

            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            Log::error('Remove Cart Item Error: ' . $e->getMessage());
            return response()->json(['success' => false], 500);
        }
    }

    public function applyCoupon(Request $request)
    {
        $couponCode = strtolower($request->input('coupon_code'));
        $userId = $request->input('user_id');
        $sub_total = $request->input('sub_total');
        $voucher = Voucher::where('code', $couponCode)->first();

        if (!$voucher) {
            return response()->json([
                'success' => false,
                'message' => 'Ungültiger Gutscheincode'
            ]);
        }
        if (!$voucher->is_active) {
            return response()->json([
                'success' => false,
                'message' => 'Dieser Gutschein ist derzeit nicht aktiv'
            ]);
        }
        $now = Carbon::now();
        if ($now->lt($voucher->start_date)) {
            return response()->json([
                'success' => false,
                'message' => 'Dieser Gutschein ist noch nicht gültig'
            ]);
        }

        if ($now->gt($voucher->expiration_date)) {
            return response()->json([
                'success' => false,
                'message' => 'Dieser Gutschein ist abgelaufen'
            ]);
        }

        // Check if total discount used across all users has reached max_discount
        $totalDiscountUsed = DB::table('bookings')
            ->whereRaw("JSON_UNQUOTE(JSON_EXTRACT(coupon_code, '$[0]')) = ?", [strtolower($voucher->code)])
            ->sum('discount');
        if ($totalDiscountUsed >= $voucher->max_discount) {
            $userDetails = User::find($userId);
            if(!$voucher->notification_send){

                Notification::create([
                    'type' => 'max_voucher_limit_exceed',
                    'status' => 'Voucher Limit Exceed', 
                    'user_id' => $userId,
                    'customer_name' => $userDetails->first_name . ' ' . $userDetails->last_name,
                    'customer_email' => $userDetails->email,
                    'customer_phone' => $userDetails->number,
                    'data' => [
                        'voucher_code' => $couponCode,
                        'exceeded_at' => now()->toDateTimeString(),
                    ],
                    'is_read' => false,
                ]);
    
                $voucher->notification_send = 1;
                $voucher->save();
            }

            return response()->json([
                'success' => false,
                'message' => 'Dieser Gutschein hat das maximale Rabattlimit erreicht'
            ]);
        }

        if ($userId) {
            $userDiscountUsed = DB::table('bookings')
                ->where('user_id', $userId)
                ->whereRaw("JSON_UNQUOTE(JSON_EXTRACT(coupon_code, '$[0]')) = ?", [strtoupper($voucher->code)])
                ->sum('discount');

            // Check if user has reached their personal redemption limit
            $userRedemptionCount = DB::table('bookings')
                ->where('user_id', $userId)
                ->whereRaw("JSON_UNQUOTE(JSON_EXTRACT(coupon_code, '$[0]')) = ?", [strtoupper($voucher->code)])
                ->count();

            if ($userRedemptionCount >= $voucher->redemption_limit) {
                return response()->json([
                    'success' => false,
                    'message' => 'Sie haben Ihr Nutzungslimit für diesen Gutschein erreicht'
                ]);
            }
        }


        $applicableProductIds = json_decode($voucher->applied_to, true) ?? [];
        $applicableSpos = json_decode($voucher->applied_to_spo, true) ?? [];

        // dd($applicableProductIds);

        $cartItemsQuery = CartItem::where('user_id', $userId)
            ->where('status', 'pending');


        $allCartItems = $cartItemsQuery->get();

        if (!empty($applicableProductIds)) {
            $productFilteredItems = $allCartItems->whereIn('product_id', $applicableProductIds);
            if ($productFilteredItems->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Dieser Gutschein kann auf keine Produkte in Ihrem Warenkorb angewendet werden.'
                ]);
            }
        } else {
            $productFilteredItems = $allCartItems; // no product restriction
        }

        if (!empty($applicableSpos)) {
            $spoFilteredItems = $productFilteredItems->filter(function ($item) use ($applicableSpos) {
                return in_array($item->vendor_id, $applicableSpos) || is_null($item->vendor_id);
            });

            if ($spoFilteredItems->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Dieser Gutschein kann für diesem Standort nicht angewendet werden.'
                ]);
            }
        } else {
            $spoFilteredItems = $productFilteredItems; // no SPO restriction
        }

        $applicableCartItems = $spoFilteredItems;
        
        // dd($applicableCartItems);
        if ($applicableCartItems->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'Dieser Gutschein kann auf keine Produkte in Ihrem Warenkorb angewendet werden.'
            ]);
        }

        $alreadyAppliedItems = $applicableCartItems->filter(function ($item) use ($couponCode) {
            if (empty($item->coupon_code)) {
                return false;
            }

            $appliedCoupons = json_decode($item->coupon_code, true) ?? [];
            return in_array($couponCode, $appliedCoupons);
        });

        if ($alreadyAppliedItems->isNotEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'Dieser Gutschein wurde bereits auf einen oder mehrere Artikel angewendet'
            ]);
        }


        $applicableCartTotal = $applicableCartItems->sum('calculated_rent');

        // Check minimum order value
        if ($sub_total < $voucher->min_order_value) {
            return response()->json([
                'success' => false,
                'message' => 'Mindestbestellwert für diesen Gutschein ist €' . $voucher->min_order_value .
                    ' (Ihr Bestellwert: €' . $sub_total . ')'
            ]);
        }

        // Check redemption limit
        if ($voucher->redemption_limit > 0 && $voucher->times_redeemed >= $voucher->redemption_limit) {
            return response()->json([
                'success' => false,
                'message' => 'This coupon has reached its redemption limit'
            ]);
        }

        // Check if another coupon is already applied
        $existingCoupons = CartItem::where('user_id', $userId)
            ->where('status', 'pending')
            ->whereNotNull('coupon_code')
            ->distinct('coupon_code')
            ->pluck('coupon_code');

        if ($existingCoupons->isNotEmpty()) {
            if (!$existingCoupons->contains($couponCode)) {
                if (!$voucher->combine_with_other_vouchers) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Ein anderer Gutschein ist bereits angewendet und dieser Gutschein kann nicht kombiniert werden'
                    ]);
                }

                // Check if the existing coupons allow combining
                $existingVouchers = Voucher::whereIn('code', $existingCoupons)->get();
                foreach ($existingVouchers as $existingVoucher) {
                    if (!$existingVoucher->combine_with_other_vouchers) {
                        return response()->json([
                            'success' => false,
                            'message' => 'Der bereits angewendete Gutschein kann nicht mit anderen Gutscheinen kombiniert werden'
                        ]);
                    }
                }
            }
        }
        // ... [previous validation checks remain the same] ...

        // Get ALL cart items (not just applicable ones)
        $allCartItems = CartItem::where('user_id', $userId)
            ->where('status', 'pending')
            ->get();

        // Get applicable cart items based on voucher's applied_to
        // $applicableProductIds = json_decode($voucher->applied_to, true) ?? [];
        // $applicableCartItems = $allCartItems->filter(function ($item) use ($applicableProductIds) {
        //     return in_array($item->product_id, $applicableProductIds);
        // });

        // Check if coupon applies to ALL items in cart
        $appliesToAllItems = $allCartItems->count() === $applicableCartItems->count();

        // Determine which total to use for discount calculation
        if ($appliesToAllItems) {
            // Use full cart subtotal if coupon applies to all items
            $discountBaseAmount = $sub_total;
        } else {
            // Otherwise use sum of only applicable items
            $discountBaseAmount = $applicableCartItems->sum('calculated_rent');
        }

        $remainingDiscount = $voucher->max_discount - $totalDiscountUsed;
        // Distribute discount to applicable items
        if ($applicableCartItems->isNotEmpty()) {

            $discountPerItem = [];
            foreach ($applicableCartItems as $item) {
                if ($voucher->type == 1) {
                    $potentialDiscount = $voucher->voucher_value;
                    // Fixed amount discount
                    // $discount = min($voucher->max_discount, $item->calculated_rent);
                } else {
                    // Percentage discount
                    $potentialDiscount = $discountBaseAmount * ($voucher->voucher_value / 100);
                    // $percentageDiscount = $item->calculated_rent * ($voucher->max_discount / 100);
                    // $discount = min($percentageDiscount, $voucher->max_discount);
                }
                // dd($potentialDiscount,$remainingDiscount);
                $totalDiscount = min($potentialDiscount, $remainingDiscount);
                $discountPerItem[$item->id] = $totalDiscount;
            }

        } else {
            if ($voucher->type == 1) {
                // Fixed amount discount
                $potentialDiscount = $voucher->voucher_value;
                // $discount = min($voucher->max_discount, $discountBaseAmount??$sub_total);
            } else {
                // Percentage discount
                $potentialDiscount = $sub_total * ($voucher->voucher_value / 100);
                // $percentageDiscount = $discountBaseAmount??$sub_total * ($voucher->max_discount / 100);
                // $discount = min($percentageDiscount, $voucher->max_discount);
            }

            $discountPerItem = $potentialDiscount;
        }


        // Ensure we don't exceed the remaining available discount
        $totalDiscount = min($potentialDiscount, $remainingDiscount);


        // Update only applicable cart items with discount
        $this->updateCartWithDiscount($applicableCartItems, $couponCode, $discountPerItem);

        // Increment times redeemed
        // $voucher->increment('times_redeemed');

        // Calculate full cart total for response
        $fullCartTotal = $sub_total;
        $totalDiscount = $applicableCartItems->sum('discount');
        $fullCartTotal = floatval(str_replace(['€', ','], ['', '.'], $fullCartTotal));

        // dd($fullCartTotal, $totalDiscount);
        return response()->json([
            'success' => true,
            'message' => 'Gutschein erfolgreich angewendet.!',
            'discount' => $totalDiscount,
            'new_total' => $fullCartTotal - $totalDiscount,
            'applicable_products_total' => $applicableCartTotal
        ]);
    }

    private function updateCartWithDiscount($cartItems, $couponCode, $discountPerItem)
    {
        // First check if coupon is already applied to ANY item in the cart
        $alreadyApplied = collect($cartItems)->contains(function ($item) use ($couponCode) {
            $appliedCoupons = json_decode($item->coupon_code, true) ?? [];
            return in_array($couponCode, $appliedCoupons);
        });

        foreach ($cartItems as $item) {
            $appliedCoupons = json_decode($item->coupon_code, true) ?? [];

            // Skip if coupon is already applied (either to this item or another item)
            if (in_array($couponCode, $appliedCoupons) || $alreadyApplied) {
                continue;
            }

            // Get the discount amount for this specific item
            $itemDiscount = is_array($discountPerItem)
                ? ($discountPerItem[$item->id] ?? 0)
                : $discountPerItem;

            // Add the coupon to this item's applied coupons
            $newAppliedCoupons = array_merge($appliedCoupons, [$couponCode]);

            $item->update([
                'coupon_code' => json_encode($newAppliedCoupons),
                'discount' => ($item->discount ?? 0) + $itemDiscount,
                'updated_at' => now()
            ]);

            // Mark as applied to prevent applying to other items
            $alreadyApplied = true;
        }
    }
    public function add(Request $request)
    {

        $rules = [
            'product_id' => 'required|exists:products,id',
            'product_name' => 'required|string',
            'manufacturer' => 'nullable|string',
            'car_year' => 'nullable|string',
            'model' => 'nullable|string',
            'trailer_hitch' => 'nullable|string',
            'roof_type' => 'nullable|string',
            'rental_from' => 'required|date',
            'rental_to' => 'required|date|after_or_equal:rental_from',
            'days' => 'required|integer|min:1',
            'calculated_rent' => 'required|numeric|min:0',
            'is_carrier_included' => 'boolean',
            'is_insured' => 'boolean',
            'carrier_length' => 'nullable|string',
            'foot_sku' => 'nullable|string',
            'footkit_sku' => 'nullable|string',
            'vendor_id' => 'nullable',
            'bundle_name' => 'nullable|string',
            'booking_item_id' => 'nullable',
            'booking_item_ids' => 'nullable',
        ];
        
        $product = Product::find($request->product_id);
        $isRoofRack = $product && (strtolower($product->name) === 'roof rack' || strtolower($product->name) === 'dachträger' );
        if (!$isRoofRack) {
            $rules['color_id'] = 'required|exists:colors,id';
            $rules['color_name'] = 'required|string';
            $rules['size_id'] = 'required';
            $rules['size_name'] = 'required|string';
        } else {
            $rules['color_id'] = 'nullable';
            $rules['color_name'] = 'nullable';
            $rules['size_id'] = 'nullable';
            $rules['size_name'] = 'nullable';
        }
        
        $validated = $request->validate($rules);



       



        if (isset($request->is_guest) && !empty($request->is_guest)) {
            $guestValidator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'surname' => 'required|string|max:255',
                'email' => [
                    'required',
                    'email',
                    'max:255',
                    // Rule::unique('users', 'email'),
                ],
            ]);
            if ($guestValidator->fails()) {
                return response()->json(['status' => false, 'message' => $guestValidator->errors()->first()]);
            }
            $user = User::where('email', $request->email)->first();
            if (!$user) {
                $user = new User();
                $user->first_name = $request->name;
                $user->last_name = $request->surname;
                $user->email = $request->email;
                $user->number = $request->number;
                $user->is_guest = 1;
                $user->password = Hash::make('12345678');
                $user->street = "";
                $user->zip_code = "";
                $user->city = "";
                $user->region = "";
                $user->save();
            }
            Session::put('guest_user', $user);

        }
        if (Auth::user()) {
            $user_id = Auth::id();
        } else {
            $user_id = $user->id;
        }

        $bookingItemData = [];
        if (!empty($validated['booking_item_id'])) {
            $bookingItemData[$validated['product_name']] = (int) $validated['booking_item_id'];
        }

        if (!empty($validated['booking_item_ids'])) {
            $extraIds = json_decode($validated['booking_item_ids'], true);
            if (is_array($extraIds)) {
                $bookingItemData = array_merge($bookingItemData, $extraIds);
            }
        }

       
        $cartItemData = [
            'user_id' => $user_id,
            'product_id' => $validated['product_id'],
            'product_name' => $validated['product_name'],
            'car_name' => $validated['manufacturer'],
            'car_year' => $validated['car_year'],
            'model' => $validated['model'],
            'trailer_hitch' => $validated['trailer_hitch'] === 'yes',
            'roof_type' => $validated['roof_type'],
            'is_carrier_included' => $validated['is_carrier_included'] ?? false,
            'is_insured' => $validated['is_insured'] ?? false,
            'pickup_date' => $validated['rental_from'],
            'dropoff_date' => $validated['rental_to'],
            'days' => $validated['days'],
            'footkit_sku' => $validated['footkit_sku'],
            'foot_sku' => $validated['foot_sku'],
            'carrier_length' => $validated['carrier_length'],
            'calculated_rent' => $validated['calculated_rent'],
            'status' => 'pending',
            'bundle_name' => $validated['bundle_name'],
            'vendor_id' => $validated['vendor_id'],
            'booking_item_id' => json_encode($bookingItemData),
            'booking_request_id' => 0
        ];

        
        if (isset($validated['color_name'])) {
            $cartItemData['color'] = $validated['color_name'];
        }
        if (isset($validated['size_name'])) {
            $cartItemData['size'] = $validated['size_name'];
        }
        
        $cartItem = CartItem::create($cartItemData);

        return response()->json([
            'redirect' => route('cart.confirm', $cartItem->id)
        ]);
    }

    public function processSession()
    {
        if (session()->has('rental_form')) {
            $r = new Request(session('rental_form'));
            $response = $this->add($r);


            $data = $response->getData();
            session()->forget('rental_form');
            return redirect()->to($data->redirect);
        }

        return redirect()->back()->withErrors('No rental data found in session.');
    }

    public function confirm(CartItem $cartItem)
    {
        return view('frontend.cart.bookingslot', compact('cartItem'));
    }

    // public function getTime(Request $request)
    // {
    //     $carbonDate = \Carbon\Carbon::parse($request->date);
    //     $container = $request->container;
    //     $date = $carbonDate->toDateString();
    //     $day = $carbonDate->format('l');

    //     $from = '09:00';
    //     $to = '17:00';

    //     $workingHour = WorkingHours::where('day', $day)->first();
    //     if ($workingHour) {
    //         $from = $workingHour->open_time;
    //         $to = $workingHour->close_time;
    //     }

    //     $from = \Carbon\Carbon::parse($from)->format('H:i'); 
    //     $to = \Carbon\Carbon::parse($to)->format('H:i');     

    //     $start = \Carbon\Carbon::createFromFormat('H:i', $from);
    //     $end = \Carbon\Carbon::createFromFormat('H:i', $to);




    //     $slots = [];
    //     $slotDuration = 15;
    //     $slotGap = 15;

    //     if($container == '#from-slots'){
    //         $bookedTimes = BookingItem::whereDate('pickup_date', $date)
    //         ->get()
    //         ->flatMap(function ($item) {
    //             return [$item->pickup_time, $item->dropoff_time];
    //         })
    //         ->filter()
    //         ->map(function ($time) {
    //             return \Carbon\Carbon::parse($time)->format('H:i');
    //         })
    //         ->unique()
    //         ->values();



    //     while ($start->copy()->addMinutes($slotDuration)->lte($end)) {
    //         $slotStart = $start->format('H:i');

    //         $slotEnd = $start->copy()->addMinutes($slotDuration)->format('H:i');


    //         $slotLabel = "$slotStart - $slotEnd";

    //         $slots[] = [
    //             'label' => $slotLabel,
    //             'disabled' => $bookedTimes->contains($slotStart),
    //         ];

    //         $start->addMinutes($slotGap);
    //     }
    //     }else{
    //         $bookedTimes = BookingItem::whereDate('dropoff_date', $date)
    //         ->get()
    //         ->flatMap(function ($item) {
    //             return [$item->pickup_time, $item->dropoff_time];
    //         })
    //         ->filter()
    //         ->map(function ($time) {
    //             return \Carbon\Carbon::parse($time)->format('H:i');
    //         })
    //         ->unique()
    //         ->values();



    //     while ($start->copy()->addMinutes($slotDuration)->lte($end)) {
    //         $slotStart = $start->format('H:i');

    //         $slotEnd = $start->copy()->addMinutes($slotDuration)->format('H:i');


    //         $slotLabel = "$slotStart - $slotEnd";

    //         $slots[] = [
    //             'label' => $slotLabel,
    //             'disabled' => $bookedTimes->contains($slotStart),
    //         ];

    //         $start->addMinutes($slotGap);
    //     }
    //     }




    //     return response()->json([
    //         'from_time' => $from,
    //         'to_time' => $to,
    //         'slots' => $slots,
    //     ]);
    // }
    public function getTime(Request $request)
    {
        $carbonDate = \Carbon\Carbon::parse($request->date);
        $container = $request->container;
        $date = $carbonDate->toDateString();
        $day = $carbonDate->format('l');

        $slotDuration = 15;
        $slotGap = 15;
        $slots = [];

        $from = '09:00';
        $to = '17:00';


        if ($request->has('vendor_id') && !empty($request->vendor_id)) {
            $vendorId = $request->vendor_id;
            $carbonDay = strtolower($carbonDate->format('l')); // monday, tuesday, ...
            $dayMap = [
                'monday'    => 'mo',
                'tuesday'   => 'di',
                'wednesday' => 'mi',
                'thursday'  => 'do',
                'friday'    => 'fr',
                'saturday'  => 'sa',
                'sunday'    => 'so',
            ];

            $day = $dayMap[$carbonDay] ?? null;
            $workingHour = \App\Models\VendorWorkingHour::where('vendor_id', $vendorId)
            ->where('day', $day)
            ->first();

            if ($workingHour && !$workingHour->is_closed) {
                $from = $workingHour->open_time;
                $to = $workingHour->close_time;
            }
            $from = substr($from, 0, 5); 
            $to   = substr($to, 0, 5);

            $start = \Carbon\Carbon::createFromFormat('H:i', $from);
            $end = \Carbon\Carbon::createFromFormat('H:i', $to);


            $unavailabilities = \App\Models\DateAvailability::where('admin_id', $vendorId)
            ->whereDate('date', $date)
            ->get();

            $unavailableSlots = collect();

            foreach ($unavailabilities as $unavailability) {
                $blockStart = \Carbon\Carbon::parse($unavailability->start_time);
                $blockEnd = \Carbon\Carbon::parse($unavailability->end_time);
    
                $current = $start->copy();
                while ($current->copy()->addMinutes($slotDuration)->lte($end)) {
                    $slotStart = $current->copy();
                    $slotEnd = $current->copy()->addMinutes($slotDuration);
    
                    // Check for overlap with each unavailability block
                    foreach ($unavailabilities as $unavailability) {
                        $blockStart = \Carbon\Carbon::parse($unavailability->start_time);
                        $blockEnd = \Carbon\Carbon::parse($unavailability->end_time);
    
                        if ($slotStart->lt($blockEnd) && $slotEnd->gt($blockStart)) {
                            $unavailableSlots->push($slotStart->format('H:i'));
                            break; // no need to check further blocks
                        }
                    }
    
                    $current->addMinutes($slotGap);
                }
            }

            $bookedTimes = BookingItem::where('vendor_id', $vendorId)
            ->get()
            ->filter(fn($item) => $item->booking && $item->booking->status === 'confirmed')
            ->flatMap(function ($item) {
                $times = [];
        
                if ($item->pickup_date && $item->pickup_time) {
                    $times[] = [
                        'date' => \Carbon\Carbon::parse($item->pickup_date)->toDateString(),
                        'time' => \Carbon\Carbon::parse($item->pickup_time)->format('H:i'),
                    ];
                }
        
                if ($item->new_dropoff_date && $item->new_dropoff_time) {
                    $times[] = [
                        'date' => \Carbon\Carbon::parse($item->new_dropoff_date)->toDateString(),
                        'time' => \Carbon\Carbon::parse($item->new_dropoff_time)->format('H:i'),
                    ];
                }
                elseif ($item->dropoff_date && $item->dropoff_time) {
                    $times[] = [
                        'date' => \Carbon\Carbon::parse($item->dropoff_date)->toDateString(),
                        'time' => \Carbon\Carbon::parse($item->dropoff_time)->format('H:i'),
                    ];
                }
        
                return $times;
            })
            ->filter(fn($t) => $t['date'] === $date) 
            ->pluck('time')
            ->unique()
            ->values();

            $current = $start->copy();
            while ($current->copy()->addMinutes($slotDuration)->lte($end)) {
                $slotStart = $current->format('H:i');
                $slotEnd = $current->copy()->addMinutes($slotDuration)->format('H:i');
                $slotLabel = "$slotStart - $slotEnd";

                $isUnavailable = $unavailableSlots->contains($slotStart);
                $isBooked = $bookedTimes->contains($slotStart);

                $slots[] = [
                    'label' => $slotLabel,
                    'disabled' => $isUnavailable || $isBooked,
                ];

                $current->addMinutes($slotGap);
            }




        }else{

       

        $workingHour = WorkingHours::where('day', $day)->first();
        if ($workingHour) {
            $from = $workingHour->open_time;
            $to = $workingHour->close_time;
        }

        $from = \Carbon\Carbon::parse($from)->format('H:i');
        $to = \Carbon\Carbon::parse($to)->format('H:i');

        $start = \Carbon\Carbon::createFromFormat('H:i', $from);
        $end = \Carbon\Carbon::createFromFormat('H:i', $to);

        $unavailabilities = \App\Models\DateAvailability::where('admin_id', 1)
        ->whereDate('date', $date)
        ->get();

        $unavailableSlots = collect();

        foreach ($unavailabilities as $unavailability) {
            $blockStart = \Carbon\Carbon::parse($unavailability->start_time);
            $blockEnd = \Carbon\Carbon::parse($unavailability->end_time);

            $current = $start->copy();
            while ($current->copy()->addMinutes($slotDuration)->lte($end)) {
                $slotStart = $current->copy();
                $slotEnd = $current->copy()->addMinutes($slotDuration);

                foreach ($unavailabilities as $unavailability) {
                    $blockStart = \Carbon\Carbon::parse($unavailability->start_time);
                    $blockEnd = \Carbon\Carbon::parse($unavailability->end_time);

                    if ($slotStart->lt($blockEnd) && $slotEnd->gt($blockStart)) {
                        $unavailableSlots->push($slotStart->format('H:i'));
                        break; 
                    }
                }

                $current->addMinutes($slotGap);
            }
        }

        $bookedTimes = BookingItem::whereNull('vendor_id')
        ->get()
        ->filter(fn($item) => $item->booking && $item->booking->status === 'confirmed')
        ->flatMap(function ($item) {
            $times = [];
    
            // Always consider pickup
            if ($item->pickup_date && $item->pickup_time) {
                $times[] = [
                    'date' => \Carbon\Carbon::parse($item->pickup_date)->toDateString(),
                    'time' => \Carbon\Carbon::parse($item->pickup_time)->format('H:i'),
                ];
            }
    
            // If new dropoff exists → consider only that
            if ($item->new_dropoff_date && $item->new_dropoff_time) {
                $times[] = [
                    'date' => \Carbon\Carbon::parse($item->new_dropoff_date)->toDateString(),
                    'time' => \Carbon\Carbon::parse($item->new_dropoff_time)->format('H:i'),
                ];
            }
            // Else fallback to old dropoff
            elseif ($item->dropoff_date && $item->dropoff_time) {
                $times[] = [
                    'date' => \Carbon\Carbon::parse($item->dropoff_date)->toDateString(),
                    'time' => \Carbon\Carbon::parse($item->dropoff_time)->format('H:i'),
                ];
            }
    
            return $times;
        })
        ->filter(fn($t) => $t['date'] === $date) 
        ->pluck('time')
        ->unique()
        ->values();

        $current = $start->copy();
        while ($current->copy()->addMinutes($slotDuration)->lte($end)) {
            $slotStart = $current->format('H:i');
            $slotEnd = $current->copy()->addMinutes($slotDuration)->format('H:i');
            $slotLabel = "$slotStart - $slotEnd";

            $isUnavailable = $unavailableSlots->contains($slotStart);
            $isBooked = $bookedTimes->contains($slotStart);

            $slots[] = [
                'label' => $slotLabel,
                'disabled' => $isUnavailable || $isBooked,
            ];

            $current->addMinutes($slotGap);
        }
    }

        return response()->json([
            'from_time' => $from,
            'to_time' => $to,
            'slots' => $slots,
        ]);
    }




    public function confirmSlot(Request $request)
    {
        $request->validate([
            'cart_item_id' => 'required|exists:cart_items,id',
            'pickup_time' => ['required', 'string', 'regex:/^\d{2}:\d{2}\s*-\s*\d{2}:\d{2}$/'],
            'dropoff_time' => ['required', 'string', 'regex:/^\d{2}:\d{2}\s*-\s*\d{2}:\d{2}$/'],
            'pickup_date' => ['required'],
            'dropoff_date' => ['required']

        ]);


        [$pickupStart, $pickupEnd] = explode(' - ', $request->pickup_time);
        [$dropoffStart, $dropoffEnd] = explode(' - ', $request->dropoff_time);

        if($request->pickup_date == $request->dropoff_date){
            if ($pickupStart === $dropoffStart) {
                return response()->json(['error' => 'Drop-off time cannot be the same as pick-up time.'], 422);
            }
        }
        

        $cartItem = CartItem::find($request->cart_item_id);
        $cartItem->pickup_time = $pickupStart;
        $cartItem->pickup_end = $pickupEnd;
        $cartItem->dropoff_time = $dropoffStart;
        $cartItem->dropoff_end = $dropoffEnd;
        $cartItem->user_timing = $request->user_timing;

        $cartItem->save();

        return response()->json([
            'redirect' => route('cart.index')
        ]);
    }

    public function storeGuestSession(Request $request)
    {
        try {
            $product = Product::findOrFail($request->product_id);
            $mainImage = $product->images->firstWhere('type', 'main');

            $cartData = $request->all();
            $cartData['product_image'] = $mainImage ? $mainImage->image_path : 'default-product-image.jpg';

            // Store in session
            session()->put('guest_cart', $cartData);

            return response()->json([
                'redirect' => route('cart.guest_time-selection')
            ]);
        } catch (\Exception $e) {
            Log::error('Guest Cart Error: ' . $e->getMessage());
            return response()->json(['success' => false], 500);
        }
    }

    public function guestTimeSelection()
    {
        if (!session()->has('guest_cart')) {
            return redirect()->route('products.index');
        }

        $cartData = session('guest_cart');

        $pickupDate = $cartData['rental_from'];
        $dropoffDate = $cartData['rental_to'];
        return view('frontend.cart..guest-time-selection', [
            'pickupDate' => $pickupDate,
            'dropoffDate' => $dropoffDate
        ]);
    }

    public function getGuestTimeSlots(Request $request)
    {
        $date = $request->date;
        $type = $request->type;
        $carbonDate = \Carbon\Carbon::parse($date);
        $day = $carbonDate->format('l');

        $workingHour = WorkingHours::where('day', $day)->first();
        $from = $workingHour ? \Carbon\Carbon::parse($workingHour->open_time)->format('H:i') : '09:00';
        $to = $workingHour ? \Carbon\Carbon::parse($workingHour->close_time)->format('H:i') : '17:00';



        $start = \Carbon\Carbon::createFromFormat('H:i', $from);
        $end = \Carbon\Carbon::createFromFormat('H:i', $to);



        // Get booked times
        $bookedTimes = BookingItem::whereDate($type . '_date', $date)
            ->get()
            ->pluck($type . '_time')
            ->filter()
            ->map(fn($time) => \Carbon\Carbon::parse($time)->format('H:i'))
            ->unique()
            ->values();

        // Generate slots
        $slots = [];

        $start = \Carbon\Carbon::createFromFormat('H:i', $from);
        $end = \Carbon\Carbon::createFromFormat('H:i', $to);
        $slotDuration = 15;
        $slotGap = 30;

        while ($start->copy()->addMinutes($slotDuration)->lte($end)) {
            $slotStart = $start->format('H:i');
            $slotEnd = $start->copy()->addMinutes($slotDuration)->format('H:i');
            $slotLabel = "$slotStart - $slotEnd";

            $slots[] = [
                'label' => $slotLabel,
                'disabled' => $bookedTimes->contains($slotStart),
                'start' => $slotStart,
                'end' => $slotEnd
            ];

            $start->addMinutes($slotGap);
        }

        return response()->json([
            'slots' => $slots,
            'type' => $type
        ]);
    }

    public function storeGuestTimeSelection(Request $request)
    {
        $request->validate([
            'pickup_time' => 'required|string',
            'dropoff_time' => 'required|string'
        ]);

        $cartData = session('guest_cart');
        $cartData['pickup_time'] = explode(' - ', $request->pickup_time)[0];
        $cartData['dropoff_time'] = explode(' - ', $request->dropoff_time)[0];

        session()->put('guest_cart', $cartData);

        return redirect()->route('checkout.guest');
    }


    public function guestCheckout()
    {
        if (!session()->has('guest_cart')) {
            return redirect()->route('products.index');
        }

        return view('frontend.checkout.guest', [
            'cartData' => session('guest_cart')
        ]);
    }

    public function removeSingleCoupon(Request $request)
    {
        $item = CartItem::find($request->input('item_id'));

        if (!$item) {
            return response()->json(['success' => false, 'message' => 'Item not found']);
        }

        $coupons = json_decode($item->applied_coupons, true) ?? [];
        $remainingCoupons = [];
        $removedDiscount = 0;

        foreach ($coupons as $coupon) {
            if ($coupon['code'] !== $request->input('coupon_code')) {
                $remainingCoupons[] = $coupon;
            } else {
                $removedDiscount = $coupon['discount'];
            }
        }

        $item->update([
            'coupon_code' => !empty($remainingCoupons) ? json_encode($remainingCoupons) : null,
            'discount' => $item->total_discount - $removedDiscount,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Gutschein entfernt',
        ]);
    }

    public function count()
    {
        if (Auth::check()) {
            $userId = Auth::id();
        } elseif (Session::has('guest_user') && isset(Session::get('guest_user')->id)) {
            $userId = Session::get('guest_user')->id;
        } else {
            return response()->json(['count' => 0]);
        }

        $count = DB::table('cart_items')
            ->where('user_id', $userId)
            ->where('status', 'pending')
            ->count();

        return response()->json(['count' => $count]);
    }


}
