<?php

namespace App\Http\Controllers\Frontend\Payment;

use App\Http\Controllers\Controller;
use App\Mail\ConfirmationEmail;
use App\Mail\GuestAccountMail;
use App\Mail\InvoiceEmail;
use App\Models\Voucher;
use App\Services\InvoiceService;
use Illuminate\Http\Request;
use Log;
use Stripe\Checkout\Session;
use Stripe\PaymentIntent;
use Stripe\Stripe;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use App\Models\Booking;
use App\Models\BookingItem;
use App\Models\BookingUserDetail;
use App\Models\UserAddressDetail;
use App\Models\UserCarDetail;
use App\Models\Product;
use App\Models\CartItem;
use App\Models\Notification;
use Illuminate\Support\Str;
use App\Models\Vendor;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Hash;
use App\Mail\PreferredTimings;


class PaymentController extends Controller
{

    public function checkout(Request $request)
    {

        if(Auth::user()){
            $user = Auth::user();    
        }else{
            $user = \Illuminate\Support\Facades\Session::get('guest_user');
            if ($request->create_customer === 'on') {
                $randomPassword = Str::random(10);
            
                \App\Models\User::where('id', $user->id)->update([
                    'email' => $request->email,
                    'zip_code' => $request->zip,
                    'city' => $request->city,
                    'region' => 'Deutschland',
                    'street' => $request->address,
                    'password' => Hash::make($randomPassword),
                    'status'=> 1,
                     'is_guest' => 0,  
                ]);

                
           
                // Mail::raw("Your account has been created with email {$request->email}. Your password is: {$randomPassword}", function ($message) use ($request) {
                //     $message->to($request->email)
                //             ->subject('Your Customer Account Details');
                // });

                try {
                    Mail::to($request->email)->send(new GuestAccountMail(
                        $request->email,
                        $randomPassword
                    ));
                    
                } catch (\Exception $e) {
    
                    return back()->with('error', 'Email sending failed: ' . $e->getMessage());
                }
            }
        }

       
       
        $cartItems = CartItem::where('user_id', $user->id)
            ->where('status', 'pending')
            ->with('product')
            ->get();
    
        if ($cartItems->isEmpty()) {
            return redirect()->back()->with('error', 'Your cart is empty');
        }
    
        $taxRate = 0.19;
        $insuranceFlatRate= 25;
        $insuranceTaxTotal = 0;
        $cartItems->each(function ($item) use ($taxRate,&$insuranceTaxTotal) {
            if ($item->is_insured) {
                $item->taxable_amount = $item->calculated_rent - 25;

                $insuranceNet = 25 / (1 + $taxRate);
                $insuranceTax = $insuranceNet * $taxRate;
                
                $insuranceTaxTotal += $insuranceTax;
            } else {
                $item->taxable_amount = $item->calculated_rent;
            }
            $item->net_price = $item->taxable_amount / (1 + $taxRate);
            $item->tax_amount = $item->net_price * $taxRate;
        });
    
        $netPrice = $cartItems->sum('net_price');
        $tax = $cartItems->sum('tax_amount')  + $insuranceTaxTotal;
        $insuranceTotal = $cartItems->where('is_insured', true)->sum(function($item) {
            return 25;
        });
        $subtotal = $netPrice + $tax - $insuranceTaxTotal ;
        $discount = $cartItems->sum('discount');
        $total = ($subtotal - $discount) + $insuranceTotal;
        $subtotal = ($total+$discount) / (1 + $taxRate)  - $discount;
        $appliedCoupons = $cartItems->flatMap(function ($item) {
            return json_decode($item->coupon_code, true) ?? [];
        })->unique()->filter()->values()->toArray();


        Voucher::whereIn('code', $appliedCoupons)
        ->each(function ($voucher) use ($appliedCoupons) {
            $timesApplied = count(array_filter($appliedCoupons, fn($code) => $code === $voucher->code));
            $voucher->increment('times_redeemed');
        });

        $vendorIds = $cartItems->pluck('vendor_id')->unique();
        $vendorId = $vendorIds->first();


        $booking = Booking::create([
            'user_id' => $user->id,
            'status' => 'pending',
            'subtotal' => $subtotal,
            'discount' => $discount,
            'coupon_code' => !empty($appliedCoupons) ? json_encode($appliedCoupons) : null,
            'tax' => $tax,
            'total' => $total,
            'payment_method' => 'N/A',
            'vendor_id' => $vendorId,
            'payment_status' => 'unpaid',
        ]);
    
        BookingUserDetail::create([
            'booking_id' => $booking->id,
            'user_id' => $user->id,
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'email' => $request->email,
            'phone' => $request->phone_code . $request->phone_number,
            'address' => $request->address,
            'company_name' => $request->company_name,
            'zip' => $request->zip,
            'city' => $request->city,
            'country' => 'Deutschland',
        ]);

        $normalizedPhone =
            $this->normalizePhone($request->phone_code) .
            $this->normalizePhone($request->phone_number);

            
        $address = UserAddressDetail::where('user_id', $user->id)
            ->where('street', $request->address)
            ->where('zipcode', $request->zip)
            ->where('city', $request->city)
            ->whereRaw(
                "REPLACE(REPLACE(contact_number, ' ', ''), '+', '') = ?",
                [$normalizedPhone]
            )
            ->first();

        if (!$address) {
            UserAddressDetail::create([
                'user_id' => $user->id,
                'name' => $request->first_name . ' ' . $request->last_name,
                'street' => $request->address,
                'zipcode' => $request->zip,
                'city' => $request->city,
                'state' => $request->region ?? '',
                'company_name' => $request->company_name,
                'email' => $request->email,
                'contact_number' => '+' . $normalizedPhone,
                'is_default' => false,
            ]);
        }


        
        foreach ($cartItems as $item) {
            $bookingItemIds = json_decode($item->booking_item_id, true) ?? [];
            $mainBookingItem  =  BookingItem::create([
                'booking_id' => $booking->id,
                'user_id' => $user->id,
                'product_id' => $item->product_id,
                'product_name' => $item->product_name,
                'car_name' => $item->car_name,
                'car_year' => $item->car_year,
                'model' => $item->model,
                'trailer_hitch' => $item->trailer_hitch,
                'roof_type' => $item->roof_type,
                'color' => $item->color,
                'size' => $item->size,
                'is_carrier_included' => $item->is_carrier_included,
                'is_insured' => $item->is_insured,
                'pickup_date' => $item->pickup_date,
                'dropoff_date' => $item->dropoff_date,
                'pickup_time' => $item->pickup_time,
                'dropoff_time' => $item->dropoff_time,
                'days' => $item->days,
                'footkit_sku' => $item->footkit_sku,
                'foot_sku' => $item->foot_sku,
                'carrier_length' => $item->carrier_length,
                'calculated_rent' => $item->calculated_rent,
                'status' => 'pending',
                'user_timing' => $item->user_timing,
                'vendor_id' => $item->vendor_id,
                'booking_item_id' => $bookingItemIds[$item->product_name] ?? null, 
                'pickup_end' => $item->pickup_end,
                'dropoff_end' => $item->dropoff_end
            ]);

            if($item->is_carrier_included || $item->product_name == 'Roof Rack' || $item->product_name =='Dachträger'){
                if($item->carrier_length){
                    BookingItem::create([
                        'booking_id' => $booking->id,
                        'user_id' => $user->id,
                        'product_id' => Product::where('name', 'Traverse')->first()->id,
                        'product_name' => 'Traverse',
                        'car_name' => $item->car_name,
                        'car_year' => $item->car_year,
                        'model' => $item->model,
                        'trailer_hitch' => $item->trailer_hitch,
                        'roof_type' => $item->roof_type,
                        'color' => 'Schwarz',
                        'size' => intval(preg_replace('/[^0-9]/', '', $item->carrier_length)) / 10,
                        'is_carrier_included' => 0,
                        'is_insured' => $item->is_insured,
                        'pickup_date' => $item->pickup_date,
                        'dropoff_date' => $item->dropoff_date,
                        'pickup_time' => $item->pickup_time,
                        'dropoff_time' => $item->dropoff_time,
                        'days' => $item->days,
                        'footkit_sku' => $item->footkit_sku,
                        'foot_sku' => $item->foot_sku,
                        'carrier_length' => $item->carrier_length,
                        'calculated_rent' => 0,
                        'status' => 'pending',
                        'user_timing' => $item->user_timing,
                        'vendor_id' => $item->vendor_id,
                        'booking_item_id' => $bookingItemIds['carrier'] ?? $bookingItemIds['Carrier'] ?? null,
                        'related_item' =>  $mainBookingItem->id,
                        'is_roof_rack_product' => $item->product_name == 'Dachträger' ? 1 : 0,
                        'pickup_end' => $item->pickup_end,
                        'dropoff_end' => $item->dropoff_end
                    ]);  
                }

                if($item->foot_sku){
                    BookingItem::create([
                        'booking_id' => $booking->id,
                        'user_id' => $user->id,
                        'product_id' => Product::where('name', 'Fuß')->first()->id,
                        'product_name' => 'Fuß',
                        'car_name' => $item->car_name,
                        'car_year' => $item->car_year,
                        'model' => $item->model,
                        'trailer_hitch' => $item->trailer_hitch,
                        'roof_type' => $item->roof_type,
                        'color' => 'Schwarz',
                        'size' => $item->foot_sku,
                        'is_carrier_included' => 0,
                        'is_insured' => $item->is_insured,
                        'pickup_date' => $item->pickup_date,
                        'dropoff_date' => $item->dropoff_date,
                        'pickup_time' => $item->pickup_time,
                        'dropoff_time' => $item->dropoff_time,
                        'days' => $item->days,
                        'footkit_sku' => $item->footkit_sku,
                        'foot_sku' => $item->foot_sku,
                        'carrier_length' => $item->carrier_length,
                        'calculated_rent' => 0,
                        'status' => 'pending',
                        'user_timing' => $item->user_timing,
                        'vendor_id' => $item->vendor_id,
                        'booking_item_id' => $bookingItemIds['fuß'] ?? null,
                        'related_item' =>   $mainBookingItem->id,
                        'is_roof_rack_product' => $item->product_name == 'Dachträger' ? 1 : 0,
                        'pickup_end' => $item->pickup_end,
                        'dropoff_end' => $item->dropoff_end
                    ]);  
                }
                if($item->footkit_sku){
                    BookingItem::create([
                        'booking_id' => $booking->id,
                        'user_id' => $user->id,
                        'product_id' => Product::where('name', 'Fußkit')->first()->id,
                        'product_name' => 'Footkit',
                        'car_name' => $item->car_name,
                        'car_year' => $item->car_year,
                        'model' => $item->model,
                        'trailer_hitch' => $item->trailer_hitch,
                        'roof_type' => $item->roof_type,
                        'color' => 'Schwarz',
                        'size' => $item->footkit_sku,
                        'is_carrier_included' => 0,
                        'is_insured' => $item->is_insured,
                        'pickup_date' => $item->pickup_date,
                        'dropoff_date' => $item->dropoff_date,
                        'pickup_time' => $item->pickup_time,
                        'dropoff_time' => $item->dropoff_time,
                        'days' => $item->days,
                        'footkit_sku' => $item->footkit_sku,
                        'foot_sku' => $item->foot_sku,
                        'carrier_length' => $item->carrier_length,
                        'calculated_rent' => 0,
                        'status' => 'pending',
                        'user_timing' => $item->user_timing,
                        'vendor_id' => $item->vendor_id,
                        'booking_item_id' => $bookingItemIds['fußkit'] ?? null, 
                        'related_item' =>   $mainBookingItem->id,
                        'is_roof_rack_product' => $item->product_name == 'Dachträger' ? 1 : 0,
                        'pickup_end' => $item->pickup_end,
                        'dropoff_end' => $item->dropoff_end
                    ]);  
                }
                
            }


            UserCarDetail::updateOrCreate(
                ['user_id' => $user->id], 
                [
                    'manufacturer' => $item->car_name,
                    'year' => $item->car_year,
                    'model' => $item->model,
                    'trailer_hitch' => $item->trailer_hitch,
                ]
            );
        }


    
        Stripe::setApiKey(env('STRIPE_SECRET_KEY'));
    
        $lineItems = [];
        foreach ($cartItems as $item) {
            $lineItems[] = [
                'price_data' => [
                    'currency' => 'eur',
                    'product_data' => [
                        'name' => $item->product_name,
                    ],
                    'unit_amount' => (int)(($item->calculated_rent - $item->discount??0) * 100), 
                ],
                'quantity' => 1,
            ];
        }
    
       
    
        // $session = Session::create([
        //     'payment_method_types' => ['card', 'klarna', 'paypal', 'sepa_debit'],
        //     'line_items' => $lineItems,
        //     'mode' => 'payment',
        //     'billing_address_collection' => 'required',
        //     'customer_email' => $request->email,
        //     'metadata' => [
        //         'booking_id' => $booking->id,
        //         'user_id' => $user->id,
        //     ],
        //     'success_url' => route('checkout.success').'?session_id={CHECKOUT_SESSION_ID}',
        //     'cancel_url' => route('checkout.cancel').'?session_id={CHECKOUT_SESSION_ID}',
        // ]);

        $stripeCustomer = \Stripe\Customer::create([
            'email' => $request->email,
            'name' => $request->first_name . ' ' . $request->last_name,
            'address' => [
                'line1' => $request->address,
                'postal_code' => $request->zip,
                'city' => $request->city,
                'country' => 'DE',
            ],
        ]);

       $session = \Stripe\Checkout\Session::create([
            'payment_method_types' => [
                'card',
                'paypal',
                
                'amazon_pay',
            ],
            'line_items' => $lineItems,
            'mode' => 'payment',
            'customer' => $stripeCustomer->id, 
            'metadata' => [
                'booking_id' => $booking->id,
                'user_id' => $user->id,
            ],
            'success_url' => route('checkout.success') . '?session_id={CHECKOUT_SESSION_ID}',
            'cancel_url' => route('checkout.cancel') . '?session_id={CHECKOUT_SESSION_ID}',
        ]);

        $booking->update([
            'stripe_session_id' => $session->id,
        ]);
    
        return redirect($session->url);
    }

    public function normalizePhone($phone)
    {
        return preg_replace('/\D+/', '', $phone); // remove spaces, +, -
    }
    

    public function success(Request $request)
    {
        $session_id = $request->query('session_id');
      
        $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET_KEY'));
        
        try {
            $session = $stripe->checkout->sessions->retrieve($session_id);
            
            $booking = Booking::where('stripe_session_id', $session_id)->firstOrFail();

            if ($session->payment_status === 'paid') {
                $paymentIntent = $stripe->paymentIntents->retrieve($session->payment_intent, [
                    'expand' => ['payment_method'],
                ]);

                $actualMethod = $paymentIntent->payment_method->type;
                $booking->update([
                    'payment_status' => 'paid',
                    'status' => 'confirmed',
                    'stripe_payment_intent' => $session->payment_intent,
                    'payment_method' => $actualMethod,
                ]);

                try {
                    $invoiceService = new InvoiceService();
                    $invoice = $invoiceService->generateInvoice($booking);
                    // Handle success
                } catch (\Exception $e) {
                    // Log error and handle failure
                    Log::error('Invoice generationsdadas failed: ' . $e->getMessage());
                    // Return error response or retry
                }
                // Send email with invoice
               
                $bookingItems = BookingItem::where('booking_id', $booking->id)->get();
                $userDetails = BookingUserDetail::where('booking_id', $booking->id)->first();
    
                // Send confirmation email

                try{
                 
                    if($booking->vendor_id){
                        $vendor = Vendor::where('id',$booking->vendor_id)->first();
                        Mail::to($userDetails->email)->send(new ConfirmationEmail(
                            $booking,
                            $bookingItems,
                            $userDetails,
                            $booking->vendor_id
                        ));
                        Mail::to('info@dachboxit.de')->send(new ConfirmationEmail(
                            $booking,
                            $bookingItems,
                            $userDetails,
                            $booking->vendor_id
                        ));

                        Mail::to($vendor->email)->send(new ConfirmationEmail(
                            $booking,
                            $bookingItems,
                            $userDetails,
                            $booking->vendor_id
                        ));

                    }else{
                        
                        Mail::to($userDetails->email)->send(new ConfirmationEmail(
                            $booking,
                            $bookingItems,
                            $userDetails
                        ));
        
        
                        Mail::to('info@dachboxit.de')->send(new ConfirmationEmail(
                            $booking,
                            $bookingItems,
                            $userDetails
                        ));
                    }

                }catch (\Exception $e) {
                    Log::error('confirmation mail failed: ' . $e->getMessage());
                }
              
                try {

                    if($booking->vendor_id){
                        Mail::to($userDetails->email)->send(new InvoiceEmail(
                            $invoice['invoice_number'],
                            $invoice['data']['invoice_date'],
                            $booking,
                            $invoice['data']['issuer'],
                            $invoice['data']['recipient'],
                            $bookingItems,
                            $booking->subtotal,
                            $booking->tax,
                            $booking->total,
                            $booking->payment_method,
                            $invoice['pdf'],
                            $userDetails,
                            $booking->vendor_id
                        ));
    
                        Mail::to('info@dachboxit.de')->send(new InvoiceEmail(
                            $invoice['invoice_number'],
                            $invoice['data']['invoice_date'],
                            $booking,
                            $invoice['data']['issuer'],
                            $invoice['data']['recipient'],
                            $bookingItems,
                            $booking->subtotal,
                            $booking->tax,
                            $booking->total,
                            $booking->payment_method,
                            $invoice['pdf'],
                            $userDetails,
                            $booking->vendor_id
                        ));
                        Mail::to($vendor->email)->send(new InvoiceEmail(
                            $invoice['invoice_number'],
                            $invoice['data']['invoice_date'],
                            $booking,
                            $invoice['data']['issuer'],
                            $invoice['data']['recipient'],
                            $bookingItems,
                            $booking->subtotal,
                            $booking->tax,
                            $booking->total,
                            $booking->payment_method,
                            $invoice['pdf'],
                            $userDetails,
                            $booking->vendor_id
                        ));
                    }else{
                        Mail::to($userDetails->email)->send(new InvoiceEmail(
                            $invoice['invoice_number'],
                            $invoice['data']['invoice_date'],
                            $booking,
                            $invoice['data']['issuer'],
                            $invoice['data']['recipient'],
                            $bookingItems,
                            $booking->subtotal,
                            $booking->tax,
                            $booking->total,
                            $booking->payment_method,
                            $invoice['pdf'],
                            $userDetails
                        ));
    
                        Mail::to('info@dachboxit.de')->send(new InvoiceEmail(
                            $invoice['invoice_number'],
                            $invoice['data']['invoice_date'],
                            $booking,
                            $invoice['data']['issuer'],
                            $invoice['data']['recipient'],
                            $bookingItems,
                            $booking->subtotal,
                            $booking->tax,
                            $booking->total,
                            $booking->payment_method,
                            $invoice['pdf'],
                            $userDetails
                        ));
                    }
                   
                    // Handle success
                } catch (\Exception $e) {
                    // Log error and handle failure
                    // Return error response or retry
                }
                
    

                $vendorIds = $bookingItems->pluck('vendor_id')->filter()->unique();
                if ($booking->last_payment_link_sent_at !== null) {
                
                    if ($vendorIds->isNotEmpty()) {
                        foreach ($vendorIds as $vendorId) {
                            Notification::create([
                                'type' => 'payment_link_paid',
                                'status' => 'Booking Paid',
                                'user_id' => $booking->user_id,
                                'customer_name' => $userDetails->first_name . ' ' . $userDetails->last_name,
                                'customer_email' => $userDetails->email,
                                'customer_phone' => $userDetails->phone,
                                'vendor_id' => $vendorId,
                                'data' => [
                                    'booking_ID' => '#' . $booking->id,
                                    'payment_method' => $booking->payment_method,
                                    'paid_at' => now()->format('d.m.Y'),
                                ],
                                'is_read' => false,
                            ]);
                        }
                    } else {
                        Notification::create([
                            'type' => 'payment_link_paid',
                            'status' => 'Booking Paid',
                            'user_id' => $booking->user_id,
                            'customer_name' => $userDetails->first_name . ' ' . $userDetails->last_name,
                            'customer_email' => $userDetails->email,
                            'customer_phone' => $userDetails->phone,
                            'data' => [
                                'booking_ID' => '#' . $booking->id,
                                'payment_method' => $booking->payment_method,
                                'paid_at' => now()->format('d.m.Y'),
                            ],
                            'is_read' => false,
                        ]);
                    }
                }
try{
    if ($bookingItems->contains('user_timing', 1)) {
        $timedItem = $bookingItems->firstWhere('user_timing', 1);

        if ($vendorIds->isNotEmpty()) {
            foreach ($vendorIds as $vendorId) {
                $vendor = Vendor::where('id',$vendorId)->first();
                Notification::create([
                    'type' => 'Preferred Timings',
                    'status' => 'Preferred Timings', 
                    'user_id' => $booking->user_id,
                    'customer_name' => $userDetails->first_name . ' ' . $userDetails->last_name,
                    'customer_email' => $userDetails->email,
                    'customer_phone' => $userDetails->phone,
                    'vendor_id' => $vendorId,
                    'data' => [
                        'booking_ID' => '#' . $booking->id,
                       'pickup_date' => $timedItem->pickup_date
                            ? Carbon::parse($timedItem->pickup_date)->format('d.m.y') . ' | ' .
                            ($timedItem->pickup_time ? Carbon::parse($timedItem->pickup_time)->format('H:i') : '') .
                            ($timedItem->pickup_end ? ' - ' . Carbon::parse($timedItem->pickup_end)->format('H:i') : '')
                            : '',

                        'dropoff_date' => $timedItem->dropoff_date
                            ? Carbon::parse($timedItem->dropoff_date)->format('d.m.y') . ' | ' .
                            ($timedItem->dropoff_time ? Carbon::parse($timedItem->dropoff_time)->format('H:i') : '') .
                            ($timedItem->dropoff_end ? ' - ' . Carbon::parse($timedItem->dropoff_end)->format('H:i') : '')
                            : '',
                    ],
                    'is_read' => false,
                ]);

                // Mail::to($userDetails->email)->send(new PreferredTimings(
                //     $booking,
                //     $userDetails,
                //     $bookingItems,
                //     $vendor
                // ));
        
                // Mail::to($vendor->email)->send(new PreferredTimings(
                //     $booking,
                //     $userDetails,
                //     $bookingItems,
                //     $vendor
                // ));
                // Mail::to('cs@dachboxit.de')->send(new PreferredTimings(
                //     $booking,
                //     $userDetails,
                //     $bookingItems,
                //     $vendor
                // ));

                // Mail::to('phplead4@deftsoft.com')->send(new PreferredTimings(
                //     $booking,
                //     $userDetails,
                //     $bookingItems,
                //     $vendor
                // ));
            }
        }else{
            Notification::create([
                'type' => 'Preferred Timings',
                'status' => 'Preferred Timings', 
                'user_id' => $booking->user_id,
                'customer_name' => $userDetails->first_name . ' ' . $userDetails->last_name,
                'customer_email' => $userDetails->email,
                'customer_phone' => $userDetails->phone,
                'data' => [
                    'booking_ID' => '#' . $booking->id,
                   'pickup_date' => $timedItem->pickup_date
                            ? Carbon::parse($timedItem->pickup_date)->format('d.m.y') . ' | ' .
                            ($timedItem->pickup_time ? Carbon::parse($timedItem->pickup_time)->format('H:i') : '') .
                            ($timedItem->pickup_end ? ' - ' . Carbon::parse($timedItem->pickup_end)->format('H:i') : '')
                            : '',

                    'dropoff_date' => $timedItem->dropoff_date
                    ? Carbon::parse($timedItem->dropoff_date)->format('d.m.y') . ' | ' .
                    ($timedItem->dropoff_time ? Carbon::parse($timedItem->dropoff_time)->format('H:i') : '') .
                    ($timedItem->dropoff_end ? ' - ' . Carbon::parse($timedItem->dropoff_end)->format('H:i') : '')
                    : '',
                ],
                'is_read' => false,
            ]);
            // Mail::to($userDetails->email)->send(new PreferredTimings(
            //     $booking,
            //     $userDetails,
            //     $bookingItems,
                
            // ));
    
            // Mail::to('cs@dachboxit.de')->send(new PreferredTimings(
            //     $booking,
            //     $userDetails,
            //     $bookingItems,
                
            // ));
            // Mail::to('phplead4@deftsoft.com')->send(new PreferredTimings(
            //     $booking,
            //     $userDetails,
            //     $bookingItems,
                
            // ));

        }
    }
}catch(\Exception $e){
    dd($e->getMessage());
}
                
                CartItem::where('user_id', $booking->user_id)
                    ->where('status', 'pending')
                    ->delete();

                $bookingItems = BookingItem::where('booking_id', $booking->id)->get();
        
                foreach ($bookingItems as $item) {
                    $item->update(['status' => 'confirmed']);

                    $booking_item_id = $item->booking_item_id ; 
                    BookingItem::where('id', $booking_item_id)
                        ->update([
                            'already_booked' => 1
                        ]);
                    // if(!$booking_item_id){
                    //     $this->updateProductQuantity($item->product_id, $item->color, $item->size, -1 , $item->vendor_id);
                    // }
                }

                session()->forget(['coupon_discount', 'coupon_code','guest_user']);

                return view('frontend.checkout.order-success', [
                    'booking' => $booking,
                    'payment_intent' => $session->payment_intent,
                ]);
            }

            return redirect()->route('checkout.cancel')->with('error', 'Payment not completed');

        } catch (\Exception $e) {
            return redirect()->route('checkout.cancel')->with('error', 'Error verifying payment');
        }
    }

    protected function updateProductQuantity($productId, $colorName, $sizeName, $change,$vendorId = null)
    {
        $color = \App\Models\Color::where('name', $colorName)->first();
        if (!$color) return false;
    
        $size = \App\Models\Size::where('name', $sizeName)->first();
        $variation = \App\Models\Variation::where('name', $sizeName)->first();
    
        $option = \App\Models\ProductVariationOption::whereHas('variation', function($query) use ($productId, $size, $variation) {
                $query->where('product_id', $productId)
                      ->where(function($q) use ($size, $variation) {
                          if ($size) {
                              $q->where('size_id', $size->id);
                          }
                          if ($variation) {
                              $q->orWhere('variation_id', $variation->id);
                          }
                      });
            })
            ->where('color_id', $color->id)
            ->first();

        if (!$option) return false;

        if (!empty($vendorId)) {
            $vendorQty = \App\Models\VendorProductQuantity::where('vendor_id', $vendorId)
                ->where('product_variation_option_id', $option->id)
                ->first();
    
            if ($vendorQty) {
                $vendorQty->quantity += $change;
                $vendorQty->save();
                return true;
            }
        } else {
            $option->quantity += $change;
            $option->save();
            return true;
        }
    
        
    
        return false;
    }

    public function cancel(Request $request)
    {
        $session_id = $request->query('session_id');
    
        $booking = Booking::where('stripe_session_id', $session_id)->first();
    
        if (!$booking) {
            return redirect()->route('home')->with('error', 'Booking not found.');
        }
    
        BookingItem::where('booking_id', $booking->id)->delete();
        BookingUserDetail::where('booking_id', $booking->id)->delete();
    
        $booking->delete();
    
        return view('frontend.payment.payment-cancel');
    }


    public function successDelay(Request $request)
    {
        $session_id = $request->query('session_id');
      
        $stripe = new \Stripe\StripeClient(env('STRIPE_SECRET_KEY'));
        
        try {
            $session = $stripe->checkout->sessions->retrieve($session_id);
            
            $booking = Booking::where('delay_session_id', $session_id)->firstOrFail();

            if ($session->payment_status === 'paid') {
                $paymentIntent = $stripe->paymentIntents->retrieve($session->payment_intent, [
                    'expand' => ['payment_method'],
                ]);
                $actualMethod = $paymentIntent->payment_method->type;
               
                try {
                    $invoiceService = new InvoiceService();
                    $invoice = $invoiceService->generateInvoice($booking);
                } catch (\Exception $e) {
                    Log::error('Invoice generationsdadas failed: ' . $e->getMessage());
                }
               
                $bookingItems = BookingItem::where('booking_id', $booking->id)->get();
                $userDetails = BookingUserDetail::where('booking_id', $booking->id)->first();
                $booking->update([
                'delay_fees_paid' => 1,
                
                ]);

              
              
                
                try {
                    Mail::to($userDetails->email)->send(new InvoiceEmail(
                        $invoice['invoice_number'],
                        $invoice['data']['invoice_date'],
                        $booking,
                        $invoice['data']['issuer'],
                        $invoice['data']['recipient'],
                        $bookingItems,
                        $booking->subtotal,
                        $booking->tax,
                        $booking->total,
                        $booking->payment_method,
                        $invoice['pdf'],
                        $userDetails
                    ));

                    Mail::to('info@dachboxit.de')->send(new InvoiceEmail(
                        $invoice['invoice_number'],
                        $invoice['data']['invoice_date'],
                        $booking,
                        $invoice['data']['issuer'],
                        $invoice['data']['recipient'],
                        $bookingItems,
                        $booking->subtotal,
                        $booking->tax,
                        $booking->total,
                        $booking->payment_method,
                        $invoice['pdf'],
                        $userDetails
                    ));
                    // Handle success
                } catch (\Exception $e) {
                    // Log error and handle failure
                    dd($e->getMessage());
                    Log::error('Invoice mail failed: ' . $e->getMessage());
                    // Return error response or retry
                }
                
    

                if ($booking->last_payment_link_sent_at !== null) {
                    Notification::create([
                        'type' => 'payment_link_paid',
                        'status' => 'Booking Paid', 
                        'user_id' => $booking->user_id,
                        'customer_name' => $userDetails->first_name . ' ' . $userDetails->last_name,
                        'customer_email' => $userDetails->email,
                        'customer_phone' => $userDetails->phone,
                        'data' => [
                            'booking_ID' => '#' . $booking->id,
                            'payment_method' => $booking->payment_method,
                            'paid_at' => now()->format('d.m.Y'),
                        ],
                        'is_read' => false,
                    ]);
                }


               

                session()->forget(['coupon_discount', 'coupon_code','guest_user']);

                return view('frontend.checkout.order-success', [
                    'booking' => $booking,
                    'payment_intent' => $session->payment_intent,
                ]);
            }

            return redirect()->route('delay.cancel')->with('error', 'Payment not completed');

        } catch (\Exception $e) {
            Log::error('Checkout Success Error: ' . $e->getMessage());
            return redirect()->route('delay.cancel')->with('error', 'Error verifying payment');
        }
    }
    

   
 
    
}
