<?php

namespace App\Http\Controllers\Frontend\User;

use App\Http\Controllers\Controller;
use App\Models\MotherFile;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\UserCarDetail;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Hash;
use App\Models\UserAddressDetail;
use App\Services\MailchimpService;
use Illuminate\Support\Facades\Storage;


class UserController extends Controller
{
    public $file_path = 'frontend.user.';

    protected $mailchimp;

    public function __construct(MailchimpService $mailchimp)
    {
        $this->mailchimp = $mailchimp;
    }


    public function index()
    {
        try {
            $user = Auth::user();
            return view($this->file_path . 'profile.index', compact('user'));
        } catch (\Exception $e) {
            dd( $e->getMessage());
            Log::error('Registration Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong! Please try again later.');
        }
    }

    public function edit()
    {
        try {
            $user = Auth::user();
            return view($this->file_path . 'profile.edit', compact('user'));
        } catch (\Exception $e) {
            dd( $e->getMessage());
            Log::error('Registration Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong! Please try again later.');
        }
    }

    public function update(Request $request)
    {
        try {
            $user = auth()->user();
    
            $validated = $request->validate([
                'title' => 'required|string',
                'first_name' => 'required|string|min:2|max:15',
                'last_name' => 'required|string|min:2|max:15',
                'street' => 'required|string|min:3',
                'zip_code' => 'required|digits_between:4,6',
                'city' => 'required|string',
                'number' => 'required|string|min:6|max:15',
                'email' => 'required|email|unique:users,email,' . $user->id,
                'profile_picture' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
                'region' => 'required|string|min:3',
                'country' => 'required|string|min:3',
                'country_code' => 'required',
            ]);
    
            if ($request->hasFile('profile_picture')) {
                if ($user->profile_image) {
                    Storage::disk('public')->delete($user->profile_image);
                }
    
                $path = $request->file('profile_picture')->store('profile_images', 'public');
                $validated['profile_image'] = $path;
            }
    
            $user->update($validated);
    
            return response()->json([
                'success' => true,
                'redirect_url' => route('user.index'),
                'message' => 'Profile updated successfully!',
            ]);
            

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'errors' => $e->errors(),
            ], 422);
        }
    }

    public function checkEmail(Request $request)
    {
        $request->validate([
            'email' => 'required|email'
        ]);

        $exists = User::where('email', $request->email)
                    ->where('id', '!=', auth()->id())
                    ->exists();

        return response()->json([
            'available' => !$exists
        ]);
    }

    public function destroy(Request $request)
    {
        $user = auth()->user();
        $user->delete();

        auth()->logout();

        return response()->json(['status' => 'deleted']);
    }

    public function logout(Request $request){

       

        auth()->logout();

        return response()->json(['status' => 'logged out']);

    }

    public function changePassword(){
        return view($this->file_path . 'password.change-password');
    }

    public function updatePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required',
            'new_password' => 'required|min:8|confirmed',
        ]);
    
        if (!Hash::check($request->current_password, Auth::user()->password)) {
            return back()->withErrors(['current_password' => 'Current password is incorrect']);
        }
    
        Auth::user()->update([
            'password' => Hash::make($request->new_password)
        ]);
    
        return back()->with('success', 'Password changed successfully');
    }

    public function showCarDetails()
    {
        $car = \App\Models\UserCarDetail::where('user_id', Auth::id())->first();
        return view('frontend.user.car-details.index', compact('car'));
    }

    public function editCarDetails()
    {
        $car = UserCarDetail::where('user_id', Auth::id())->first();
        $manufacturers = MotherFile::distinct()->pluck('car_make');
        return view('frontend.user.car-details.edit', compact('car','manufacturers'));
    }

    public function updateCarDetails(Request $request)
    {
        $request->validate([
            'manufacturer' => 'required|string|max:255',
            'year' => 'required|string|max:255',
            'model' => 'required|string|max:255',
            'trailer_hitch' => 'required|in:Yes,No',
        ]);

        $car = UserCarDetail::where('user_id', Auth::id())->first();

        if ($car) {
            $car->update([
                'manufacturer' => $request->manufacturer,
                'year' => $request->year,
                'model' => $request->model,
                'trailer_hitch' => $request->trailer_hitch === 'Yes' ? 1 : 0,
            ]);
        } else {
            UserCarDetail::create([
                'user_id' => Auth::id(),
                'manufacturer' => $request->manufacturer,
                'year' => $request->year,
                'model' => $request->model,
                'trailer_hitch' => $request->trailer_hitch === 'Yes' ? 1 : 0,
                'is_default' => true,
            ]);
        }

        return redirect()->route('user.carDetails')->with('success', 'Car details updated successfully.');
    }


    public function showBillingAddress()
    {
        $user = auth()->user();
        if (!UserAddressDetail::where('user_id', auth()->id())->where('is_default', true)->exists()) {
            $latest = UserAddressDetail::where('user_id', auth()->id())->latest()->first();
            if ($latest) {
                $latest->is_default = true;
                $latest->save();
            }
        }
        $billingAddresses = $user->billingAddresses; 
    
        return view('frontend.user.billing-address.index', compact('billingAddresses'));
    }


    public function createBillingAddress()
    {
        return view('frontend.user.billing-address.create');
    }

    public function storeBillingAddress(Request $request)
    {
       
        $request->validate([
            'first_name' => 'required',
            'last_name' => 'required',
            'street' => 'required',
            'zipcode' => 'required',
            'city' => 'required',
            'state' => 'required',
            'email' => 'required|email',
            'contact_number' => 'required',
            'country_code' => 'required',
        ]);

        if ($request->has('is_default')) {
            UserAddressDetail::where('user_id', auth()->id())
                ->where('is_default', true)
                ->update(['is_default' => false]);
        }

        UserAddressDetail::create([
            'user_id' => auth()->id(),
            'name' => $request->first_name . ' ' . $request->last_name,
        'company_name' =>$request->company_name,
            'street' => $request->street,
            'zipcode' => $request->zipcode,
            'city' => $request->city,
            'state' => $request->state,
            'email' => $request->email,
            'contact_number' => $request->country_code . $request->contact_number,
            'is_default' => $request->has('is_default')
        ]);

        return redirect()->route('user.billingAddress')->with('success', 'Address added successfully.');
    }

    public function editBillingAddress($id)
    {
        $address = UserAddressDetail::where('id', $id)->where('user_id', auth()->id())->firstOrFail();
        return view('frontend.user.billing-address.edit', compact('address'));
    }

    public function updateBillingAddress(Request $request, $id)
    {
        $request->validate([
            'first_name' => 'required',
           'country_code' => 'required',
            'street' => 'required',
            'zipcode' => 'required',
            'city' => 'required',
            'state' => 'required',
            'email' => 'required|email',
            'contact_number' => 'required',
        ]);

        $address = UserAddressDetail::where('id', $id)->where('user_id', auth()->id())->firstOrFail();
        if ($request->has('is_default')) {
            UserAddressDetail::where('user_id', auth()->id())
                ->where('is_default', true)
                ->where('id', '!=', $id)
                ->update(['is_default' => false]);
        }

        $address->update([
            'name' => $request->first_name ,
            'street' => $request->street,
            'zipcode' => $request->zipcode,
            'city' => $request->city,
            'company_name' =>$request->company_name,
            'state' => $request->state,
            'email' => $request->email,
            'contact_number' => $request->country_code . ' '. $request->contact_number,
            'is_default' => $request->has('is_default'),
        ]);

        return redirect()->route('user.billingAddress')->with('success', 'Address updated successfully.');
    }

    public function showNewsLetter(){
        $email = auth()->user()->email;
        $isSubscribed = $this->mailchimp->isSubscribed($email);
        // $isSubscribed = 0;
        return view('frontend.user.newsletter.index', compact('isSubscribed'));
    }


    public function subscribe(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
        ]);
        $isSubscribed = $this->mailchimp->isSubscribed($request->email);
        if ($isSubscribed) {
            return back()->with('info', 'You are already subscribed!');
        }

        $this->mailchimp->subscribe($request->email);

        return back()->with('success', 'Subscribed successfully!');
    }



    public function destroyAddress($id)
    {
        $address = \App\Models\UserAddressDetail::findOrFail($id);

        // Optionally restrict by user:
        if ($address->user_id !== auth()->id()) {
            abort(403);
        }

        $address->delete();

        return redirect()->route('user.billingAddress')->with('success', 'Address deleted successfully.');
    }
    


    
}
