<?php

namespace App\Http\Controllers\Frontend;

use App\Models\Category;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Models\Admin;
use Carbon\Carbon;

use App\Models\Blog;
use Exception;
use Illuminate\Support\Facades\Log;
use App\Models\ContentManagement;
use App\Models\MotherFile;
use App\Models\Product;
use App\Models\ProductPrice;
use App\Models\Size;
use App\Models\Color;
use Illuminate\Support\Str;

use Illuminate\Support\Facades\DB;
use App\Models\Holidays;
use App\Models\ProductVariationOption;
use App\Models\ProductVariation;
use App\Models\WorkingHours;
use App\Models\Variation;
use App\Models\Vendor;
use App\Models\Notification;
use Illuminate\Support\Facades\Mail;
use App\Models\BookingRequest;


class WebController extends Controller
{
    public $file_path = 'frontend.pages.';
    public $file_main = '.index';

    public function getaPageData($slug)
    {
        return ContentManagement::where('slug', $slug)->first();
    }
    function normalizeProductName($name)
    {
        $synonyms = [
            'Roof Bike Rack'     => 'Roof Bike Carrier',
            'Clutch Bike Rack'   => 'Clutch Bike Carrier',
            'Dachfahrradträger' => 'Dachfahrradträger',
            'Kupplungsfahrradträger' => 'Kupplungsfahrradträger'
            // add more pairs if needed
        ];

        // Standardize spacing and casing
        $cleanName = trim(strtolower($name));

        foreach ($synonyms as $a => $b) {
            $a = strtolower($a);
            $b = strtolower($b);
            if ($cleanName === $a || $cleanName === $b) {
                return $a; // or $b, just choose one as canonical
            }
        }

        return $cleanName;
    }

    public function index()
    {
        try {
            $products = Product::with(['homePhoto', 'category'])->get();
    
            foreach ($products as $product) {
                $priceRecord = ProductPrice::where('product_name', $product->name)->first();
    
                if ($priceRecord && $priceRecord->price) {
                    $prices = $priceRecord->price;
                    $product->first_price = $prices[0] ?? null;
                } else {
                    $product->first_price = null;
                }
            }
    
            // Define your custom order
            // $order = [
            //     'Roof Box',
            //     'Rearbox',
            //     'Roof Bike Carrier',
            //     'Clutch Bike Rack',
            //     'Roof Rack',
            //     'Carrier',
            //     'Foot',
            //     'Footkit',
            // ];

            $order = [
                'Dachbox',
                'Heckbox',
                'Dachfahrradträger',
                'Kupplungsfahrradträger',
                'Dachträger',
                'Traverse',
                'Fuß',
                'Fußkit',
            ];
    
            // Sort the products by category name based on $order
            $products = $products->sortBy(function ($product) use ($order) {
                $categoryName = $product->category->name ?? '';
                $index = array_search($categoryName, $order);
                return $index !== false ? $index : PHP_INT_MAX;
            });

            
    
            // Re-index the collection after sorting
            $products = $products->values();
    
            return view($this->file_path . 'home' . $this->file_main, compact('products'));
        } catch (Exception $e) {
            dd($e->getMessage());
            Log::error('Error in home(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }

    public function about()
    {
        try {
            
            $page_type = 'about-us';
            return view($this->file_path . 'about-us' . $this->file_main, compact('page_type'));
        } catch (Exception $e) {
            Log::error('Error in index(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }
    public function workWithUs()
    {
        try {
            
            $page_type = 'work-with-us';
            return view($this->file_path . 'workWithUs' . $this->file_main, compact('page_type'));
        } catch (Exception $e) {
            Log::error('Error in index(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }

    public function product()
    {
        try {
      
            $products = Product::with(['homePhoto','category'])->get();
            foreach ($products as $product) {
                $normalizedName = $this->normalizeProductName($product->name);

                $priceRecord = ProductPrice::get()->first(function ($record) use ($normalizedName) {
                    return $this->normalizeProductName($record->product_name) === $normalizedName;
                });
               
                if ($priceRecord && $priceRecord->price) {
                    $prices = $priceRecord->price;
                    $product->first_price = $prices[1] ?? null;
                } else {
                    $product->first_price = null;
                }
            }

            // $order = [
            //     'Roof Box',
            //     'Rearbox',
            //     'Roof Bike Carrier',
            //     'Clutch Bike Rack',
            //     'Roof Rack',
            //     'Carrier',
            //     'Foot',
            //     'Footkit',
            // ];

            $order = [
                'Dachbox',
                'Heckbox',
                'Dachfahrradträger',
                'Kupplungsfahrradträger',
                'Dachträger',
                'Traverse',
                'Fuß',
                'Fußkit',
            ];
    
            // Sort the products by category name based on $order
            $products = $products->sortBy(function ($product) use ($order) {
                $categoryName = $product->category->name ?? '';
                $index = array_search($categoryName, $order);
                return $index !== false ? $index : PHP_INT_MAX;
            });
    
            // Re-index the collection after sorting
            $products = $products->values();
            
            // dd($products);
            return view($this->file_path . 'product' . $this->file_main, compact('products'));
        } catch (Exception $e) {
            Log::error('Error in product(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }




    public function productDetails($slug)
    {
        $categoryId = Category::where('slug',$slug)->pluck('id');
    
        $manufacturers = MotherFile::distinct()->pluck('car_make');
        $product = Product::with([
            'category',
            'brand',
            'variations.options.color',
            'variations',
            'images',
            'models',
            'otherInfo',
        ])->where('category_id', $categoryId)->first();
        $normalizedName = $this->normalizeProductName($product->name);

        $priceRecord = ProductPrice::get()->first(function ($record) use ($normalizedName) {
            return $this->normalizeProductName($record->product_name) === $normalizedName;
        });
        $roof_rackPrice = ProductPrice::where('product_name', 'Dachträger ergänzt zu Dachbox/ Dachfahrradträger')->first();
        if ($priceRecord && $priceRecord->price) {
            $prices = $priceRecord->price;
            $product->first_price = $prices[1] ?? null;

            $rawPrices = ($priceRecord->price);
            $dailyPrices = array_map(function ($price) {
                return floatval(str_replace(['€', ',', ' '], ['.', '.', ''], $price));
            }, $rawPrices);

            $rawRoofRack = ($roof_rackPrice->price);
            $roofRackPrices =  array_map(function ($price) {
                return floatval(str_replace(['€', ',', ' '], ['.', '.', ''], $price));
            }, $rawRoofRack);

           
        } else {
            $product->first_price = null;
        }

        $category = $product->category;

        // $vendors = Vendor::where('status','active')->get(['id', 'street','zip_code','city']);

        $vendors = Vendor::where('status', 'active')
            ->whereHas('products', function($query) use ($product) {
                $query->where('products.id', $product->id);
            })
            ->get(['id', 'street','zip_code','city']);

        $roofTypes = [
            'Normal Roof',
            'Normal roof without glass roof',
            'Normal Roof with glass roof',
            'Fixpoints',
            'Fixpoints without glass roof',
            'Fixpoints with glass roof',
            'Integrated Roof Railing',
            'Elevated Roof Rails',
            'T-Profile',
            'Raingutter'
        ];
    

        $sizeIds = is_array($category->size_ids) ? $category->size_ids : json_decode($category->size_ids, true);
        $variationIds = is_array($category->variation_ids) ? $category->variation_ids : json_decode($category->variation_ids, true);
        $colorIds = is_array($category->color_ids) ? $category->color_ids : json_decode($category->color_ids, true);
    

        $inStockVariationRecords = \App\Models\ProductVariation::where('product_id', $product->id)
            ->where('status', 'in_stock')
            ->get();
        $inStockSizeIds = $inStockVariationRecords->pluck('size_id')->filter()->unique()->toArray();
        $inStockVariationIds = $inStockVariationRecords->pluck('variation_id')->filter()->unique()->toArray();
        $sizes = $inStockSizeIds ? Size::whereIn('id', $inStockSizeIds)->get() : collect();
        $variations = $inStockVariationIds ? Variation::whereIn('id', $inStockVariationIds)->get() : collect();

        $colors = $colorIds ? Color::whereIn('id', $colorIds)->get() : collect();

        $holidays = \App\Models\Holidays::pluck('date')->toArray();

        $closedDates = \App\Models\DateAvailability::where('is_closed', true)
            ->pluck('date')
            ->map(fn($date) => $date instanceof \Carbon\Carbon ? $date->format('Y-m-d') : (string) $date)
            ->toArray();
        
        $holidays = array_unique(array_merge($holidays, $closedDates));
        $nonworkingDays = WorkingHours::whereNull('open_time')
            ->orWhereNull('close_time')
            ->pluck('day')
            ->map(fn($day) => strtolower($day)) // make lowercase for matching JS getDay()
            ->toArray();
        $relatedVariationOptionIds = DB::table('related_product_variations')
        ->where('product_id', $product->id)
        ->pluck('variation_option_id');

       

        $relatedProductIds = Product::whereIn('id', $relatedVariationOptionIds)
        ->where('id', '!=', $product->id)
        ->pluck('id')
        ->unique();

      
        $prices = ProductPrice::get()->keyBy(function ($record) {
            return $this->normalizeProductName($record->product_name);
        });
        
        $relatedProducts = Product::whereIn('id', $relatedProductIds)
            ->with(['images' => function($query) {
                $query->where('type', 'home');
            }, 'category'])
            ->limit(3)
            ->get()
            ->map(function ($relatedProduct) {
                $normalizedName = $this->normalizeProductName($relatedProduct->name);
                $priceRecord = ProductPrice::get()->first(function ($record) use ($normalizedName) {
                    return $this->normalizeProductName($record->product_name) === $normalizedName;
                });
        
                $relatedProduct->first_price = $priceRecord && $priceRecord->price 
                    ? ($priceRecord->price[1] ?? '8.00') 
                    : '8.00';
        
                return $relatedProduct;
            });
         

            
        if($product->name == 'Roof Rack' || $product->name =='Dachträger'){
            return view($this->file_path . 'product.roof_rack_detail', compact('manufacturers','product','priceRecord','colors','sizes','variations','roofTypes','dailyPrices','roofRackPrices', 'holidays','relatedProducts','nonworkingDays','vendors'));
        }else if($product->name == 'Carrier' || $product->name == 'Foot' || $product->name == 'Footkit' || $product->name == 'Traverse' || $product->name == 'Fuß' || $product->name == 'Fußkit'){
            return view($this->file_path . 'product.extra_detail', compact('manufacturers','product','priceRecord','colors','sizes','variations','roofTypes','dailyPrices','roofRackPrices', 'holidays','relatedProducts','nonworkingDays','vendors'));

        }

        return view($this->file_path . 'product.details', compact('manufacturers','product','priceRecord','colors','sizes','variations','roofTypes','dailyPrices','roofRackPrices', 'holidays','relatedProducts','nonworkingDays','vendors'));

    }

    public function getYears(Request $request)
    {
        $years = MotherFile::where('car_make', $request->manufacturer)
            ->where('model', $request->model)
            ->distinct()->pluck('type_year');
        return response()->json($years);
    }

    public function getModels(Request $request)
    {
        $models = MotherFile::where('car_make', $request->manufacturer)
            ->distinct()->pluck('model');
        return response()->json($models);
    }

    public function getRecommendation(Request $request)
    {
        $records = MotherFile::where('car_make', $request->manufacturer)
            ->where('type_year', $request->year)
            ->where('model', $request->model)
            ->get();
       
    
        $data = $records->map(function ($record) {
            return [
                'rooftype_ger' => $record->rooftype_ger,
                'rooftype_eng' => $record->rooftype_eng,
                'recommended_roofbox_size' => $record->recommended_roofbox_size,
                'carrier_length' => $record->carrier_length,
                'foot_sku' => $record->foot_sku,
                'footkit_sku' => $record->footkit_sku,
                'bundle_name' => $record->bundle_name,
            ];
        });
    
        return response()->json($data);
    }

    // public function checkAvailability(Product $product, Request $request)
    // {

    //     $colorId = $request->input('color_id');
        
    //     $variations = $product->variations()
    //         ->with(['options' => function($query) use ($colorId) {
    //             $query->where('color_id', $colorId);
    //         }])
    //         ->get();
        
    //     $availability = [];
        
    //     foreach ($variations as $variation) {
    //         $key = $variation->size_id ?? $variation->variation_id;
    //         $availability[$key] = $variation->options->sum('quantity');
    //     }
        
    //     return response()->json($availability);
    // }

    public function checkAvailability(Product $product, Request $request)
    {
        $colorId   = $request->input('color_id');
        $fromDate  = Carbon::parse($request->input('from_date'))->startOfDay();
        $toDate    = Carbon::parse($request->input('to_date'))->startOfDay();
        $colorName = $request->input('color_name');
        $vendorId  = $request->input('vendor_id');
    
        $carrierLength = $request->input('carrier_length');
        $footSku       = $request->input('foot_sku');
        $footkitSku    = $request->input('footkit_sku');
    
        $variations = $product->variations()
            ->with(['options' => function($q) use ($colorId) {
                if ($colorId) $q->where('color_id', $colorId);
            }])
            ->get();
    
        if ($vendorId !== 'admin') {
            foreach ($variations as $variation) {
                foreach ($variation->options as $option) {
                    $vendorQty = $option->vendorQuantities()->where('vendor_id', $vendorId)->first();
                    $option->quantity = $vendorQty?->quantity ?? 0;
                }
            }
        }
    
        $availability = [];
    
        $period = [];
        for ($d = $fromDate->copy(); $d->lte($toDate); $d->addDay()) {
            $period[] = $d->format('Y-m-d');
        }
    
        foreach ($variations as $variation) {
            $key = $variation->size_id ?? $variation->variation_id;
            $sizeName = Size::find($variation->size_id)?->name
                        ?? Variation::find($variation->variation_id)?->name
                        ?? '';
    
            $totalQuantity = $variation->options->sum('quantity');
    
            if ($vendorId === 'admin' && $variation->status !== 'in_stock') {
                $availability[$key] = [
                    'available_units' => 0,
                    'booking_item_ids' => []
                ];
                continue;
            }
    
            if ($vendorId !== 'admin') {
                $isDisabled = DB::table('vendor_disabled_variants')
                    ->where('vendor_id', $vendorId)
                    ->where('product_variation_id', $variation->id)
                    ->exists();
                if ($isDisabled) {
                    $availability[$key] = [
                        'available_units' => 0,
                        'booking_item_ids' => []
                    ];
                    continue;
                }
            }
    
            $query = DB::table('booking_items')
                ->join('bookings', 'booking_items.booking_id', '=', 'bookings.id')
                ->where('booking_items.product_id', $product->id)
                ->where('booking_items.color', $colorName)
                ->where('booking_items.size', $sizeName)
                ->where('booking_items.already_booked', '!=', 1)
                ->where('booking_items.damage', '!=', 1)
                ->whereIn('bookings.status', ['confirmed']); // check status on bookings table
            
            if ($vendorId !== 'admin') {
                $query->where('booking_items.vendor_id', $vendorId);
            } else {
                $query->whereNull('booking_items.vendor_id');
            }
            
            $overlappingBookings = $query
                ->whereDate(DB::raw('COALESCE(booking_items.new_dropoff_date, booking_items.dropoff_date)'), '>=', $fromDate->toDateString())
                ->whereDate('booking_items.pickup_date', '<=', $toDate->toDateString())
            ->get();
    
            $occupancy = array_fill_keys($period, 0);
            $bookingItemIds = [];
    
            foreach ($overlappingBookings as $b) {
                $bStart = Carbon::parse($b->pickup_date)->startOfDay();
                $bEnd   = Carbon::parse($b->new_dropoff_date ?? $b->dropoff_date)->startOfDay();
    
                $bQty = $b->quantity ?? 1;
    
                $cursor = $bStart->copy();
                while ($cursor->lte($bEnd)) {
                    $dateKey = $cursor->format('Y-m-d');
                    if (array_key_exists($dateKey, $occupancy)) {
                        $occupancy[$dateKey] += $bQty;
                        if (!in_array($b->id, $bookingItemIds)) {
                            $bookingItemIds[] = $b->id;
                        }
                    }
                    $cursor->addDay();
                }
            }
    
            $minAvailable = $totalQuantity;
            foreach ($occupancy as $date => $used) {
                $availableOnDate = max(0, $totalQuantity - $used);
                if ($availableOnDate < $minAvailable) $minAvailable = $availableOnDate;
            }
    
            $availability[$key] = [
                'available_units'   => $minAvailable,
                'booking_item_ids'  => [],
            ];
        }
    
        if ($carrierLength || $footSku || $footkitSku) {
            $roofRackAvailability = $this->checkAvailabilityForRoofRack($product, $request)->getData(true);
    
            $availability['roof_rack'] = [
                'carrier' => $roofRackAvailability['carrier'] ?? ['available_units' => 0, 'booking_item_ids' => []],
                'foot'    => $roofRackAvailability['foot'] ?? ['available_units' => 0, 'booking_item_ids' => []],
                'footkit' => $roofRackAvailability['footkit'] ?? ['available_units' => 0, 'booking_item_ids' => []],
                'available_quantity' => $roofRackAvailability['available_quantity'] ?? 0,
            ];
        }
    
        return response()->json($availability);
    }


    
    public function checkAvailabilityForRoofRack(Product $roofRack, Request $request)
    {
        $fromDate = Carbon::parse($request->input('from_date'))->format('Y-m-d');
        $toDate   = Carbon::parse($request->input('to_date'))->format('Y-m-d');
        $carrierLength = $request->input('carrier_length'); 
        $footSku = $request->input('foot_sku');
        $footkitSku = $request->input('footkit_sku');
        $vendorId = $request->input('vendor_id');

        $carrierProduct = Product::where('name', 'Traverse')->first();
        $footProduct = Product::where('name', 'Fuß')->first();
        $footkitProduct = Product::where('name', 'Fußkit')->first();

        if (!$carrierProduct || !$footProduct) {
            return response()->json(['error' => 'Required products not found'], 404);
        }

        $carrierAvailability = $this->runAvailabilityCheck(
            $carrierProduct,
            $vendorId,
            $fromDate,
            $toDate,
            intval(preg_replace('/[^0-9]/', '', $carrierLength)) / 10 
        );

        $footAvailability = $this->runAvailabilityCheck(
            $footProduct,
            $vendorId,
            $fromDate,
            $toDate,
            $footSku
        );
        if ($request->filled('footkit_sku')) {
            if ($footkitProduct) {
                $footkitAvailability = $this->runAvailabilityCheck($footkitProduct, $vendorId, $fromDate, $toDate, $footkitSku);
            } else {
                $footkitAvailability = ['available_units' => 0, 'booking_item_ids' => []];
            }
        } else {
            $footkitAvailability = ['available_units' => 1, 'booking_item_ids' => []];
        }

        $availability = [
            'carrier' => $carrierAvailability,
            'foot' => $footAvailability,
            'footkit' => $footkitAvailability,
            'available_quantity' => min(
                $carrierAvailability['available_units'],
                $footAvailability['available_units'],
                $footkitAvailability['available_units']
            )
        ];

        return response()->json($availability);
    }


    // private function runAvailabilityCheck(Product $product, $vendorId, $fromDate, $toDate, $sizeOrSku)
    // {
    //     $colorId = 2; 
    //     $colorName = 'Black'; 

    //     $variations = $product->variations()
    //         ->with(['options' => function ($query) use ($colorId) {
    //             $query->where('color_id', $colorId);
    //         }])
    //         ->get();

    //     if ($vendorId !== 'admin') {
    //         foreach ($variations as $variation) {
    //             foreach ($variation->options as $option) {
    //                 $vendorQty = $option->vendorQuantities()
    //                     ->where('vendor_id', $vendorId)
    //                     ->first();
    //                 $option->quantity = $vendorQty?->quantity ?? 0;
    //             }
    //         }
    //     }

    //     $targetVariation = $variations->first(function ($variation) use ($sizeOrSku) {
    //         $sizeName = Size::where('id', $variation->size_id)->first()->name
    //             ?? Variation::where('id', $variation->variation_id)->first()->name;
    //         return $sizeName == $sizeOrSku;
    //     });

    //     if (!$targetVariation) {
    //         return ['available_units' => 0, 'booking_item_ids' => []];
    //     }

    //     $quantity = $targetVariation->options->sum('quantity');
    //     $sizeName = Size::where('id', $targetVariation->size_id)->first()->name
    //         ?? Variation::where('id', $targetVariation->variation_id)->first()->name;

    //     // ADMIN LOGIC
    //     if ($vendorId === 'admin') {
    //         if ($targetVariation->status === 'in_stock') {
               
    //             $overlappingBookings = DB::table('booking_items')
    //                 ->where('product_id', $product->id)
    //                 ->where('size', $sizeName)
    //                 ->whereIn('status', ['confirmed'])
    //                 ->where('already_booked', '!=', 1)
    //                 ->where('damage', '!=', 1)
    //                 ->where(function($query) use ($fromDate, $toDate) {
    //                     $query->where(function($q) use ($fromDate, $toDate) {
    //                         $q->where('pickup_date', '<=', $toDate)
    //                         ->where(DB::raw('COALESCE(new_dropoff_date, dropoff_date)'), '>=', $fromDate);
    //                     });
    //                 })->get();
    
    
    //             $activeOverlap = count($overlappingBookings);
    
    //             $availableUnits = max(0, $quantity - $activeOverlap);
    
                
    
            
    
    //             return [
    //                 'available_units' => max($availableUnits, 0),
    //                 'booking_item_ids' => [],
    
    //             ];
    //         }
    //     }

    //     else {
            
    //         if ($targetVariation->status === 'in_stock' ) {
     
    //             $overlappingBookings = DB::table('booking_items')
    //                 ->where('product_id', $product->id)
    //                 ->where('size', $sizeName)
    //                 ->where('status', 'confirmed')
    //                 ->where('vendor_id', $vendorId)
    //                 ->where('already_booked', '!=', 1)
    //                 ->where('damage', '!=', 1)
    //                 ->where(function($query) use ($fromDate, $toDate) {
    //                     $query->where(function($q) use ($fromDate, $toDate) {
    //                         $q->where('pickup_date', '<=', $toDate)
    //                         ->where(DB::raw('COALESCE(new_dropoff_date, dropoff_date)'), '>=', $fromDate);
    //                     });
    //                 })->get();
    
    //                 $activeOverlap = count($overlappingBookings);
    
    //                 $availableUnits = max(0, $quantity - $activeOverlap);
        
                    
        
                
        
    //                 return [
    //                     'available_units' => max($availableUnits, 0),
    //                     'booking_item_ids' => [],
        
    //                 ];
    
               
    //         }

    //     }
    // }

    private function runAvailabilityCheck(Product $product, $vendorId, $fromDate, $toDate, $sizeOrSku)
    {
        $colorName = 'Schwarz'; 
        $colorId= Color::where('name',$colorName)->first()->id;

        $variations = $product->variations()
            ->with(['options' => function ($query) use ($colorId) {
                $query->where('color_id', $colorId);
            }])
            ->get();
        if ($vendorId !== 'admin') {
            foreach ($variations as $variation) {
                foreach ($variation->options as $option) {
                    $vendorQty = $option->vendorQuantities()
                        ->where('vendor_id', $vendorId)
                        ->first();
                    $option->quantity = $vendorQty?->quantity ?? 0;
                }
            }
        }

        $targetVariation = $variations->first(function ($variation) use ($sizeOrSku) {
            $sizeName = Size::find($variation->size_id)?->name
                    ?? Variation::find($variation->variation_id)?->name;
            return $sizeName == $sizeOrSku;
        });

        if (!$targetVariation) {
            return ['available_units' => 0, 'booking_item_ids' => []];
        }

        $quantity = $targetVariation->options->sum('quantity');
        $sizeName = Size::find($targetVariation->size_id)?->name
                ?? Variation::find($targetVariation->variation_id)?->name;

        if ($targetVariation->status !== 'in_stock') {
            return ['available_units' => 0, 'booking_item_ids' => []];
        }

        $from = Carbon::parse($fromDate)->startOfDay();
        $to   = Carbon::parse($toDate)->startOfDay();
        
        $period = [];
        for ($d = $from->copy(); $d->lte($to); $d->addDay()) {
            $period[] = $d->format('Y-m-d');
        }

        // $query = DB::table('booking_items')
        //     ->where('product_id', $product->id)
        //     ->where('size', $sizeName)
        //     ->where('status', 'confirmed')
        //     ->where('already_booked', '!=', 1)
        //     ->where('damage', '!=', 1);

        // if ($vendorId !== 'admin') {
        //     $query->where('vendor_id', $vendorId);
        // }else{
        //     $query->wherenull('vendor_id');
        // }

        // $overlappingBookings = $query
        //     ->whereDate(DB::raw('COALESCE(new_dropoff_date, dropoff_date)'), '>=', $from->toDateString())
        //     ->whereDate('pickup_date', '<=', $to->toDateString())
        //     ->get();


            $query = DB::table('booking_items')
            ->join('bookings', 'booking_items.booking_id', '=', 'bookings.id')
            ->where('booking_items.product_id', $product->id)
            ->where('booking_items.size', $sizeName)
            ->where('booking_items.already_booked', '!=', 1)
            ->where('booking_items.damage', '!=', 1)
            ->whereIn('bookings.status', ['confirmed']); // check status on bookings table
        
        if ($vendorId !== 'admin') {
            $query->where('booking_items.vendor_id', $vendorId);
        } else {
            $query->whereNull('booking_items.vendor_id');
        }
        
        $overlappingBookings = $query
            ->whereDate(DB::raw('COALESCE(booking_items.new_dropoff_date, booking_items.dropoff_date)'), '>=', $from->toDateString())
            ->whereDate('booking_items.pickup_date', '<=', $to->toDateString())
            ->get();


        $occupancy = array_fill_keys($period, 0);
        $bookingItemIds = [];

        foreach ($overlappingBookings as $b) {
            $bStart = Carbon::parse($b->pickup_date)->startOfDay();
            $bEnd   = Carbon::parse($b->new_dropoff_date ?? $b->dropoff_date)->startOfDay();

            $bQty = $b->quantity ?? 1;

            $cursor = $bStart->copy();
            while ($cursor->lte($bEnd)) {
                $dateKey = $cursor->format('Y-m-d');
                if (array_key_exists($dateKey, $occupancy)) {
                    $occupancy[$dateKey] += $bQty;
                    if (!in_array($b->id, $bookingItemIds)) {
                        $bookingItemIds[] = $b->id;
                    }
                }
                $cursor->addDay();
            }
        }

        $minAvailable = $quantity;
        foreach ($occupancy as $date => $used) {
            $availableOnDate = max(0, $quantity - $used);
            if ($availableOnDate < $minAvailable) {
                $minAvailable = $availableOnDate;
            }
        }

        return [
            'available_units'  => $minAvailable,
            'booking_item_ids' => [],
        ];
    }
    

    // private function checkCarrierAvailability($roofRack,$product, $carrierLength, $vendorId, $fromDate, $toDate)
    // {
       
    //     if ($vendorId == 'admin') {
    //         $variations = $product->variations()
    //             ->with(['options' => function($query) {
    //                 $query->where('color_id', 2);
    //             }])
    //             ->get();
    //     } else {
    //         $variations = $product->variations()
    //             ->with(['options' => function ($query) {
    //                 $query->where('color_id', 2);
    //             }])
    //             ->get();
    
    //         foreach ($variations as $variation) {
    //             foreach ($variation->options as $option) {
    //                 $vendorQty = $option->vendorQuantities()
    //                     ->where('vendor_id', $vendorId)
    //                     ->first();
    //                 $option->quantity = $vendorQty?->quantity ?? 0;
    //             }
    //         }
    //     }
    
    //     // Find variation that matches the carrier length (comparing as float values)
    //     $targetVariation = null;
    //     foreach ($variations as $variation) {
    //         if ((float)$variation->length == $carrierLength) {
    //             $targetVariation = $variation;
    //             break;
    //         }
    //     }
       
    //     if (!$targetVariation) {
    //         return 0;
    //     }
    
    //     return $this->calculateAvailableQuantity($roofRack,$product, $targetVariation, $vendorId, $fromDate, $toDate, 'carrier');

    // }
    
    // private function checkFootAvailability($roofRack,$product, $footSku, $vendorId, $fromDate, $toDate)
    // {
    //     if ($vendorId == 'admin') {
    //         $variations = $product->variations()
    //             ->with(['options' => function($query) {
    //                 $query->where('color_id', 2);
    //             }])
    //             ->get();
    //     } else {
    //         $variations = $product->variations()
    //             ->with(['options' => function ($query) {
    //                 $query->where('color_id', 2);
    //             }])
    //             ->get();
    
    //         foreach ($variations as $variation) {
    //             foreach ($variation->options as $option) {
    //                 $vendorQty = $option->vendorQuantities()
    //                     ->where('vendor_id', $vendorId)
    //                     ->first();
    //                 $option->quantity = $vendorQty?->quantity ?? 0;
    //             }
    //         }
    //     }
    
    //     // Find variation that matches the foot SKU
    //     $targetVariation = null;
    //     foreach ($variations as $variation) {
    //         if ( Variation::where('id',$variation->variation_id)->first()->name == $footSku) {
    //             $targetVariation = $variation;
    //             break;
    //         }
    //     }
    //     if (!$targetVariation) {
    //         return 0;
    //     }
    
    //     return $this->calculateAvailableQuantity($roofRack,$product, $targetVariation, $vendorId, $fromDate, $toDate, 'foot');
    // }
    
    // private function checkFootkitAvailability($roofRack,$product, $footkitSku, $vendorId, $fromDate, $toDate)
    // {
    //     if ($vendorId == 'admin') {
    //         $variations = $product->variations()
    //             ->with(['options' => function($query) {
    //                 $query->where('color_id', 2);
    //             }])
    //             ->get();
    //     } else {
    //         $variations = $product->variations()
    //             ->with(['options' => function ($query) {
    //                 $query->where('color_id', 2);
    //             }])
    //             ->get();
    
    //         foreach ($variations as $variation) {
    //             foreach ($variation->options as $option) {
    //                 $vendorQty = $option->vendorQuantities()
    //                     ->where('vendor_id', $vendorId)
    //                     ->first();
    //                 $option->quantity = $vendorQty?->quantity ?? 0;
    //             }
    //         }
    //     }
    
    //     // Find variation that matches the footkit SKU
    //     $targetVariation = null;
    //     foreach ($variations as $variation) {
    //         if ($variation->size_id == $footkitSku || $variation->variation_id == $footkitSku) {
    //             $targetVariation = $variation;
    //             break;
    //         }
    //     }
    
    //     if (!$targetVariation) {
    //         return 0;
    //     }
    
    //     return $this->calculateAvailableQuantity($roofRack,$product, $targetVariation, $vendorId, $fromDate, $toDate, 'footkit');
    // }
    
    // private function calculateAvailableQuantity($roofRack,$product, $variation, $vendorId, $fromDate, $toDate, $componentType = null)
    // {
    //     $quantity = $variation->options->sum('quantity');

    //     if ($vendorId == 'admin') {
    //         if ($variation->status == 'in_stock') {
    //             if ($quantity > 0) {
    //                 return $quantity;
    //             } else {
    //                 $overlappingBookings = DB::table('booking_items')
    //                     ->where('product_id', $roofRack->id)
    //                     ->where('status', 'confirmed')
    //                     ->whereDate(DB::raw('COALESCE(new_dropoff_date, dropoff_date)'), '>=', now())
    //                     ->where('already_booked', '!=', 1)
    //                     ->where('damage' , '!=' , 1)
    //                     ->when($componentType === 'carrier', function($query) use ($variation) {
    //                         $length = (float)$variation->length;
    //                         return $query->whereRaw('CAST(REPLACE(carrier_length, " mm", "") AS DECIMAL(10,2)) = ?', [$length]);
    //                     })
    //                     ->when($componentType === 'foot', function($query) use ($variation) {
    //                         return $query->where('foot_sku', Variation::where('id',$variation->variation_id)->first()->name);
    //                     })
    //                     ->when($componentType === 'footkit', function($query) use ($variation) {
    //                         return $query->where('footkit_sku', Variation::where('id',$variation->variation_id)->first()->name);
    //                     })
    //                     ->get();

    //                 $availableUnits = 0;
                        
    //                 foreach ($overlappingBookings as $booking) {
    //                     if($booking->new_dropoff_date){
    //                         if (Carbon::parse($booking->new_dropoff_date)->lt($fromDate)){
    //                             $availableUnits++;
                               
    //                         }
    //                     }
    //                     else if(Carbon::parse($booking->dropoff_date)->lt($fromDate)){
    //                         $availableUnits++;
                           
    //                     }
    //                 }

    //                 return $availableUnits > 0 ? $availableUnits : 0;
    //             }
    //         }
    //         return 0;
    //     } else {
    //         $isDisabled = DB::table('vendor_disabled_variants')
    //             ->where('vendor_id', $vendorId)
    //             ->where('product_variation_id', $variation->id)
    //             ->exists();

    //         if ($isDisabled ) {
    //             return 0;
    //         }

    //         if ($quantity > 0) {
    //             return $quantity;
    //         } else {
    //             $overlappingBookings = DB::table('booking_items')
    //                 ->where('product_id', $roofRack->id)
    //                 ->where('vendor_id', $vendorId)
    //                 ->where('status', 'confirmed')
    //                 ->where('already_booked', '!=', 1)
    //                 ->where('damage' , '!=' , 1)
    //                 ->whereDate(DB::raw('COALESCE(new_dropoff_date, dropoff_date)'), '>=', now())
    //                 ->when($componentType === 'carrier', function($query) use ($variation) {
    //                     $length = (float)$variation->length;
                        
    //                     return $query->whereRaw("CAST(TRIM(TRAILING 'mm' FROM carrier_length) AS UNSIGNED) = ?", [(int)$length]);
    //                 })
    //                 ->when($componentType === 'foot', function($query) use ($variation) {
    //                     return $query->where('foot_sku', Variation::where('id',$variation->variation_id)->first()->name);
    //                 })
    //                 ->when($componentType === 'footkit', function($query) use ($variation) {
    //                     return $query->where('footkit_sku', Variation::where('id',$variation->variation_id)->first()->name);
    //                 })
    //                 ->get();
    //                 $availableUnits = 0;
    //                 foreach ($overlappingBookings as $booking) {
    //                     if($booking->new_dropoff_date){
    //                         if (Carbon::parse($booking->new_dropoff_date)->lt($fromDate)){
    //                             $availableUnits++;
                              
    //                         }
    //                     }
    //                     else if(Carbon::parse($booking->dropoff_date)->lt($fromDate)){
    //                         $availableUnits++;
                         
    //                     }
    //                 }

    //             return $availableUnits > 0 ? $availableUnits : 0;
    //         }
    //     }
    // }
    
    public function blog()
    {
        try {
            $blogs = Blog::orderBy('id', 'desc')->get();
            $admin = Admin::select('first_name','last_name')->first();

            return view($this->file_path . 'blog' . $this->file_main,compact('blogs','admin'));
        } catch (Exception $e) {
            Log::error('Error in blog(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }

    public function blogDetails($slug)
    {
        try {
            $blog = Blog::where('slug',$slug)->first();
            $admin = Admin::select('first_name','last_name','image')->first();
            $randomBlog = Blog::inRandomOrder()->take(3)->get();


            return view($this->file_path . 'blog.details',compact('blog','admin','randomBlog'));
        } catch (Exception $e) {
            Log::error('Error in blog(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }

    public function contact()
    {
        try {
            return view($this->file_path . 'contact' . $this->file_main);
        } catch (Exception $e) {
            Log::error('Error in contact(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }

    public function impresum()
    {
        try {
            $page_type = 'Impressum';
            return view($this->file_path . 'impresusm' . $this->file_main, compact('page_type'));
        } catch (Exception $e) {
            Log::error('Error in impresum(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }

    public function termCondition()
    {
        try {
            $data = $this->getaPageData('term-condition');
            return view($this->file_path . 'term-condition' . $this->file_main, compact('data'));
        } catch (Exception $e) {
            Log::error('Error in termCondition(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }

    public function returnProcess()
    {
        try {
            $data = $this->getaPageData('return-process');
            return view($this->file_path . 'return-process' . $this->file_main, compact('data'));
        } catch (Exception $e) {
            Log::error('Error in returnProcess(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }

    public function withdrawlRights()
    {
        try {
            $data = $this->getaPageData('withdrawal-right');
            return view($this->file_path . 'withdrawl-rights' . $this->file_main, compact('data'));
        } catch (Exception $e) {
            Log::error('Error in withdrawlRights(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }

    public function gdpr()
    {
        try {
            $data = $this->getaPageData('gdpr');
            return view($this->file_path . 'gdpr' . $this->file_main, compact('data'));
        } catch (Exception $e) {
            Log::error('Error in gdpr(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }

    public function roofbox()
    {
        try {
            return view($this->file_path . 'roofbox' . $this->file_main);
        } catch (Exception $e) {
            Log::error('Error in roofbox(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }
    public function getSizeDetails(Product $product, Request $request)
    {
        $sizeId = $request->input('size_id');
        $variationId = $request->input('variation_id');

        if ($variationId) {
            $variation = ProductVariationOption::with('variation')
                ->where('id', $variationId)
                ->first();

            if ($variation && $variation->variation) {
                return response()->json([
                    'length' => $variation->variation->length,
                    'width' => $variation->variation->width,
                    'height' => $variation->variation->height,
                    'weight' => $variation->variation->weight,
                    'volume' => $variation->variation->volume,
                    'max_load' => $variation->variation->max_load,
                ]);
            }
        }

        if ($sizeId) {
            $productVariation = ProductVariation::where('product_id', $product->id)
                ->where('size_id', $sizeId)
                ->first();

            if ($productVariation) {
                return response()->json([
                    'length' => $productVariation->length,
                    'width' => $productVariation->width,
                    'height' => $productVariation->height,
                    'weight' => $productVariation->weight,
                    'volume' => $productVariation->volume,
                    'max_load' => $productVariation->max_load,
                ]);
            }
        }

        // If no specific size/variation data, return product's general data
        return response()->json([
            'length' => $product->length,
            'width' => $product->width,
            'height' => $product->height,
            'weight' => $product->weight,
            'volume' => $product->volume,
            'max_load' => $product->max_load,
        ]);
    }


    public function getQuote()
    {
        $categories = Category::with(['products' => function($query) {
            $query->with(['brand', 'images' => function($imageQuery) {
                $imageQuery->where('type', 'home');
            }]);
        }])->get();

        $manufacturers = MotherFile::distinct()->pluck('car_make');
        
        // Get the first product to pre-load the details
        $firstProduct = Product::with([
            'category',
            'brand',
            'variations.options.color',
            'variations',
            'images',
            'models',
            'otherInfo',
        ])->first();

        if (!$firstProduct) {
            abort(404, 'No products found');
        }

        $normalizedName = $this->normalizeProductName($firstProduct->name);

        $priceRecord = ProductPrice::get()->first(function ($record) use ($normalizedName) {
            return $this->normalizeProductName($record->product_name) === $normalizedName;
        });
        $roof_rackPrice = ProductPrice::where('product_name', 'Dachträger ergänzt zu Dachbox/ Dachfahrradträger')->first();
        
        if ($priceRecord && $priceRecord->price) {
            $prices = $priceRecord->price;
            $firstProduct->first_price = $prices[1] ?? null;

            $rawPrices = ($priceRecord->price);
            $dailyPrices = array_map(function ($price) {
                return floatval(str_replace(['€', ',', ' '], ['.', '.', ''], $price));
            }, $rawPrices);

            $rawRoofRack = ($roof_rackPrice->price);
            $roofRackPrices = array_map(function ($price) {
                return floatval(str_replace(['€', ',', ' '], ['.', '.', ''], $price));
            }, $rawRoofRack);
        } else {
            $firstProduct->first_price = null;
            $dailyPrices = [];
            $roofRackPrices = [];
        }

        $category = $firstProduct->category;

        $roofTypes = [
            'Normal Roof',
            'Normal roof without glass roof',
            'Normal Roof with glass roof',
            'Fixpoints',
            'Fixpoints without glass roof',
            'Fixpoints with glass roof',
            'Integrated Roof Railing',
            'Elevated Roof Rails',
            'T-Profile',
            'Raingutter'
        ];

        $sizeIds = is_array($category->size_ids) ? $category->size_ids : json_decode($category->size_ids, true);
        $variationIds = is_array($category->variation_ids) ? $category->variation_ids : json_decode($category->variation_ids, true);
        $colorIds = is_array($category->color_ids) ? $category->color_ids : json_decode($category->color_ids, true);

        $inStockVariationRecords = \App\Models\ProductVariation::where('product_id', $firstProduct->id)
            ->where('status', 'in_stock')
            ->get();
        $inStockSizeIds = $inStockVariationRecords->pluck('size_id')->filter()->unique()->toArray();
        $inStockVariationIds = $inStockVariationRecords->pluck('variation_id')->filter()->unique()->toArray();
        $sizes = $inStockSizeIds ? Size::whereIn('id', $inStockSizeIds)->get() : collect();
        $variations = $inStockVariationIds ? Variation::whereIn('id', $inStockVariationIds)->get() : collect();

        // $sizes = $sizeIds ? Size::whereIn('id', $sizeIds)->get() : collect();
        // $variations = $variationIds ? Variation::whereIn('id', $variationIds)->get() : collect();
        $colors = $colorIds ? Color::whereIn('id', $colorIds)->get() : collect();
        $holidays = \App\Models\Holidays::pluck('date')->toArray();
        $closedDates = \App\Models\DateAvailability::where('is_closed', true)
            ->pluck('date')
            ->map(fn($date) => $date instanceof \Carbon\Carbon ? $date->format('Y-m-d') : (string) $date)
            ->toArray();
        
        $holidays = array_unique(array_merge($holidays, $closedDates));

        $nonworkingDays = WorkingHours::whereNull('open_time')
            ->orWhereNull('close_time')
            ->pluck('day')
            ->map(fn($day) => strtolower($day)) // make lowercase for matching JS getDay()
            ->toArray();

        // $vendors = Vendor::where('status','active')->get(['id', 'street','zip_code','city']);

        $vendors = Vendor::where('status', 'active')
            ->whereHas('products', function($query) use ($firstProduct) {
                $query->where('products.id', $firstProduct->id);
            })
            ->get(['id', 'street','zip_code','city']);
        
        $productType =  $this->determineProductType($firstProduct->name);

        return view($this->file_path . 'product.quote', compact(
            'categories',
            'manufacturers',
            'firstProduct',
            'priceRecord',
            'colors',
            'sizes',
            'variations',
            'roofTypes',
            'dailyPrices',
            'roofRackPrices',
            'holidays',
            'nonworkingDays',
            'vendors',
            'productType'
        ));
    }

    public function getProductDetails(Request $request)
    {
        $product = Product::with([
            'category',
            'brand',
            'variations.options.color',
            'variations',
            'images',
            'models',
            'otherInfo',
        ])->find($request->product_id);


        $productType = $this->determineProductType($product->name);



        if (!$product) {
            return response()->json(['error' => 'Product not found'], 404);
        }
        $manufacturers = MotherFile::distinct()->pluck('car_make');

        $normalizedName = $this->normalizeProductName($product->name);

        $priceRecord = ProductPrice::get()->first(function ($record) use ($normalizedName) {
            return $this->normalizeProductName($record->product_name) === $normalizedName;
        });
        $roof_rackPrice = ProductPrice::where('product_name', 'Dachträger ergänzt zu Dachbox/ Dachfahrradträger')->first();
        
        if ($priceRecord && $priceRecord->price) {
            $prices = $priceRecord->price;
            $product->first_price = $prices[1] ?? null;

            $rawPrices = ($priceRecord->price);
            $dailyPrices = array_map(function ($price) {
                return floatval(str_replace(['€', ',', ' '], ['.', '.', ''], $price));
            }, $rawPrices);

            $rawRoofRack = ($roof_rackPrice->price);
            $roofRackPrices = array_map(function ($price) {
                return floatval(str_replace(['€', ',', ' '], ['.', '.', ''], $price));
            }, $rawRoofRack);
        } else {
            $product->first_price = null;
            $dailyPrices = [];
            $roofRackPrices = [];
        }

        $category = $product->category;

        $roofTypes = [
            'Normal Roof',
            'Normal roof without glass roof',
            'Normal Roof with glass roof',
            'Fixpoints',
            'Fixpoints without glass roof',
            'Fixpoints with glass roof',
            'Integrated Roof Railing',
            'Elevated Roof Rails',
            'T-Profile',
            'Raingutter'
        ];

        $sizeIds = is_array($category->size_ids) ? $category->size_ids : json_decode($category->size_ids, true);
        $variationIds = is_array($category->variation_ids) ? $category->variation_ids : json_decode($category->variation_ids, true);
        $colorIds = is_array($category->color_ids) ? $category->color_ids : json_decode($category->color_ids, true);


        $inStockVariationRecords = \App\Models\ProductVariation::where('product_id', $product->id)
            ->where('status', 'in_stock')
            ->get();
        $inStockSizeIds = $inStockVariationRecords->pluck('size_id')->filter()->unique()->toArray();
        $inStockVariationIds = $inStockVariationRecords->pluck('variation_id')->filter()->unique()->toArray();
        $sizes = $inStockSizeIds ? Size::whereIn('id', $inStockSizeIds)->get() : collect();
        $variations = $inStockVariationIds ? Variation::whereIn('id', $inStockVariationIds)->get() : collect();

        // $sizes = $sizeIds ? Size::whereIn('id', $sizeIds)->get() : collect();
        // $variations = $variationIds ? Variation::whereIn('id', $variationIds)->get() : collect();
        $colors = $colorIds ? Color::whereIn('id', $colorIds)->get() : collect();

        $holidays = \App\Models\Holidays::pluck('date')->toArray();
        $closedDates = \App\Models\DateAvailability::where('is_closed', true)
            ->pluck('date')
            ->map(fn($date) => $date instanceof \Carbon\Carbon ? $date->format('Y-m-d') : (string) $date)
            ->toArray();
        
        $holidays = array_unique(array_merge($holidays, $closedDates));
        $nonworkingDays = WorkingHours::whereNull('open_time')
        ->orWhereNull('close_time')
        ->pluck('day')
        ->map(fn($day) => strtolower($day)) // make lowercase for matching JS getDay()
        ->toArray();


        

        // $vendors = Vendor::where('status','active')->get(['id', 'street','zip_code','city']);

        $vendors = Vendor::where('status', 'active')
        ->whereHas('products', function($query) use ($product) {
            $query->where('products.id', $product->id);
        })
        ->get(['id', 'street','zip_code','city']);


        $modalView = match ($productType) {
            'roof_rack' => 'frontend.pages.product.partials.roofRackModal',
            'carrier_foot' => 'frontend.pages.product.partials.carrierModal',
            default => 'frontend.pages.product.partials.modals',
        };
        $modalHtml = view($modalView, [
            'dailyPrices' => $dailyPrices,
        ])->render();

        return response()->json([
            'product' => $product,
            'priceRecord' => $priceRecord,
            'colors' => $colors,
            'sizes' => $sizes,
            'variations' => $variations,
            'dailyPrices' => $dailyPrices,
            'roofRackPrices' => $roofRackPrices,
            'productType' => $productType,
            'nonworkingDays' => $nonworkingDays,
            'modalHtml' => $modalHtml,
            'holidays' => $holidays,
            'view' => view($this->file_path . 'product.partials.dynamic_product', compact(
                'product',
                'priceRecord',
                'colors',
                'sizes',
                'variations',
                'roofTypes',
                'dailyPrices',
                'roofRackPrices',
                'manufacturers',
                'holidays',
                'nonworkingDays',
                'vendors',
                'productType'
            ))->render()
        ]);
    }

    // private function determineProductType($productName)
    // {
    //     if (str_contains($productName, 'Roof Rack')) return 'roof_rack';
    //     if (in_array($productName, ['Carrier', 'Foot', 'Footkit'])) return 'carrier_foot';
    //     return 'standard';
    // }
    private function determineProductType($productName)
    {
        if (str_contains($productName, 'Roof Rack')) return 'roof_rack';
        if (str_contains($productName, 'Dachträger')) return 'roof_rack';
        if (in_array($productName, ['Carrier', 'Foot', 'Footkit' , 'Traverse' , 'Fuß', 'Fußkit'])) return 'carrier_foot';
        return 'standard';
    }


    public function rearbox()
    {
        try {
            return view($this->file_path . 'rearbox' . $this->file_main);
        } catch (Exception $e) {
            Log::error('Error in rearbox(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }

    public function roofrack()
    {
        try {
            return view($this->file_path . 'roofrack' . $this->file_main);
        } catch (Exception $e) {
            Log::error('Error in roofrack(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }

    public function roofBikeRack()
    {
        try {
            return view($this->file_path . 'roof-bike-rack' . $this->file_main);
        } catch (Exception $e) {
            Log::error('Error in roofBikeRack(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }

    public function clutchBikeRack()
    {
        try {
            return view($this->file_path . 'clutch-bike-rack' . $this->file_main);
        } catch (Exception $e) {
            Log::error('Error in clutchBikeRack(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }

    public function howTo()
    {
        try {
            $page_type = 'how-to';
            return view($this->file_path . 'howto' . $this->file_main, compact('page_type'));
        } catch (Exception $e) {
            Log::error('Error in gdpr(): ' . $e->getMessage());
            return response()->view('errors.500', [], 500);
        }
    }
    public function search(Request $request)
    {
        $query = $request->input('query');
    
        if (empty($query)) {
            return response()->json([]);
        }
    
        $products = Product::with(['mainPhoto', 'category'])
            ->where(function ($q) use ($query) {
                $q->where('name', 'like', "%$query%")
                  ->orWhere('description', 'like', "%$query%");
            })
            ->select('id', 'name', 'category_id') 
            ->limit(8)
            ->get();
    
        $products->each(function ($product) {
            $product->product_url = $product->category
                ? route('product.details', ['slug' => $product->category->slug])
                : '#';
            $product->image_url = $product->mainPhoto
                ? asset('storage/' . $product->mainPhoto->image_path)
                : asset('frontend/images/placeholder-product.jpg');
        });
    
        return response()->json($products);
    }


    public function bookingRequest(Request $request){
        $vars = $request->all();
        extract($vars);


        $year = date('Y');

        $vendorstreet = $vendorzipCode = $vendorcityName = null;

        if (preg_match('/^(.*)\s(\d{5})\s(.+)$/', trim($pickup_location), $matches)) {
            $vendorstreet   = trim($matches[1]);
            $vendorzipCode  = $matches[2];
            $vendorcityName = trim($matches[3]);
        }

        $vendor = Vendor::whereRaw("CONCAT(street, ' ', zip_code, ' ', city) = ?", [$pickup_location])->first();

        $vendorId = $vendor ? $vendor->id : null;


        $count = Notification::where('type', 'new_booking_request')
            ->whereYear('created_at', $year)
            ->count();

        $bookingRequestId = '#' . $year . '-' . ($count + 1);
        $old_product_name = $product_name;
        if($product_name == 'Roof Rack' || $product_name == 'Dachträger'){
            $product_name = $manufacturer . ' | '. $car_model . ' | '.  $car_year  . ' | '. ucwords($roof_type) ;
        }

        $title = __('messages.title_' . strtolower(str_replace('.', '', $title)));

        if($vendorId){
            Notification::create([
                'type' => 'new_booking_request',
                'status' => 'New Booking Request', 
                'customer_name' => $title . ' ' .$first_name . ' ' . $last_name,
                'customer_email' => $email,
                'vendor_id' => $vendorId,
                'customer_phone' => $country_code . ' ' .$phone,
               'data' => [
                    ['label' => 'Booking Request ID', 'value' =>  $bookingRequestId],
                    ['label' => 'PickUp Location', 'value' => $pickup_location],
                    ['label' => 'Unavailable Components', 'value' => $unavailable_components],
                    ['label' => 'Product Name', 'value' => $product_name],
                    ['label' => 'Color Name', 'value' => $color_name ?? ''],
                    ['label' => 'Size/Variation', 'value' => $size_name ?? $variation_name ?? ''],
                    ['label' => 'Carrier Length', 'value' => $carrier_length ?? ''],
                    ['label' => 'Foot Sku', 'value' => $foot_sku ?? ''],
                    ['label' => 'Footkit Sku', 'value' => $footkit_sku ?? ''],
                    ['label' => 'Manufacturer', 'value' => $manufacturer],
                    ['label' => 'Car Model', 'value' => $car_model],
                    ['label' => 'Car Year', 'value' => $car_year],
                    ['label' => 'Trailer Hitch', 'value' => $trailer_hitch ? ucwords($trailer_hitch) : ''],
                    ['label' => 'Roof Type', 'value' => $roof_type ? ucwords($roof_type) :  ''],
                    ['label' => 'Rental Period', 'value' => $rental_from . ' to ' . $rental_to],
                    ['label' => 'City', 'value' => $city],
                    ['label' => 'Zipcode', 'value' => $zipcode],
                    ['label' => 'Street', 'value' => $street],
                    ['label' => 'State', 'value' => $state],
                    ['label' => 'Message', 'value' => $message],
                ],
                'is_read' => false,
            ]);
        }else{
            Notification::create([
                'type' => 'new_booking_request',
                'status' => 'New Booking Request', 
                'customer_name' => $title . ' ' .$first_name . ' ' . $last_name,
                'customer_email' => $email,
                'customer_phone' => $country_code . ' ' .$phone,
               'data' => [
                    ['label' => 'Booking Request ID', 'value' =>$bookingRequestId],
                    ['label' => 'PickUp Location', 'value' => $pickup_location],
                    ['label' => 'Unavailable Components', 'value' => $unavailable_components],
                    ['label' => 'Product Name', 'value' => $product_name],
                    ['label' => 'Color Name', 'value' => $color_name ?? ''],
                    ['label' => 'Size/Variation', 'value' => $size_name ?? $variation_name ?? ''],
                    ['label' => 'Carrier Length', 'value' => $carrier_length ?? ''],
                    ['label' => 'Foot Sku', 'value' => $foot_sku ?? ''],
                    ['label' => 'Footkit Sku', 'value' => $footkit_sku ?? ''],
                    ['label' => 'Manufacturer', 'value' => $manufacturer],
                    ['label' => 'Car Model', 'value' => $car_model],
                    ['label' => 'Car Year', 'value' => $car_year],
                    ['label' => 'Trailer Hitch', 'value' => $trailer_hitch ? ucwords($trailer_hitch) : ''],
                    ['label' => 'Roof Type', 'value' => $roof_type ? ucwords($roof_type) :  ''],
                    ['label' => 'Rental Period', 'value' => $rental_from . ' to ' . $rental_to],
                    ['label' => 'City', 'value' => $city],
                    ['label' => 'Zipcode', 'value' => $zipcode],
                    ['label' => 'Street', 'value' => $street],
                    ['label' => 'State', 'value' => $state],
                    ['label' => 'Message', 'value' => $message],
                ],
                'is_read' => false,
            ]);
        }
     
        

        $rentalFromFormatted = Carbon::createFromFormat('d.m.Y', $rental_from)->format('Y-m-d');
        $rentalToFormatted   = Carbon::createFromFormat('d.m.Y', $rental_to)->format('Y-m-d');


        BookingRequest::create([
            'booking_request_id' => $bookingRequestId,
            'customer_name' => $title . ' ' .$first_name . ' ' . $last_name,
            'customer_email' => $email,
            'customer_phone' => $country_code . ' ' .$phone,
            'product_name' => $old_product_name,
            'color_name' => $color_name ?? '',
            'size_or_variation' => $size_name ?? $variation_name ?? '',
            'carrier_length' => $carrier_length ?? '',
            'foot_sku' => $foot_sku ?? '',
            'footkit_sku' => $footkit_sku ?? '',
            'manufacturer' => $manufacturer,
            'car_model' => $car_model,
            'car_year' => $car_year,
            'trailer_hitch' => $trailer_hitch ? ucwords($trailer_hitch) : '',
            'roof_type' => $roof_type ? ucwords($roof_type) : '',
            'rental_from' => $rentalFromFormatted,
            'rental_to' => $rentalToFormatted,
            'pickup_location' => $pickup_location,
            'city' => $city,
            'zipcode' => $zipcode,
            'street' => $street,
            'state' => $state,
            'unavailable_components' => $unavailable_components,
            'message' => $message,
        ]);


        $customerMessage = $request->input('message');
        $admin = Admin::first();
        try{
            Mail::send('email.request_booking_email', [
                'title' => $title ,
                'first_name' => $first_name,
                'last_name' => $last_name,
                'pickup_location' => $pickup_location,
                'vendorstreet' => $vendorstreet,
                'vendorzipCode' => $vendorzipCode,
                'vendorcityName' => $vendorcityName,
                'unavailable_components' => $unavailable_components,
                'color_name' => $color_name ?? '',
                'size_name' => $size_name ?? $variation_name ?? '',
                'foot_sku' => $foot_sku,
                'footkit_sku' => $footkit_sku,
                'carrier_length' => $carrier_length,
                'manufacturer' => $manufacturer,
                'car_model' => $car_model,
                'car_year' => $car_year,
                'trailer_hitch' => $trailer_hitch,
                'roof_type' => $roof_type,
                'rental_from' => $rental_from,
                'rental_to'=> $rental_to,
                'city' => $city,
                'email' => $email,
                'zipcode' => $zipcode,
                'product_name' => $product_name,
                'street' => $street,
                'state' => $state,
                'messages' => $customerMessage,
                'phone' => $country_code . ' ' .$phone,
                'booking_request_id' => $bookingRequestId,
                'admin' => $admin,
            ], function ($message) use ($request) {
                $message->to('info@dachboxit.de')
                        ->subject('Buchungsanfrage');
            });
            Mail::send('email.request_booking_email', [
                'title' => $title ,
                'first_name' => $first_name,
                'last_name' => $last_name,
                'pickup_location' => $pickup_location,
                 'vendorstreet' => $vendorstreet,
                'vendorzipCode' => $vendorzipCode,
                'vendorcityName' => $vendorcityName,
                'unavailable_components' => $unavailable_components,
                'color_name' => $color_name ?? '',
                'size_name' => $size_name ?? $variation_name ?? '',
                'foot_sku' => $foot_sku,
                'footkit_sku' => $footkit_sku,
                'carrier_length' => $carrier_length,
                'manufacturer' => $manufacturer,
                'car_model' => $car_model,
                'car_year' => $car_year,
                'trailer_hitch' => $trailer_hitch,
                'roof_type' => $roof_type,
                'rental_from' => $rental_from,
                'rental_to'=> $rental_to,
                'city' => $city,
                'email' => $email,
                'zipcode' => $zipcode,
                'product_name' => $product_name,
                'street' => $street,
                'state' => $state,
                'messages' => $customerMessage,
                'phone' => $country_code . ' ' .$phone,
                'booking_request_id' => $bookingRequestId,
                'admin' => $admin,
            ], function ($message) use ($request,$email) {
                $message->to($email)
                        ->subject('Buchungsanfrage');
            });
            Mail::send('email.request_booking_email', [
                'title' => $title ,
                'first_name' => $first_name,
                'last_name' => $last_name,
                'pickup_location' => $pickup_location,
                 'vendorstreet' => $vendorstreet,
                'vendorzipCode' => $vendorzipCode,
                'vendorcityName' => $vendorcityName,
                'unavailable_components' => $unavailable_components,
                'color_name' => $color_name ?? '',
                'size_name' => $size_name ?? $variation_name ?? '',
                'foot_sku' => $foot_sku,
                'footkit_sku' => $footkit_sku,
                'carrier_length' => $carrier_length,
                'manufacturer' => $manufacturer,
                'car_model' => $car_model,
                'car_year' => $car_year,
                'trailer_hitch' => $trailer_hitch,
                'roof_type' => $roof_type,
                'rental_from' => $rental_from,
                'rental_to'=> $rental_to,
                'city' => $city,
                'email' => $email,
                'zipcode' => $zipcode,
                'product_name' => $product_name,
                'street' => $street,
                'state' => $state,
                'messages' => $customerMessage,
                'phone' => $country_code . ' ' .$phone,
                'booking_request_id' => $bookingRequestId,
                'admin' => $admin,
            ], function ($message) use ($request) {
                $message->to('cs@dachboxit.de')
                        ->subject('Buchungsanfrage');
            });

            if ($vendorId) {
                $vendor = Vendor::find($vendorId);
            
                if ($vendor && $vendor->email) {
                    Mail::send('email.request_booking_email', [
                        'title' => $title,
                        'first_name' => $first_name,
                        'last_name' => $last_name,
                        'pickup_location' => $pickup_location,
                         'vendorstreet' => $vendorstreet,
                        'vendorzipCode' => $vendorzipCode,
                        'vendorcityName' => $vendorcityName,
                        'unavailable_components' => $unavailable_components,
                        'color_name' => $color_name ?? '',
                        'size_name' => $size_name ?? $variation_name ?? '',
                        'foot_sku' => $foot_sku,
                        'footkit_sku' => $footkit_sku,
                        'carrier_length' => $carrier_length,
                        'manufacturer' => $manufacturer,
                        'car_model' => $car_model,
                        'car_year' => $car_year,
                        'trailer_hitch' => $trailer_hitch,
                        'roof_type' => $roof_type,
                        'rental_from' => $rental_from,
                        'rental_to' => $rental_to,
                        'city' => $city,
                        'email' => $email,
                        'zipcode' => $zipcode,
                        'product_name' => $product_name,
                        'street' => $street,
                        'state' => $state,
                        'messages' => $customerMessage,
                        'phone' => $country_code . ' ' . $phone,
                        'booking_request_id' => $bookingRequestId,
                        'admin' => $admin,
                    ], function ($message) use ($vendor) {
                        $message->to($vendor->email)
                                ->subject('Buchungsanfrage');
                    });
                }
            }
        }
        catch(Exception $e){
            dd($e->getMessage());
        }
       

       


        return response()->json([
            'success' => true,
            'message' => 'Booking request submitted successfully.'
        ]);

    }

    public function getVendorCalendarData(Request $request){
        $vendorId = $request->vendor_id;

        if ($vendorId === 'admin') {
            $holidays = \App\Models\Holidays::pluck('date')->toArray();

            $closedDates = \App\Models\DateAvailability::where('is_closed', 1)
                ->where('admin_id',1)
                ->pluck('date')
                ->map(fn($date) => $date instanceof \Carbon\Carbon ? $date->format('Y-m-d') : (string) $date)
                ->toArray();
            
            $holidays = array_unique(array_merge($holidays, $closedDates));
            $nonworkingDays = WorkingHours::whereNull('open_time')
                ->orWhereNull('close_time')
                ->pluck('day')
                ->map(fn($day) => strtolower($day)) 
                ->toArray();
        }else{
            $holidays = \App\Models\VendorHoliday::pluck('date')->toArray();
            $closedDates = \App\Models\DateAvailability::where('is_closed', 1)
                ->where('admin_id',$vendorId)
                ->pluck('date')
                ->map(fn($date) => $date instanceof \Carbon\Carbon ? $date->format('Y-m-d') : (string) $date)
                ->toArray();
            

            $holidays = array_unique(array_merge($holidays, $closedDates));

            $nonworkingDays = \App\Models\VendorWorkingHour::where('vendor_id', $vendorId)
                ->where(function ($q) {
                    $q->where('is_closed', 1)
                    ->orWhereNull('open_time')
                    ->orWhereNull('close_time');
                })
                ->pluck('day')
                ->map(function ($day) {
                    $map = [
                        'mo' => 'monday',
                        'di' => 'tuesday',
                        'mi' => 'wednesday',
                        'do' => 'thursday',
                        'fr' => 'friday',
                        'sa' => 'saturday',
                        'so' => 'sunday',
                    ];
                    return $map[$day] ?? strtolower($day);
                })
                ->toArray();
        }

        return response()->json([
            'holidays' => $holidays,
            'nonworkingDays' => $nonworkingDays,
        ]);
    }
    public function userCarDetail(Request $request){

        $car = \App\Models\UserCarDetail::where('user_id', auth()->id())->first();
        $data = [
            'manufacturer' => $car->manufacturer,
            'car_model' => $car->model,
            'car_year' => $car->year,
            'trailer_hitch' => $car->trailer_hitch
        ];
        return response()->json([
            'data' => $data,
        ]);
    }
}
