<?php

namespace App\Http\Controllers\Vendor\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Mail\UserForgotPassword;
use App\Models\Vendor;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\URL;
use App\Mail\VerifyEmail;
use App\Models\Notification;
use Illuminate\Support\Facades\File;


class AuthController extends Controller
{
    public function index()
    {
        return view('vendor.auth.login');
    }

    public function login(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'email' => 'required|email',
                'password' => 'required',
            ], [
                'email.required' => 'Please enter email address.',
                'email.email' => 'Please enter valid email address.',
                'password.required' => 'Please enter password.',
               
            ]);

            if ($validator->fails()) {
                return redirect()->back()
                    ->withErrors($validator)
                    ->withInput($request->only('email', 'remember'));
            }

            $vendor = Vendor::where('email', strtolower($request->email))->first();
            if (!$vendor) {
                return redirect()->back()
                    ->with('error', 'Please enter valid email address or password.')
                    ->withInput($request->only('email', 'remember'));
            }

            
            
            if (!$vendor->password) {
                return redirect()->back()
                    ->with('error', 'Please ask the admin to create a password for you in the SA backend');
            }
            

            if (!$vendor) {
                return redirect()->back()
                    ->with('error', 'Please enter valid email address or password.')
                    ->withInput($request->only('email', 'remember'));
            }

            if (!Auth::guard('vendor')->attempt(['email' => strtolower($request->email), 'password' => $request->password])) {
                return redirect()->back()
                    ->with('error', 'Please enter valid email address or password.')
                    ->withInput($request->only('email', 'remember'));
            }
            
            $vendor = Auth::guard('vendor')->user();
            
            if ($vendor->status !== 'active') {
                Auth::guard('vendor')->logout();
                return redirect()->back()
                    ->with('error', 'Your account is inactive. Please contact support.')
                    ->withInput($request->only('email', 'remember'));
            }

            return redirect()->route(
               $vendor->products()->exists()
                    ? 'vendor.dashboard'
                    : 'vendor.productSelection'
            )->with('success', 'Login successful!');

        } catch (\Exception $e) {
            Log::error('Login Error: ' . $e->getMessage());
            dd($e->getMessage());
            return redirect()->back()
                ->with('error', 'Something went wrong! Please try again later.')
                ->withInput($request->only('email', 'remember'));
        }
    }

    public function logout(Request $request)
    {
        try {
            
            Auth::guard('vendor')->logout();
            return redirect()->route('vendor.login')->with('success', 'Logout successful!');
        } catch (Exception $e) {
            return back()->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }

    public function forgotView()
    {
        return view('vendor.auth.forgot-password');
    }

    public function forgotPassword(Request $request)
    {
        try {
            $request->validate(['email' => 'required|email']);
            $user = Vendor::where('email', strtolower($request->email))->first();
            if (!$user) {
                return back()->withErrors(['email' => 'Email not found!']);
            }
            if ($user->status != 'active') {
                return redirect()->back()->with('error', 'Account is inactive.');
            }
           
            DB::table('password_reset_tokens')->where('email',  $user->email)->delete();
            
            $existingToken = DB::table('password_reset_tokens')->where('email', $user->email)->first();
            $token = Str::random(60);
            
            if ($existingToken && Carbon::parse($existingToken->created_at)->diffInMinutes(Carbon::now()) <= 15) {
                $token = $existingToken->token;
            } else {
                DB::table('password_reset_tokens')->updateOrInsert(
                    ['email' => $user->email],
                    ['token' => $token, 'created_at' => now()]
                );
            }
            
            
            try {
                Mail::to($user->email)->send(new UserForgotPassword($token, $user->first_name, $user->last_name));
                return back()->with('success', 'Password reset link sent successfully. Please check your email');
            } catch (\Exception $e) {
                Log::error('Email sending failed: ' . $e->getMessage());
                return back()->with('error', 'An error occurred: ' . $e->getMessage());
            }
        } catch (Exception $e) {
            return back()->withErrors(['email' => 'An error occurred: ' . $e->getMessage()]);
        }
    }

    public function resetPasswordView(Request $request)
    {
     $usertoken = $request->query('token');
        return view('vendor.auth.reset-password',compact('usertoken'));
    }

    public function resetPassword(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'password' => 'required|min:8',
                'token' => 'required',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            $resetRecord = DB::table('password_reset_tokens')->where('token', $request->token)->first();
            if (!$resetRecord) {
                return redirect()->back()->with('error','Invalid token.');
            }

            $user = Vendor::where('email', $resetRecord->email)->first();
            if (!$user) {
                return redirect()->back()->with('error','Opps something went wrong.');
            }

            $user->update(['password' => Hash::make($request->password)]);
            DB::table('password_reset_tokens')->where('email', $resetRecord->email)->delete();

            return redirect()->route('login')->with('success', 'Password reset successfully.');
        } catch (Exception $e) {
            return back()->with('error', 'An error occurred: ' . $e->getMessage());
        }
    }

    public function registerView()
    {
        $country_code = json_decode(File::get(public_path('country-codes.json')), true);
        return view('vendor.auth.register',compact('country_code'));
    }

    public function register(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'title' => 'required',
                'first_name' => 'required|string|max:50|min:2',
                'last_name' => 'required|string|max:50|min:2',
                'email' => 'required|email|unique:vendors,email',
                'street' => 'required|string|max:100',
                'zip_code' => 'required|string|max:20',
                'phone_number' => 'required|numeric|digits_between:8,15',
                'phone_code' => 'required',
                'city' => 'required|string|max:50',
                'region' => 'required|string|max:50',
                'business_model' => 'required|string',
                'size_of_location' => 'required|string',
            ], [
                'title.required' => 'Please select a title.',
                'first_name.required' => 'Please enter your first name.',
                'first_name.min' => 'First name must be at least 2 characters.',
                'first_name.max' => 'First name must be less than 50 characters.',
                'last_name.required' => 'Please enter your surname.',
                'last_name.min' => 'Surname must be at least 2 characters.',
                'last_name.max' => 'Surname must be less than 50 characters.',
                'email.required' => 'Please enter email address.',
                'email.email' => 'Please enter valid email address.',
                'email.unique' => 'This email is already taken.',
                'street.required' => 'Please enter your street and house number.',
                'zip_code.required' => 'Please enter your zip code.',
                'city.required' => 'Please enter your city.',
                'region.required' => 'Please enter your region or state.',
                'phone_code.required' => 'Please select phone code.',
                'phone_number.required' => 'Please enter your phone number.',
                'phone_number.numeric' => 'Please enter valid phone number.',
                'phone_number.digits_between' => 'Phone number should be between 8 to 15 digits.',
                'business_model.required' => 'Please select business model.',
                'size_of_location.required' => 'Please select size of location.',
            ]);

            if ($validator->fails()) {
                return redirect()->back()->withErrors($validator)->withInput();
            }

            DB::beginTransaction();

            $vendor = Vendor::create([
                'title' => $request->title,
                'first_name' => $request->first_name,
                'last_name' => $request->last_name,
                'email' => strtolower($request->email),
                'password' => '', // empty for now
                'street' => $request->street,
                'zip_code' => $request->zip_code,
                'city' => $request->city,
                'region' => $request->region,
                'country_code' => $request->phone_code,
                'number' => $request->phone_number,
                'business_model' => $request->business_model,
                'size_of_location' => $request->size_of_location,
                'new_in_business' => $request->has('new_in_business') ? 1 : 0,
                'status' => 'inactive',
            ]);

            $vendor->save();

            DB::commit();

            $vars = $request->all();
            extract($vars);
         try{

             Notification::create([
                 'type' => 'new_SPO_registration',
                 'status' => 'New SPO Registration', 
                 'customer_name' => $title . ' ' .$first_name . ' ' . $last_name,
                 'customer_email' => $email,
                 'customer_phone' => $phone_code . ' ' .$phone_number,
                'data' => [
                     ['label' => 'Business Model', 'value' => $business_model],
                     ['label' => 'Size Of Location', 'value' => $size_of_location],
                     ['label' => 'New In Business', 'value' => $request->has('new_in_business') ? 'Yes' : 'No'],
                     ['label' => 'Address', 'value' => $street],
                     ['label' => 'Zipcode', 'value' => $zip_code],
                     ['label' => 'State', 'value' => $region],
                 ],
                 'is_read' => false,
             ]);
         }catch(\Exception $e){
            dd($e->getMessage());
         }


            return redirect()->route('vendor.registration.success')->with('success', 'Registration successful!...');

        } catch (\Exception $e) {
            DB::rollback();
            Log::error('Registration Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong! Please try again later.');
        }
    }

    public function registrationSuccess()
    {
        return view('vendor.auth.registration-success');
    }

    public function verify(Request $request)
    {
        try {
            $user = User::where('id', $request->id)
                ->where('email_verification_token', $request->token)
                ->first();

            if (!$user) {
                return redirect()->route('login')->with('error', 'Invalid or expired verification link.');
            }

            $user->email_verified_at = now();
            $user->email_verification_token = null;
            $user->save();

            return redirect()->route('login')->with('success', 'Email verified! You can now log in.');
        } catch (\Exception $e) {
            Log::error('Registration Error: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Something went wrong! Please try again later.');
        }
    }
}
