<?php

namespace App\Http\Controllers\Vendor\Product;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ProductBrand;
use App\Models\Category;
use App\Models\Size;
use App\Models\Bundle;
use App\Models\Color;
use App\Models\Variation;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;
use App\Models\BookingItem;
use Illuminate\Support\Facades\Response;
use App\Models\ProductPrice;
use App\Models\Product;
use App\Models\ProductVariation;
use App\Models\ProductVariationOption;
use App\Models\RelatedProductVariation;
use App\Models\ProductImage;
use App\Models\ProductModel;
use App\Models\ProductOtherInfo;
use Illuminate\Support\Facades\Storage;
use App\Exports\ProductsExport;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\VendorProductsExport;
use PDF;
use League\Csv\Reader;
use Illuminate\Support\Facades\DB;
use League\Csv\Writer;
use Illuminate\Support\Facades\Auth;
use App\Models\VendorProductQuantity;
use App\Models\Notification;
use Illuminate\Support\Str;


class ProductController extends Controller
{
    public $filePath = 'vendor.product';
     
    // public function index(Request $request)
    // {
    //     $vendor = Auth::guard('vendor')->user();
    
    //     $products = $vendor->products()->with([
    //         'category',
    //         'variations.options.color',
    //         'variations.size',
    //         'variations.variation',
    //     ]);
    
    //     if ($request->has('time_period')) {
    //         $timePeriod = $request->input('time_period');
    
    //         switch ($timePeriod) {
    //             case 'year':
    //                 $products->whereYear('products.created_at', date('Y'));
    //                 break;
    //             case 'month':
    //                 $products->whereMonth('products.created_at', date('m'))
    //                          ->whereYear('products.created_at', date('Y'));
    //                 break;
    //             case 'week':
    //                 $products->whereBetween('products.created_at', [
    //                     now()->startOfWeek(),
    //                     now()->endOfWeek()
    //                 ]);
    //                 break;
    //             case 'day':
    //                 $products->whereDate('products.created_at', date('Y-m-d'));
    //                 break;
    //         }
    //     }
    
    //     $products = $products->get();
    
    //     $totalProducts = $products->count();
    //     $activeRentals = 0;
    //     $returnedRentals = 0;
    //     $availableProducts = $products->sum('quantity');
    
    //     $groupedByCategory = $products->groupBy(fn ($product) => $product->category->name ?? 'Uncategorized');
    
    //     return view($this->filePath . '.index', compact(
    //         'groupedByCategory',
    //         'totalProducts',
    //         'activeRentals',
    //         'returnedRentals',
    //         'availableProducts',
    //         'vendor'
    //     ));
    // }

    public function index(Request $request)
    {

        $rentedCounts = $this->getRentedItemsCount();
        $vendor = Auth::guard('vendor')->user();

        $products = $vendor->products()->with([
            'category',
            'variations.options.color',
            'variations.size',
            'variations.variation',
        ]);

        if ($request->has('time_period')) {
            $timePeriod = $request->input('time_period');
            switch ($timePeriod) {
                case 'year':
                    $products->whereYear('products.created_at', date('Y'));
                    break;
                case 'month':
                    $products->whereMonth('products.created_at', date('m'))
                             ->whereYear('products.created_at', date('Y'));
                    break;
                case 'week':
                    $products->whereBetween('products.created_at', [
                        now()->startOfWeek(),
                        now()->endOfWeek()
                    ]);
                    break;
                case 'day':
                    $products->whereDate('products.created_at', date('Y-m-d'));
                    break;
            }
        }

        $products = $products->get();

        // Inject vendor-level quantity
        foreach ($products as $product) {
            foreach ($product->variations as $variation) {
                foreach ($variation->options as $option) {
                    $vendorQty = $option->vendorQuantities()
                        ->where('vendor_id', $vendor->id)
                        ->first();
                    $option->quantity = $vendorQty?->quantity ?? 0;
                }
            }
        }

        $disabledVariants = $vendor->disabledVariants()->pluck('product_variation_id')->toArray();


        // Inject vendor-level quantity
        // foreach ($products as $product) {
        //     foreach ($product->variations as $variation) {
        //         $variation->is_disabled = in_array($variation->id, $disabledVariants);
        //         $variation->status  = $variation->is_disabled ? 'out_of_stock' : 'in_stock';
        //         foreach ($variation->options as $option) {
        //             $vendorQty = $option->vendorQuantities()
        //                 ->where('vendor_id', $vendor->id)
        //                 ->first();
        //             $option->quantity = $vendorQty?->quantity ?? 0;
                
        //         }
        //     }
        // }

        foreach ($products as $product) {
            if ($product->variations->isEmpty()) {
                $rentedCount = $rentedCounts['product'][$product->id] ?? 0;
                $product->total_quantity = $product->quantity - $rentedCount;
            } else {
                foreach ($product->variations as $variation) {
                    $variation->is_disabled = in_array($variation->id, $disabledVariants);
                    $variation->status  = $variation->is_disabled ? 'out_of_stock' : 'in_stock';
                    if (!$variation->options->isEmpty()) {
                        foreach ($variation->options as $option) {
                            $vendorQty = $option->vendorQuantities()
                                ->where('vendor_id', $vendor->id)
                                ->first();
                            $option->quantity = $vendorQty?->quantity ?? 0;
                            $rentedCount = $rentedCounts['option'][$option->id] ?? 0;
                            $option->total_quantity = $option->quantity - $rentedCount;
                        }
                    } else {
                        $rentedCount = $rentedCounts['variation'][$variation->id] ?? 0;
                        $variation->total_quantity = $variation->quantity - $rentedCount;
                    }
                }
            }
        }
        $disabledVariants = $vendor->disabledVariants()->pluck('product_variation_id')->count();
        $totalProductCategories = Category::count();

        $bundleProducts = Bundle::count();

        $totalProducts = $vendor->products()->count();
        $activeRentals = 0;
        $returnedRentals = 0;
        $availableProducts = $products->sum('quantity');

        $groupedByCategory = $products->groupBy(fn ($product) => $product->category->name ?? 'Uncategorized');

        // $order = [
        //     'Roof Box',
        //     'Rearbox',
        //     'Roof Bike Carrier',
        //     'Clutch Bike Rack',
        //     'Roof Rack',
        //     'Carrier',
        //     'Foot',
        //     'Footkit',
        // ];

        $order = [
            'Dachbox',
            'Heckbox',
            'Dachfahrradträger',
            'Kupplungsfahrradträger',
            'Dachträger',
            'Traverse',
            'Fuß',
            'Fußkit',
        ];

        $groupedByCategory = $groupedByCategory->sortBy(function ($products, $categoryName) use ($order) {
            $index = array_search($categoryName, $order);
            return $index !== false ? $index : PHP_INT_MAX; // categories not in list go to end
        });

        return view($this->filePath . '.index', compact(
            'groupedByCategory',
            'totalProducts',
            'activeRentals',
            'disabledVariants',
            'returnedRentals',
            'availableProducts',
            'totalProductCategories',
            'vendor',
            'bundleProducts',
            'rentedCounts'
        ));
    }

    private function getRentedItemsCount()
    {
        $today = now()->format('Y-m-d');
        $vendor = Auth::guard('vendor')->user();
        $rentedItems = BookingItem::where('damage', '!=', 1)
            ->where('already_booked', 0)
            ->where('vendor_id', $vendor->id)
            ->whereHas('booking', function ($q) {
                $q->where('status', 'confirmed');
            })
            ->where(function($query) use ($today) {
                $query->where(function($q) use ($today) {
                    $q->where('pickup_date', '<=', $today)
                    ->where(function($q2) use ($today) {
                        $q2->where(function($q3) use ($today) {
                            $q3->whereNotNull('new_dropoff_date')
                                ->where('new_dropoff_date', '>=', $today);
                        })->orWhere(function($q4) use ($today) {
                            $q4->whereNull('new_dropoff_date')
                                ->where('dropoff_date', '>=', $today);
                        });
                    });
                });
            })
            ->get();



        $rentedCounts = [
            'product' => [],
            'variation' => [],
            'option' => []
        ];

        foreach ($rentedItems as $item) {
            // Count by product_id (for simple products like roof racks)
            if ($item->product_id) {
                if (!isset($rentedCounts['product'][$item->product_id])) {
                    $rentedCounts['product'][$item->product_id] = 0;
                }
                $rentedCounts['product'][$item->product_id]++;
            }
            
          
           
            if ($item->product_id && $item->size) {
                $product = Product::find($item->product_id);
                if ($product && $product->variations) {
                    foreach ($product->variations as $variation) {
                        if (($variation->size && $variation->size->name == $item->size)  || ($variation->variation && $variation->variation->name == $item->size)) {
                           
                            if (!isset($rentedCounts['variation'][$variation->id])) {
                                $rentedCounts['variation'][$variation->id] = 0;
                            }
                            $rentedCounts['variation'][$variation->id]++;
                            $name = strtolower($item->product_name);

                            if (in_array($name, ['carrier', 'foot', 'footkit', 'traverse', 'fuß', 'fußkit'])) {
                                $item->color = 'Schwarz';
                            }
                            // If there are options, find the matching one by color
                            if ($item->color && $variation->options) {
                                foreach ($variation->options as $option) {
                                    if ($option->color && $option->color->name == $item->color) {
                                        if (!isset($rentedCounts['option'][$option->id])) {
                                            $rentedCounts['option'][$option->id] = 0;
                                        }
                                        $rentedCounts['option'][$option->id]++;
                                        break;
                                    }
                                }
                            }
                            break;
                        }
                    }
                }
            }
        }
        // dd($rentedCounts);
        return $rentedCounts;
    }

    public function getAvailableProducts()
    {
        $vendor = Auth::guard('vendor')->user();
        
        $allProducts = Product::with(['homePhoto'])
            ->get();
        
        $enabledProductIds = $vendor->products()->pluck('products.id')->toArray();
        
        $products = $allProducts->map(function($product) use ($enabledProductIds) {
            return [
                'id' => $product->id,
                'name' => $product->name,
                'image' => $product->homePhoto ? asset('storage/' . $product->homePhoto->image_path) : asset('frontend/images/placeholder.png'),
                'selected' => in_array($product->id, $enabledProductIds)
            ];
        });
        
        return response()->json([
            'products' => $products
        ]);
    }

    public function saveSelectedProducts(Request $request)
    {
        $vendor = Auth::guard('vendor')->user();
        $productIds = $request->input('product_ids', []);

        $vendor->products()->sync($productIds);

        foreach ($productIds as $productId) {
            $variationOptionIds = ProductVariationOption::whereHas('variation', function ($q) use ($productId) {
                $q->where('product_id', $productId);
            })->pluck('id','sku');


            foreach ($variationOptionIds as $sku => $optionId) {
                VendorProductQuantity::firstOrCreate(
                    [
                        'vendor_id' => $vendor->id,
                        'product_variation_option_id' => $optionId,
                    ],
                    [
                        'quantity' => 0,
                        'sku' => $sku
                    ]
                );
            }
        }

        return redirect()->route('vendor.dashboard')->with('success', 'Products updated successfully.');
    }
   

    public function updateVariantStatus(Request $request, $id)
    {
        $vendor = Auth::guard('vendor')->user();
        $variant = ProductVariation::findOrFail($id);

        if ($request->status === 'out_of_stock') {
            $vendor->disabledVariants()->syncWithoutDetaching([$id]);
        } else {
            $vendor->disabledVariants()->detach($id);
        }

        return response()->json(['success' => true]);
    }

    public function showVariant($productId, $variantId ,$optionId = null)
    {
        $variant = ProductVariation::with([
            'product.category',
            'product.brand',
            'options.color',
            'size',
            'variation',
            'product.images',
            'product.otherInfo',
            'product.models',
            
        ])->where('id', $variantId)
        ->where('product_id', $productId)
        ->firstOrFail();

        
        $product = $variant->product;
        $productPrice = ProductPrice::where('product_name', $product->name)->first();

        $relatedProducts = Product::with('homePhoto')
        ->whereIn('id', RelatedProductVariation::where('product_id', $product->id)
            ->pluck('variation_option_id')
        )
        ->get();
        

        

        $mainOption = null;
        if ($optionId) {
            $mainOption = $variant->options->firstWhere('id', $optionId);
        }
        $vendor = Auth::guard('vendor')->user();
        return view($this->filePath . '.product_detail', [
            'product' => $variant->product,
            'variant' => $variant,
            'mainOption' => $mainOption,
            'relatedProducts' => $relatedProducts,
            'productPrice' => $productPrice,
            'vendor' => $vendor

            
        ]);
    }

    public function updateQuantity(Request $request)
    {
        $request->validate([
            'option_id' => 'required|exists:product_variation_options,id',
            'quantity' => 'required|integer|min:0',
        ]);

        $vendorId = Auth::guard('vendor')->user()->id;

        VendorProductQuantity::updateOrCreate(
            [
                'vendor_id' => $vendorId,
                'product_variation_option_id' => $request->option_id
            ],
            [
                'quantity' => $request->quantity
            ]
        );

        return redirect()->back()->with('success', 'Quantity updated successfully.');
    }


    public function updateSelectedProducts(Request $request)
    {

        $vendor = Auth::guard('vendor')->user();
        $productIds = $request->input('product_ids', []);

        // Previously enabled
        $previouslyEnabledIds = $vendor->products()->pluck('products.id')->toArray();

        // Determine products to enable and disable
        $productsToEnable = array_diff($productIds, $previouslyEnabledIds);
        $productsToDisable = array_diff($previouslyEnabledIds, $productIds);

        // Get names for better readability
        $previouslyEnabledNames = Product::whereIn('id', $previouslyEnabledIds)->pluck('name')->toArray();
        $enableNames = Product::whereIn('id', $productsToEnable)->pluck('name')->toArray();
        $disableNames = Product::whereIn('id', $productsToDisable)->pluck('name')->toArray();



        // Create notification
        Notification::create([
            'type' => 'SPO_product_administration',
            'status' => 'SPO Product Administration',
            'user_id' => $vendor->id,
            'customer_name' => $vendor->first_name . ' ' . $vendor->last_name,
            'customer_email' => $vendor->email ?? '',
            'customer_phone' => $vendor->number ?? '',
            'data' => [
                ['label' => 'SPO ID', 'value' => '#' . $vendor->id],
                ['label' => 'Earlier Enabled Products', 'value' => implode(', ', $previouslyEnabledNames)],
                ['label' => 'Products to Enable', 'value' => implode(', ', $enableNames)],
                ['label' => 'Products to Disable', 'value' => implode(', ', $disableNames)],
            ],
            'is_read' => false,
        ]);

       


    
        return response()->json(['success' => true]);
    }


    public function addProduct(Request $request)
    {
        $vendor = Auth::guard('vendor')->user();

        Notification::create([
            'type' => 'SPO_product_request',
            'status' => 'SPO Product Request',
            'user_id' => $vendor->id, 
            'customer_name' => $vendor->first_name . ' ' . $vendor->last_name,
            'customer_email' => $vendor->email,
            'customer_phone' => $vendor->number, 
            'data' => [
                'product_name' => $request->productName,
                'reason' => $request->productReq,
                'ID' => '#' . $vendor->id 
            ],
            'is_read' => false,
        ]);

        return redirect()->back()->with('success', 'Thank you, your request has been successfully sent to the admin for approval.');


    }

    public function downloadPdf()
    {
        $vendor = Auth::guard('vendor')->user();
        $rentedCounts = $this->getRentedItemsCount();
        
        $products = $vendor->products()->with([
            'category', 
            'variations.options.color',
            'variations.size',
            'variations.variation',
            'variations.options.vendorQuantities' => function($q) use ($vendor) {
                $q->where('vendor_id', $vendor->id);
            }
        ])->get();

        foreach ($products as $product) {
            if ($product->variations->isEmpty()) {
                $rentedCount = $rentedCounts['product'][$product->id] ?? 0;
                $product->total_quantity = ($product->quantity ?? 0) - $rentedCount;
            } else {
                foreach ($product->variations as $variation) {
                    if (!$variation->options->isEmpty()) {
                        foreach ($variation->options as $option) {
                            $vendorQty = $option->vendorQuantities()
                                ->where('vendor_id', $vendor->id)
                                ->first();
                            $option->quantity = $vendorQty?->quantity ?? 0;
                            $rentedCount = $rentedCounts['option'][$option->id] ?? 0;
                            $option->total_quantity = $option->quantity - $rentedCount;
                        }
                    } else {
                        $rentedCount = $rentedCounts['variation'][$variation->id] ?? 0;
                        $variation->total_quantity = ($variation->quantity ?? 0) - $rentedCount;
                    }
                }
            }
        }

        $disabledVariants = $vendor->disabledVariants()->pluck('product_variation_id')->toArray();

        $data = [
            'products' => $products,
            'disabledVariants' => $disabledVariants,
            'vendor' => $vendor,
            'title' => 'Vendor Product Analytics Report - ' . $vendor->business_name
        ];
        
        $pdf = PDF::loadView('vendor.product.pdf', $data);
        return $pdf->download('vendor-product-report-'.now()->format('Y-m-d').'.pdf');
    }

    public function downloadCsv()
    {
        $vendor = Auth::guard('vendor')->user();
        return Excel::download(new VendorProductsExport($vendor), 
            'vendor-products-'.now()->format('Y-m-d').'.csv', 
            \Maatwebsite\Excel\Excel::CSV, [
                'Content-Type' => 'text/csv',
            ]);
    }

    public function downloadQuantityTemplate()
{
    $vendorId = Auth::guard('vendor')->user()->id;
    $vendor = Auth::guard('vendor')->user();
    $vendorId = $vendor->id;
    $vendorName = Str::slug($vendor->first_name . '_' . $vendor->last_name, '_');

    
    // Get all products and variations available to this vendor
    $vendorProducts = VendorProductQuantity::with(['variationOption.variation.product', 'variationOption.color'])
        ->where('vendor_id', $vendorId)
        ->get();
    
        $fileName = "quantity_template_{$vendorName}_{$vendorId}_" . date('Y-m-d') . '.csv';
    $headers = [
        'Content-Type' => 'text/csv',
        'Content-Disposition' => 'attachment; filename=' . $fileName,
    ];
    
    $callback = function() use ($vendorProducts) {
        $file = fopen('php://output', 'w');
        
        // CSV header
        fputcsv($file, [
            'product_id', 
            'product_name', 
            'variation_id', 
            'variation_name', 
            'color', 
            'size', 
           'total_quantity',
            'new_quantity'
        ]);
        
        foreach ($vendorProducts as $vendorProduct) {
            $option = $vendorProduct->variationOption;
            $variation = $option->variation;
            $product = $variation->product;
            
            fputcsv($file, [
                $product->id,
                $product->name,
                $option->id,
                $variation->size ? $variation->size->name : ($variation->variation ? $variation->variation->name : 'Default'),
                $option->color ? $option->color->name : 'Default',
                $variation->size ? $variation->size->name : 'N/A',
                $vendorProduct->quantity,
                0 // Placeholder for new quantity
            ]);
        }
        
        fclose($file);
    };
    
    return Response::stream($callback, 200, $headers);
}

public function importQuantities(Request $request)
{
    $request->validate([
        'csv_file' => 'required|file|mimes:csv,txt'
    ]);
    
    $vendorId = Auth::guard('vendor')->user()->id;
    $file = $request->file('csv_file');
    
    try {
        $csv = Reader::createFromPath($file->getPathname(), 'r');
        $csv->setHeaderOffset(0);
        $records = $csv->getRecords();
        
        $updated = 0;
        $errors = [];
        
        foreach ($records as $index => $record) {
            // Skip empty rows
            if (empty(array_filter($record))) {
                continue;
            }
            
            // Validate required fields
            if (empty($record['variation_id']) || !isset($record['new_quantity'])) {
                $errors[] = "Row " . ($index + 2) . ": Missing variation_id or new_quantity";
                continue;
            }
            
            // Validate quantity is a non-negative integer
            if (!is_numeric($record['new_quantity']) || $record['new_quantity'] < 0) {
                $errors[] = "Row " . ($index + 2) . ": Quantity must be a non-negative number";
                continue;
            }
            
            $variationId = $record['variation_id'];
            $quantity = (int) $record['new_quantity'];
            
            // Update or create the quantity record
            VendorProductQuantity::updateOrCreate(
                [
                    'vendor_id' => $vendorId,
                    'product_variation_option_id' => $variationId
                ],
                [
                    'quantity' => $quantity
                ]
            );
            
            $updated++;
        }
        
        if (!empty($errors)) {
            return response()->json([
                'success' => false,
                'message' => 'Some quantities failed to update',
                'errors' => $errors
            ], 422);
        }
        
        return response()->json([
            'success' => true,
            'message' => "{$updated}  Produkte wurden erfolgreich aktualisiert"
            
        ]);
        
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error processing CSV file: ' . $e->getMessage()
        ], 500);
    }
}

}

