<?php
namespace App\Http\Controllers\Vendor;

use App\Http\Controllers\Controller;
use App\Models\Admin;
use App\Models\VendorWorkingHour;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\File;
use App\Models\ProductVariationOption;
use App\Models\VendorProductQuantity;
use App\Models\VendorHoliday;
use App\Models\WorkingHours;
use App\Models\Product;
use App\Models\Vendor;

class VendorController extends Controller
{
    public function dashboard()
    {
        return view('vendor.dashboard.index');
    }

    public function profileView()
    {
        $vendor = Auth::guard('vendor')->user();
        $time = VendorWorkingHour::where('vendor_id', $vendor->id)->get();
        $holidays = VendorHoliday::where('vendor_id', $vendor->id)->get();
        $country_code = json_decode(File::get(public_path('country-codes.json')), true);

        return view('vendor.auth.profile', compact('vendor', 'time', 'holidays', 'country_code'));
    }

    // public function editProfile()
    // {
    //     $admin = Auth::guard('admin')->user();
    //     $country_code = json_decode(File::get(public_path('country-codes.json')), true);
    //     $times = WorkingHours::where('admin_id', $admin->id)->get();

    //     $admin  = Auth::guard('admin')->user();
    //     return view('admin.auth.edit-profile', compact('admin', 'country_code', 'times'));
    // }



    public function editProfile()
    {
        $vendor = Auth::guard('vendor')->user();
        return view('vendor.auth.editProfile', compact('vendor'));
    }
    
    public function updateProfile(Request $request, Vendor $vendor)
    {

        DB::beginTransaction();
    
        try {
            // Validate the request data
            $validated = $request->validate([
               'password' => 'nullable|string|min:8|confirmed',
                'days_closed' => 'nullable|array',
                'holidays' => 'nullable|array',
                'holidays.*.id' => 'nullable|integer',
                'holidays.*.name' => 'required|string',
                'holidays.*.date' => 'required|date',
                'mo_from' => 'nullable|date_format:H:i',
                'mo_to' => 'nullable|date_format:H:i|after:mo_from',
                'di_from' => 'nullable|date_format:H:i',
                'di_to' => 'nullable|date_format:H:i|after:di_from',
                'mi_from' => 'nullable|date_format:H:i',
                'mi_to' => 'nullable|date_format:H:i|after:mi_from',
                'do_from' => 'nullable|date_format:H:i',
                'do_to' => 'nullable|date_format:H:i|after:do_from',
                'fr_from' => 'nullable|date_format:H:i',
                'fr_to' => 'nullable|date_format:H:i|after:fr_from',
                'sa_from' => 'nullable|date_format:H:i',
                'sa_to' => 'nullable|date_format:H:i|after:sa_from',
                'so_from' => 'nullable|date_format:H:i',
                'so_to' => 'nullable|date_format:H:i|after:so_from',
            ]);
    
            if ($request->hasFile('profile_image')) {
                // Delete old image if it exists
                if ($vendor->profile_image) {
                    Storage::disk('public')->delete($vendor->profile_image);
                }
                
                $path = $request->file('profile_image')->store('vendor_images', 'public');
                $validated['profile_image'] = $path;
            }
    
            if ($request->filled('password')) {
                $validated['password'] = Hash::make($validated['password']);
            } else {
                unset($validated['password']);
            }
            $vendor->update($validated);
    
            $days = ['mo', 'di', 'mi', 'do', 'fr', 'sa', 'so'];
            foreach ($days as $day) {
                $isClosed = in_array($day, $request->days_closed ?? []);
                
                $workingHour = $vendor->workingHours()->where('day', $day)->first();
                
                if ($workingHour) {
                    $workingHour->update([
                        'open_time' => $isClosed ? null : $request->input("{$day}_from"),
                        'close_time' => $isClosed ? null : $request->input("{$day}_to"),
                        'is_closed' => $isClosed
                    ]);
                } else {
                    $vendor->workingHours()->create([
                        'day' => $day,
                        'open_time' => $isClosed ? null : $request->input("{$day}_from"),
                        'close_time' => $isClosed ? null : $request->input("{$day}_to"),
                        'is_closed' => $isClosed
                    ]);
                }
            }
    
            if ($request->holidays) {
                $existingHolidayIds = $vendor->holidays->pluck('id')->toArray();
                $updatedHolidayIds = [];
                foreach ($request->holidays as $holidayData) {
                    if (isset($holidayData['id']) && $holidayData['new'] == 0) {
                        $holiday = $vendor->holidays()->find($holidayData['id']);
                        if ($holiday) {
                            $holiday->update([
                                'name' => $holidayData['name'],
                                'date' => $holidayData['date']
                            ]);
                            $updatedHolidayIds[] = $holiday->id;
                        }
                    } else {
                        $newHoliday = $vendor->holidays()->create([
                            'name' => $holidayData['name'],
                            'date' => $holidayData['date']
                        ]);
                        $updatedHolidayIds[] = $newHoliday->id;
                    }
                }
                
                // Delete holidays that were removed
                $holidaysToDelete = array_diff($existingHolidayIds, $updatedHolidayIds);
                if (!empty($holidaysToDelete)) {
                    $vendor->holidays()->whereIn('id', $holidaysToDelete)->delete();
                }
            } else {
                // If no holidays submitted, delete all existing ones
                $vendor->holidays()->delete();
            }
    
            DB::commit();
    
            return redirect()->route('vendor.profile', $vendor)->with('success', 'SPO updated successfully!');
        } catch (ValidationException $e) {
            DB::rollBack();
            return back()->withErrors($e->errors())->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()->with('error', 'Error updating SPO: ' . $e->getMessage());
        }
    }



   
    public function getChangePassword()
    {
        return view('vendor.auth.change-password');
    }

    public function updateChangePassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'old_password' => 'required',
            'new_password' => [
                'required',
                'min:8'
            ],
            'confirm_new_password' => 'required|same:new_password',
        ], [
            'old_password.required' => 'Please enter old password.',
            'new_password.required' => 'Please enter new password.',
            'new_password.min' => 'New password must be at least 8 characters long.',
            'confirm_new_password.required' => 'Please enter confirm new password.',
            'confirm_new_password.same' => 'New password and confirm new password does not match.',
        ]);
        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $admin = Auth::guard('vendor')->user();

        if (!Hash::check($request->old_password, $admin->password)) {
            return redirect()->back()->with('error', 'Please enter correct current password.');
        }

        $admin->password = Hash::make($request->new_password);
        $admin->save();

        Auth::guard('vendor')->logout();

        return redirect()->route('vendor.login')->with('success', 'Password changed successfully');
    }

    public function addHolidays(Request $request)
    {
        // Validate input
        $request->validate([
            'holiday_name' => 'required|string|max:255',
            'holiday_date' => 'required|date',
        ]);

        $admin = Auth::guard('admin')->user();
        $formattedDate = date('Y-m-d', strtotime($request->holiday_date));

        $existingHoliday = Holidays::where('admin_id', $admin->id)
            ->where('date', $formattedDate)
            ->first();

        if ($existingHoliday) {
            return redirect()->back()->with('error', 'A holiday on this date already exists.');
        }

        // Save the new holiday
        $holiday = new Holidays();
        $holiday->admin_id = $admin->id;
        $holiday->name = $request->holiday_name;
        $holiday->date = $formattedDate;
        $holiday->save();

        return redirect()->back()->with('success', 'Holiday added successfully!');
    }


    // Delete a holiday
    public function deleteHoliday($id)
    {
        try {
            $holiday = Holidays::findOrFail($id);
            $holiday->delete(); 

            return response()->json(['success' => true, 'message' => 'Holiday deleted successfully.']);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }


    public function productSelection(){
        $products = Product::with(['homePhoto','category'])->get();
        return view('vendor.productSelection.index',compact('products'));
    }

    // public function saveSelectedProducts(Request $request)
    // {
    //     $vendor = Auth::guard('vendor')->user();
    //     $productIds = $request->input('product_ids', []);
    
    //     $vendor->products()->sync($productIds); 
    
    //     return redirect()->route('vendor.dashboard')->with('success', 'Products updated successfully.');
    // }

    public function saveSelectedProducts(Request $request)
    {
        $vendor = Auth::guard('vendor')->user();
        $productIds = $request->input('product_ids', []);

        $vendor->products()->sync($productIds);

        // Sync vendor quantities
        foreach ($productIds as $productId) {
            $variationOptionIds = ProductVariationOption::whereHas('variation', function ($q) use ($productId) {
                $q->where('product_id', $productId);
            })->pluck('id','sku');


            foreach ($variationOptionIds as $sku => $optionId) {
                VendorProductQuantity::firstOrCreate(
                    [
                        'vendor_id' => $vendor->id,
                        'product_variation_option_id' => $optionId,
                    ],
                    [
                        'quantity' => 0,
                        'sku' => $sku
                    ]
                );
            }
        }

        return redirect()->route('vendor.dashboard')->with('success', 'Products updated successfully.');
    }
   

}
