<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CartItem extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'user_id',
        'product_id',
        'product_name',
        'car_name',
        'car_year',
        'model',
        'trailer_hitch',
        'roof_type',
        'color',
        'size',
        'is_carrier_included',
        'is_insured',
        'pickup_date',
        'days',
        'dropoff_date',
        'dropoff_time',
        'calculated_rent',
        'status',
        'carrier_length',
        'foot_sku' ,
        'footkit_sku',
        'user_timing',
        'coupon_code',
        'discount',
        'vendor_id',
        'booking_item_id',
        'booking_request_id',
        'pickup_end',
        'dropoff_end'
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'trailer_hitch' => 'boolean',
        'is_carrier_included' => 'boolean',
        'is_insured' => 'boolean',
        'pickup_date' => 'date',
        'dropoff_date' => 'date',
        'calculated_rent' => 'decimal:2',
    ];

    /**
     * Get the user that owns the cart item.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the product associated with the cart item.
     */
    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    /**
     * Scope a query to only include pending cart items.
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope a query to only include ready for checkout cart items.
     */
    public function scopeReadyForCheckout($query)
    {
        return $query->where('status', 'ready_for_checkout');
    }

    /**
     * Scope a query to only include completed cart items.
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * Calculate the total rent including insurance if selected.
     */
    public function calculateTotal()
    {
        $total = $this->calculated_rent;
        
        if ($this->is_insured) {
            $total += 25; // Insurance fee
        }
        
        return $total;
    }
}