<?php

namespace App\Services;

use App\Models\Admin;
use App\Models\Booking;
use App\Models\BookingItem;
use App\Models\BookingUserDetail;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;

class InvoiceService
{
    public function generateInvoice(Booking $booking)
    {
        $bookingItems = BookingItem::where('booking_id', $booking->id)->get();
        $userDetails = BookingUserDetail::where('booking_id', $booking->id)->first();

        if (empty($booking->invoice_number)) {
            $booking->invoice_number = $this->generateInvoiceNumber($booking);
            $booking->save();
        }
        
        $invoiceData = [
            'issuer' => $this->getIssuerDetails(),
            'recipient' => $this->getRecipientDetails($userDetails),
            'invoice_number' => $booking->invoice_number,
            'invoice_date' => Carbon::now(),
            'booking_date' => $booking->created_at,
            'booking' => $booking,
            'booking_items' => $this->formatBookingItems($bookingItems),
            'subtotal' => $booking->subtotal,
            'tax' => $booking->tax,
            'total' => $booking->total,
            'payment_method' => $booking->payment_method,
        ];

        $pdf = PDF::loadView('frontend.payment.invoice.invoice', $invoiceData);
        
        return [
            'pdf' => $pdf,
            'data' => $invoiceData,
            'invoice_number' => $invoiceData['invoice_number']
        ];
    }

    private function generateInvoiceNumber(Booking $booking)
    {
        $prefix = 'BO';
        return $prefix . $booking->id;
    }

    private function formatBookingItems($bookingItems)
    {
        return $bookingItems->map(function ($item) {
            $pickupTime = $item->pickup_time ? Carbon::parse($item->pickup_time)->format('H:i') : '';
            $dropoffTime = $item->dropoff_time ? Carbon::parse($item->dropoff_time)->format('H:i') : '';
            
            return [
                'product_name' => $item->product_name,
                'size' => $item->size,
                'color' => $item->color ?? 'N/A', // Handle missing color
                'roof_type' => $item->roof_type ?? 'N/A', // Handle missing roof type
                'car_name' => $item->car_name,
                'car_year' => $item->car_year,
                'model' => $item->model,
                'is_insured' => $item->is_insured,
                'is_carrier_included' => $item->is_carrier_included,
                'pickup_date' => Carbon::parse($item->pickup_date)->format('d.m.Y'),
                'pickup_time' => $pickupTime,
                'dropoff_date' => Carbon::parse($item->dropoff_date)->format('d.m.Y'),
                'dropoff_time' => $dropoffTime,
                'calculated_rent' => $item->calculated_rent
            ];
        });
    }

    private function getIssuerDetails()
    {
        $admin = Admin::first();
        
        if (!$admin) {
            throw new \RuntimeException('Admin details not found');
        }
        
        return [
            'name' => $admin->company_name,
            'address' => $admin->address . ', ' . $admin->zip_code . ' ' . $admin->city . ', Deutschland',
            'tax_number' => $admin->tax_number,
            'vat_number' => $admin->umsatzsteuer,
        ];
    }

    private function getRecipientDetails($userDetails)
    {
        if (!$userDetails) {
            throw new \RuntimeException('User details not found');
        }
        
        return [
            'name' => $userDetails->first_name . ' ' . $userDetails->last_name,
            'address' => $userDetails->address,
            'zip' => $userDetails->zip,
            'city' => $userDetails->city,
            'company_name' => $userDetails->company_name,
            'country' => $userDetails->country,
            'email' => $userDetails->email,
            'phone' => $userDetails->phone,
        ];
    }
}