<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;

class MailchimpService
{
    public function subscribe(string $email, string $listId = null)
    {
        $listId = $listId ?? config('services.mailchimp.list_id');
        $server = config('services.mailchimp.server');
        $apiKey = config('services.mailchimp.key');

        $subscriberHash = md5(strtolower($email));
        $user = auth()->user(); // logged-in user

        $mergeFields = [
            'FNAME' => $user->first_name ?? '',
            'LNAME' => $user->last_name ?? '',
            'PHONE' => $user->number ?? '',
            'ADDRESS' => [
                'addr1'   => $user->street ?? '',
                'city'    => $user->city ?? '',
                'state'   => $user->region ?? '',
                'zip'     => $user->zip_code ?? '',
                'country' => $user->country ?? '',
            ],
        ];

        $url = "https://{$server}.api.mailchimp.com/3.0/lists/{$listId}/members/{$subscriberHash}";

        // -----------------------
        // CHECK IF ALREADY EXISTS
        // -----------------------
        $response = Http::withBasicAuth('anystring', $apiKey)->get($url);

        if ($response->ok()) {
            $status = $response->json('status');

            if ($status === 'subscribed') {
                return [
                    'status' => 'already_subscribed',
                    'message' => 'You are already subscribed.'
                ];
            }

            // -----------------------
            // RESUBSCRIBE EXISTING
            // -----------------------
            Http::withBasicAuth('anystring', $apiKey)->put($url, [
                'status' => 'subscribed',
                'merge_fields' => $mergeFields
            ]);

            return [
                'status' => 'subscribed',
                'message' => 'You have been resubscribed successfully.'
            ];
        }

        // -----------------------
        // ADD NEW SUBSCRIBER
        // -----------------------
        $addUrl = "https://{$server}.api.mailchimp.com/3.0/lists/{$listId}/members/";

        $response = Http::withBasicAuth('anystring', $apiKey)->post($addUrl, [
            'email_address' => $email,
            'status' => 'subscribed',
            'merge_fields' => $mergeFields
        ]);

        if ($response->successful()) {
            return [
                'status' => 'subscribed',
                'message' => 'You are now subscribed!'
            ];
        }

        // Fallback
        return [
            'status' => 'fail',
            'message' => $response->json('detail') ?? 'Subscription failed.'
        ];
    }


    public function isSubscribed(string $email, string $listId = null)
    {
        $listId = $listId ?? config('services.mailchimp.list_id');
        $server = config('services.mailchimp.server');
        $apiKey = config('services.mailchimp.key');

        $subscriberHash = md5(strtolower($email));
        $url = "https://{$server}.api.mailchimp.com/3.0/lists/{$listId}/members/{$subscriberHash}";

        $response = Http::withBasicAuth('anystring', $apiKey)->get($url);
        if (!$response->successful()) return false;

        return $response->json('status') === 'subscribed';
    }

    public function unsubscribe(string $email, string $listId = null)
    {
        $listId = $listId ?? config('services.mailchimp.list_id');
        $server = config('services.mailchimp.server');
        $apiKey = config('services.mailchimp.key');

        $subscriberHash = md5(strtolower($email));
        $url = "https://{$server}.api.mailchimp.com/3.0/lists/{$listId}/members/{$subscriberHash}";

        $response = Http::withBasicAuth('anystring', $apiKey)->patch($url, [
            "status" => "unsubscribed"
        ]);

        return $response->successful();
    }
}
